//===-- temu-c/Events.h - Event Queue API -----------------------*- C++ -*-===//
//
// TEMU: The Terma Emulator
// (c) Terma 2015, 2016
// Authors: Mattias Holm <maho (at) terma.com>
//
//===----------------------------------------------------------------------===//

#ifndef TEMU_EVENTS_H
#define TEMU_EVENTS_H

#include "temu-c/Support/Attributes.h"
#include "temu-c/Support/Objsys.h"
#include <stdint.h>
#ifdef __cplusplus
extern "C" {
#endif

// Synchronisation events
typedef enum {
  teSE_Cpu,     // Trigger event when CPU reaches timer
  teSE_Machine, // Synchronise event on machine
} temu_SyncEvent;

typedef struct temu_EventQueue temu_EventQueue;

// NOTE: TimeSource is an internal type at the moment,
//       users should not implement their own time sources.
struct temu_TimeSource {
  temu_Object Super;
  temu_TimeSource *ParentTimeSource;
  temu_EventQueue *Queue;
  int64_t Frequency;
  int64_t Steps;
  temu_SyncEvent Level;
  int64_t NextEvent;
  int64_t StartSteps;
  int64_t TargetSteps;
  double InstructionsPerCycle; // Inverse of CyclesPerInstruction (CPI)
  double CyclesPerInstruction; // Inverse of InstructionsPerCycle (IPC)
};

typedef struct temu_Event {
  int64_t Steps;          //!< Managed internally, do not touch
  int64_t Time;           //!< Managed internally, do not touch
  int64_t EventId;        //!< Managed internally, do not touch
  uint32_t Flags;         //!< Internal use, do not touch
  int QueueIdx;           //!< Managed internally, do not touch
  temu_TimeSource *Queue; //!< Managed internally, do not touch
  temu_Object *Obj;       //!< Managed internally, only for reading
  //! Event callback
  void (*Func)(struct temu_Event *);
  int64_t Period;       //!< Managed: Period (cycles) for cyclic events
  int64_t RTTime;       //!< Managed: Time (monotonic ns) for RT exec
  int64_t RTPeriod;     //!< Managed: Period (ns) for RT events
  int64_t TriggerCount; //!< Managed: Number of time event has been triggered
} temu_Event;

//! Posting callback from other threads
typedef void (*temu_ThreadSafeCb)(void *);

/*!
  Publish a preallocated event object.

  The event objects can be embedded in your own class if needed.

  \param EvName Name of the event
  \param Ev Pointer to the event struct
  \param Obj The object associated with the event
  \param Func The event callback function.
  \result The event ID.
 */
TEMU_API int64_t temu_eventPublishStruct(const char *EvName, temu_Event *Ev,
                                         void *Obj, void (*Func)(temu_Event *));

/*!
 * It is possible to depublish events explicitly, but this is rarely
 * needed as it is managed at termination time automatically.
 *
 * \param EvID Event ID
 */
TEMU_API void temu_eventDepublish(int64_t EvID);

/*!
  Publish an event.

  The function will allocate an event structure in the global event
  heap and return the event ID.

  A typical use is to call the function in the object constructor
  and assign the event id to a field in the object. This field should not
  be snapshotted (i.e. it will be reassigned in the constructor) when
  restoring an object anyway.

  \note This function is non thread safe. Object construction is expected
        to be done in a single thread (e.g. the main thread).

  \param EvName Name of the event
  \param Obj The object associated with the event
  \param Func The event callback function.
  \result The event ID >= 0 in case of success. Negative values
          indicate errors.
 */
TEMU_API int64_t temu_eventPublish(const char *EvName, void *Obj,
                                   void (*Func)(temu_Event *));

/*!
 * Post events with a relative trigger in steps
 * 
 * Posting an event in steps is not effected by the IPC/CPI setting of the clock.
 *
 * \param TS The time source object (normally a CPU object)
 * \param EvID The even ID returned by one of the event publish functions.
 * \param Delta The number of CPU steps in the future to post the event.
 *        This should be > 0.
 * \param Sync whether the event should be executed on the CPU queue or the
 *        machine queue.
 */
TEMU_API void temu_eventPostSteps(temu_TimeSource *TS, int64_t EvID,
                                  int64_t Delta, temu_SyncEvent Sync);
/*!
 * Post events with a relative time in cycles in the future
 *
 * Note that if posting a scheduled event, a warning will be printed
 * and the event will be automatically descheduled before being
 * inserted in the event queue.
 *
 * \param TS The time source object (normally a CPU object)
 * \param EvID The even ID returned by one of the event publish functions.
 * \param Delta The number of CPU clock cycles in the future to post the event.
 *        This should be > 0.
 * \param Sync whether the event should be executed on the CPU queue or the
 *        machine queue.
 */
TEMU_API void temu_eventPostCycles(temu_TimeSource *TS, int64_t EvID,
                                   int64_t Delta, temu_SyncEvent Sync);
/*!
 * Post events with a relative time in nanoseconds in the future
 * \param TS The time source object (normally a CPU object)
 * \param EvID The even ID returned by one of the event publish functions.
 * \param Delta The number of nanoseconds in the future to post the event.
 *        This should be > 0.
 * \param Sync whether the event should be executed on the CPU queue or the
 *        machine queue.
 */

TEMU_API void temu_eventPostNanos(temu_TimeSource *TS, int64_t EvID,
                                  int64_t Delta, temu_SyncEvent Sync);
/*!
 * Post events with a relative time in seconds in the future
 * \param TS The time source object (normally a CPU object)
 * \param EvID The even ID returned by one of the event publish functions.
 * \param Delta The number of seconds in the future to post the event.
 *        This should be > 0.
 * \param Sync whether the event should be executed on the CPU queue or the
 *        machine queue.
 */
TEMU_API void temu_eventPostSecs(temu_TimeSource *TS, int64_t EvID,
                                 double Delta, temu_SyncEvent Sync);

/*!
 * Post events in the event queue stack
 *
 * Stacked events are executed either after the current instruction is
 * finished or when the machine quanta expires in case of machine
 * synchronised events.
 *
 * \param TS The time source object (normally a CPU object)
 * \param EvID The even ID returned by one of the event publish functions.
 * \param Sync whether the event should be executed on the CPU queue or the
 *        machine queue.
 */
TEMU_API void temu_eventPostStack(temu_TimeSource *TS, int64_t EvID,
                                  temu_SyncEvent Sync);

/*!
 * Post events in the immediate event queue
 *
 * Immediate events are executed either after the current instruction is
 * finished or when the machine quanta expires in case of machine
 * synchronised events. This is the same as for stacked events,
 * however immediate events are executed in FIFO order.
 *
 * \param TS The time source object (normally a CPU object)
 * \param EvID The even ID returned by one of the event publish functions.
 * \param Sync whether the event should be executed on the CPU queue or the
 *        machine queue.
 */
TEMU_API void temu_eventPostImmediate(temu_TimeSource *TS, int64_t EvID,
                                      temu_SyncEvent Sync);



/*!
 * Post events with an absolute time in cycles
 * The function is only well defined for times >= current time.
 * Timestamps < current time will have undefined behaviour.
 * \param TS The time source object (normally a CPU object)
 * \param EvID The even ID returned by one of the event publish functions.
 * \param AbsTime The absolute time stamp to run the event at, this should be >
 *                current time.
 * \param Sync whether the event should be executed on the CPU
 * queue or the machine queue.
 */
TEMU_API void temu_eventPostCyclesAbsolute(temu_TimeSource *TS, int64_t EvID,
                                           int64_t AbsTime, temu_SyncEvent Sync);

/*!
 * Post events with an absolute time in nanoseconds
 * The function is only well defined for times >= current time.
 * Timestamps < current time will have undefined behaviour.
 * \param TS The time source object (normally a CPU object)
 * \param EvID The even ID returned by one of the event publish functions.
 * \param AbsTime The absolute time stamp to run the event at, this should be >
 *                current time.
 * \param Sync whether the event should be executed on the CPU
 * queue or the machine queue.
 */

TEMU_API void temu_eventPostNanosAbsolute(temu_TimeSource *TS, int64_t EvID,
                                          int64_t AbsTime, temu_SyncEvent Sync);


/*!
 * Reschedule event with relative time in cycles
 *
 * This function automatically deschedules the event before scheduling it again.
 * The function ensures only one lock is taken for the pattern:
 * 
 * - is scheduled?
 * - de-schedule
 * - post
 *
 * \param TS The time source object (normally a CPU object)
 * \param EvID The even ID returned by one of the event publish functions.
 * \param Delta The number of CPU clock cycles in the future to post the event.
 *        This should be > 0.
 * \param Sync whether the event should be executed on the CPU queue or the
 *        machine queue.
 */

TEMU_API 
void temu_eventRescheduleCycles(temu_TimeSource *TS, int64_t EvID, int64_t Delta,
                                temu_SyncEvent Sync);
/*!
 * Check if event is scheduled
 *
 * \param EvID The event ID to check whether it is scheduled.
 *
 * \result Zero (0) in case the event is not scheduled, otherwise
 *         non-zero for scheduled events.
 */
TEMU_API int temu_eventIsScheduled(int64_t EvID);

/*!
 * Deschedule event
 *
 * \param EvID The event ID of the event to deschedule.
 */
TEMU_API void temu_eventDeschedule(int64_t EvID);

/*!
 * Get delta time in cycles for the given event and queue
 *
 * \param TS The time source object (normally a CPU object)
 * \param EvID the event to get the cycles for.
 * \result Number of simulated cycles in the future the event will trigger
 */
TEMU_API int64_t temu_eventGetCycles(temu_TimeSource *TS, int64_t EvID);

/*!
 * Get delta time in nanoseconds for the given event and queue
 *
 * \param TS The time source object (normally a CPU object)
 * \param EvID the event to get the nanoseconds for.
 * \result Number of simulated nanoseconds in the future when the event will
 * trigger
 */
TEMU_API int64_t temu_eventGetNanos(temu_TimeSource *TS, int64_t EvID);

/*!
 * Get delta time in seconds for the given event and queue
 *
 * \param TS The time source object (normally a CPU object)
 * \param EvID the event to get the seconds for.
 * \result Number of simulated seconds in the future when the event will trigger
 */
TEMU_API double temu_eventGetSecs(temu_TimeSource *TS, int64_t EvID);

//======================== EXPERIMENTAL API ==================================//
/*! Flags for async functions below. */
#define TEMU_ASYNC_CYCLIC 1
#define TEMU_ASYNC_READ (1 << 1)
#define TEMU_ASYNC_WRITE (1 << 2)

/*!
 * Add asynchronously activated wall-clock timed events
 *
 * The callback function will be called synchronously by the emulator
 * core associated with Q. That means that when CB is executing it is
 * safe to do anything that can be done from a normal event or MMIO
 * handler.
 *
 * Note that the event is only called when a CPU core or machine is
 * running. When the timer has triggered, it is temporarily disabled
 * and the CB is posted on the synchornous event queue as an async
 * event. This will be executed when the next normal event expires
 * (e.g. at the end of the current quanta). After the event has been
 * called, depending on wether the timer is cyclic or not, it will be
 * reactivated. This means that if the emulator is paused, at most one
 * call to the event handler will be issued, and this will be done
 * when the emulator is resumed.
 *
 * \param TS The time source object (normally a CPU object)
 *
 * \param T Delta seconds in the future for the first event to be posted.
 *
 * \param Flags Set to TEMU_ASYNC_CYCLIC if the event should be
 *        executed as a cyclic event.
 * \param CB Call back function on when timer expires
 * \param Data Context data to be passed to the callback function
 *
 * \result Returns a file descriptor or timer ID.
 */
TEMU_API int temu_asyncTimerAdd(temu_TimeSource *TS, double T, unsigned Flags,
                                void (*CB)(void *), void *Data);

/*!
 * Remove an async timer
 *
 * \param Fd Timer ID to remove
 */
TEMU_API void temu_asyncTimerRemove(int Fd);

/*!
 * Add asynchronous event triggered by file descriptor changes
 *
 * \param TS the synchronous queue or time source object (normally a CPU object)
 *
 * \param Sock File descriptor for the socket. This can also be a pipe
 *             FD. Note that the function is likely to be renamed.
 *             Note that in case you are intending to read from the socket,
 *             make sure it is non-blocking.
 *
 * \param Flags TEMU_ASYNC_READ in case you wish to be notified about
 * data available to read.
 * \param CB Callback when event is triggered.
 * \param Data Data pointer passed to callback.
 *
 *  \result Returns the file descriptor (Sock)
 * if successful, otherwise -1.
 */
TEMU_API int temu_asyncSocketAdd(temu_TimeSource *TS, int Sock, unsigned Flags,
                                 void (*CB)(void *), void *Data);

/*!
 * Remove an asynchronous event triggered by file descriptor changes
 *
 * \param Fd The ID of the socket to  remove
 * \param Flags The flags of removing
 */
TEMU_API void temu_asyncSocketRemove(int Fd, unsigned Flags);

/*!
 * Post an event to an asynchronous queue
 *
 * \param TS the asynchronous queue or time source object (normally a CPU
 * object) \param CB The callback function to call when the event happens \param
 * Data The context data to be passed  to the callback function \param Sync
 * Execute on CPU or machine level.
 */
TEMU_API void temu_eventPostAsync(temu_TimeSource *TS, temu_ThreadSafeCb CB,
                                  void *Data, temu_SyncEvent Sync);

/*!
 * Mark an event as real-time.
 *
 * With a real-time event, it is meant that the event will execute at
 * roughly real-time. In the case when the event is posted, it will
 * compute a rough triggering time in wall clock. When the event is
 * executed it will do two things:
 *
 *   1. Check if the event wall-clock time is before the current time,
 *      if so emit a "temu.realtime-slip" notification".
 *
 *   2. Check if the event is triggered in the future (with respect to
 *      wall clock), if so sleep until the wall-clock has caught
 *      up.
 *
 * Due to the behaviour, the real-time events are suitable only for
 * relatively short event times, as long sleep times may make the
 * emulator non responsive. Especially at present, this will block the
 * execution of async events as the event queue is halted until the
 * wall-clock catches up to the simulated time.
 *
 * Note that RT event support is primarily used to slow down the
 * emulator artificially, this is accomplished by enabling real-time
 * mode on the relevant model.
 *
 * This function should only be called on in-flight events.
 *
 * \param EvID Event ID to make real-time
 * \result 0 On success.
 */
TEMU_API int temu_eventSetRealTime(int64_t EvID);

/*!
 * Set the cycles property on an event.
 *
 * Periodic events have the advantage that they do not slip due to
 * reposting of event with respect to the event triggering time and
 * behaves as if the reposting is 1, automatic and 2, is relative to
 * the event schedule time. which differ from the triggering time by
 * possibly a few cycles. Note, calling this function does not
 * reschedule the event.
 *
 * \param EvID Event ID
 * \param Period Cycles to add as period.
 */
TEMU_API void temu_eventSetPeriodCycles(int64_t EvID, int64_t Period);

/*!
 * Set the RT period (nanoseconds), this should be the same as the
 * cycle period, but it should be in nanoseconds and tno cycles
 *
 * \param EvID Event ID
 * \param Period Period in nanoseconds
 */
TEMU_API void temu_eventSetRTPeriodNanos(int64_t EvID, int64_t Period);

/*!
 * Set the RT time (nanoseconds absoulute time computed from
 * temu_timeGetMonotonicWct()). This time is used by events flagged
 * with TEMU_EVENT_RT to delay the execution of the event handler
 * function.
 *
 * \param EvID Event ID
 * \param Time Absolute time to trigger event at in WCT
 */

TEMU_API void temu_eventSetRTTime(int64_t EvID, int64_t Time);

/*!
 * Get the frequency in Hz of the given time source.
 * \param TS Time source
 */
TEMU_API uint64_t temu_eventQueueGetFreq(temu_TimeSource *TS);

//================= END OF EXPERIMENTAL API ==================================//

#ifdef __cplusplus
}
#endif

#endif /*  ! TEMU_EVENTS_H */
