//===-- temu-c/Objsys.h - TEMU Object System -------------------*- C++ -*-===//
//
// TEMU: The Terma Emulator
// (c) Terma 2015, 2016, 2021
// Authors: Mattias Holm <maho (at) terma.com>
//
//===----------------------------------------------------------------------===//

#ifndef TEMU_OBJSYS_C_H
#define TEMU_OBJSYS_C_H

#include "temu-c/Support/Attributes.h"
#include "temu-c/Support/Temu2Compat.h"
#include "temu-c/Support/Temu3Compat.h"
#include <assert.h>
#include <stddef.h>
#include <stdint.h>
#include <stdlib.h>

#ifdef __cplusplus
extern "C" {
#endif

#ifdef __cplusplus
#define TEMU_PLUGIN_INIT extern "C" TEMU_API void temu_pluginInit(void)
#else
#define TEMU_PLUGIN_INIT TEMU_API void temu_pluginInit(void)
#endif

#if defined(__has_attribute)
#if __has_attribute(annotate)
#define TEMU_NO_WRAP __attribute__((annotate("temu-no-wrap")))
#endif
#endif

#ifdef __cplusplus
#if (__cplusplus < 201103L) && !defined(nullptr)
#define nullptr 0
#endif
#endif

#if !defined(TEMU_NO_WRAP)
#define TEMU_NO_WRAP
#endif

#ifndef TEMU_BUFF_DEFINED
#define TEMU_BUFF_DEFINED
typedef struct {
  uintptr_t data0;
  uint32_t data1;
  uint32_t data2;
} temu_Buff;
#endif // !TEMU_BUFF_DEFINED

#ifndef TEMU_COMPONENT_DEFINED
#define TEMU_COMPONENT_DEFINED
typedef struct temu_Component temu_Component;
#endif // !TEMU_COMPONENT_DEFINED

typedef struct temu_Class temu_Class;
typedef void temu_MetaIface;

typedef struct temu_TimeSource temu_TimeSource;

/*!
  Root object type for all TEMU objects.

  The TEMU object system is used for defining models in TEMU.
  All models must derive from `temu_Object`.

  Inheritance in C is done by placing a `temu_Object` field
  as the first field in the derived struct.

  TEMU by convention use the name `Super` for the parent field name.

  The type contains a `UserData` void pointer that can be used by for example
  simulator integrators.
  The `UserData` field can be written by the user.
  The field is guaranteed to not be modified by the TEMU runtime.

  \note While the convention in C is safe,
    in {cpp} it is the responsibility of the user to ensure
    that the derived type is a _standard layout type_.
 */
typedef struct temu_Object {
  temu_Class *Class;               //!< Class pointer
  char *Name;                      //!< Object name
  temu_TimeSource *TimeSource;     //!< Timesource object
  temu_Component *Component;       //!< Parent component (null for root comp)
  uint64_t LoggingFlags;           //!< Log category enabled/disabled
  int64_t WillDisposeNotification; //!< Notification emitted before object is
                                   //!< deleted
  int64_t DisposedNotification; //!< Notification emitted after object has been
                                //!< deleted

  void *UserData; //!< User data pointer. This is not saved in snapshots.

  uint64_t IsClassObject : 1;    //!< The object is a class object
  uint64_t IsCheckpointable : 1; //!< The object is snapshottable
  uint64_t IsTimeSource : 1;     //!< The object is a time source (can be safely
                                 //!< casted to temu_TimeSource)
  uint64_t TraceMemoryReads : 1; //!< Memory read accesses to this object will
                                 //!< be traced by memory space
  uint64_t TraceMemoryWrites : 1; //!< Memory write accesses to this object will
                                  //!< be traced by memory space
  uint64_t BreakOnMemoryRead : 1; //!< Memory read accesses to this object will
                                  //!< trigger break
  uint64_t BreakOnMemoryWrite : 1; //!< Memory write accesses to this object
                                   //!< will trigger break
} temu_Object;

/*!
  Generic interface

  In TEMU interfaces are referred to using an object / interface pointer pair.
  The object is in general passed as the first parameter to functions in the
  interface. This type provides a generic interface reference where the
  interface pointer itself is type erased.

  While the type is rarely used by itself, it is commonly returned by the TEMU
  API. To convert to / from this type from / to typed interface references, it
  is possible to either memcopy or cast field by field.
 */
typedef struct temu_IfaceRef {
  temu_Object_ *Obj; //!< Object pointer (first field of interface reference)
  void *Iface;       //!< Type erased interface pointer
} temu_IfaceRef;

/*!
  Dynamic interface reference array

  In some cases, the number of objects we know about is unknown.
  To solve that TEMU provides a vector like type.

  \note It is not the intention that the fields in the array are manipulated
  directly. Instead, use the `temu_ifaceRefArray*` functions.
*/
typedef struct temu_IfaceRefArray {
  uint32_t Size;         //!< Number of used items in array
  uint32_t Reserved;     //!< Number of slots allocated for array
  temu_IfaceRef *Ifaces; //!< Interface references
} temu_IfaceRefArray;

/*!
  Allocate interface array.

  \param Reserve Number of entries to reserve initially.
  \result The allocated interface array.
 */
TEMU_API temu_IfaceRefArray temu_ifaceRefArrayAlloc(unsigned Reserve);

/*!
  Push an object and raw interface pointer to an interface array.
  \param Arr The interface reference array.
  \param Obj The object implementing the relevant interface
  \param Iface Pointer to the interface struct.
 */
TEMU_API void temu_ifaceRefArrayPush2(temu_IfaceRefArray *Arr TEMU_NONNULL,
                                      temu_Object_ *Obj TEMU_NONNULL,
                                      void *Iface TEMU_NONNULL);

/*!
  Push an interface reference to the interface array.
  \param Arr The interface reference array
  \param Iface The interface reference (object-interface pointer pair)
 */

TEMU_API void temu_ifaceRefArrayPush(temu_IfaceRefArray *Arr TEMU_NONNULL,
                                     temu_IfaceRef Iface);

/*!
  Get the length of a dynamically allocated interface array.
  \param Arr The interface array to get the size from.
  \result The length in number of entries.
 */
TEMU_API unsigned
temu_ifaceRefArraySize(temu_IfaceRefArray *Arr); // Return size

/*!
  Get the index of the interface reference array.
  \param Arr The interface reference array
  \param idx is the index of the array
  \result The interface array with in bounds.
 */
TEMU_API temu_IfaceRef temu_ifaceRefArrayGet(temu_IfaceRefArray *Arr,
                                             unsigned idx);

TEMU_API void temu_ifaceRefArrayDispose(temu_IfaceRefArray *Arr);

/*!
  Easy way of declaring typed interface references and interface reference
  arrays.
 */
#define TEMU_IFACE_REFERENCE_TYPE(N)                                           \
  typedef struct {                                                             \
    temu_Object_ *Obj; /*!< Object pointer */                                  \
    N##Iface *Iface;   /*!< Typed interface pointer */                         \
  } N##IfaceRef;                                                               \
  typedef struct {                                                             \
    uint32_t Size;                                                             \
    uint32_t Reserved;                                                         \
    N##IfaceRef *Ifaces;                                                       \
  } N##IfaceRefArray;                                                          \
  static inline N##IfaceRefArray N##IfaceRefArrayAlloc(unsigned Reserve)       \
  {                                                                            \
    temu_IfaceRefArray Arr = temu_ifaceRefArrayAlloc(Reserve);                 \
    N##IfaceRefArray Res;                                                      \
    Res.Size = Arr.Size;                                                       \
    Res.Reserved = Arr.Reserved;                                               \
    Res.Ifaces = (N##IfaceRef *)Arr.Ifaces;                                    \
    return Res;                                                                \
  }                                                                            \
  static inline void N##IfaceRefArrayDispose(N##IfaceRefArray *Arr)            \
  {                                                                            \
    temu_ifaceRefArrayDispose((temu_IfaceRefArray *)Arr);                      \
  }                                                                            \
  static inline void N##IfaceRefArrayPush2(N##IfaceRefArray *Arr,              \
                                           temu_Object_ *Obj, void *Iface)     \
  {                                                                            \
    temu_ifaceRefArrayPush2((temu_IfaceRefArray *)Arr, Obj, Iface);            \
  }                                                                            \
  static inline void N##IfaceRefArrayPush(N##IfaceRefArray *Arr,               \
                                          N##IfaceRef Iface)                   \
  {                                                                            \
    temu_IfaceRef Iface2;                                                      \
    Iface2.Obj = Iface.Obj;                                                    \
    Iface2.Iface = (void *)Iface.Iface;                                        \
                                                                               \
    temu_ifaceRefArrayPush((temu_IfaceRefArray *)Arr, Iface2);                 \
  }                                                                            \
  static inline unsigned N##IfaceRefArraySize(N##IfaceRefArray *Arr)           \
  {                                                                            \
    return Arr->Size;                                                          \
  }                                                                            \
  static inline void N##IfaceRefArrayPop(N##IfaceRefArray *Arr)                \
  {                                                                            \
    Arr->Size--;                                                               \
  }

#define TEMU_DYN_ARRAY_TYPE(T, P)                                              \
  typedef struct {                                                             \
    uint32_t Size;                                                             \
    uint32_t Reserved;                                                         \
    T *Values;                                                                 \
  } temu_##P##Array;                                                           \
  static inline temu_##P##Array TEMU_MAYBE_UNUSED temu_##P##ArrayAlloc(        \
      unsigned Reserve)                                                        \
  {                                                                            \
    assert(Reserve > 0);                                                       \
    temu_##P##Array Arr;                                                       \
    Arr.Size = 0;                                                              \
    Arr.Reserved = Reserve;                                                    \
    Arr.Values = (T *)calloc(Reserve, sizeof(T));                              \
    assert(Arr.Values);                                                        \
    return Arr;                                                                \
  }                                                                            \
  static inline void TEMU_MAYBE_UNUSED temu_##P##ArrayPush(                    \
      temu_##P##Array *Arr, T Val)                                             \
  {                                                                            \
    if (Arr->Reserved >= Arr->Size) {                                          \
      T *NewValues = (T *)realloc(Arr->Values, Arr->Reserved * 2);             \
      if (NewValues) {                                                         \
        Arr->Values = NewValues;                                               \
      } else {                                                                 \
        abort();                                                               \
      }                                                                        \
    }                                                                          \
    Arr->Values[Arr->Size++] = Val;                                            \
  }                                                                            \
  static inline unsigned TEMU_MAYBE_UNUSED temu_##P##ArraySize(                \
      temu_##P##Array *Arr)                                                    \
  {                                                                            \
    return Arr->Size;                                                          \
  }                                                                            \
  static inline void TEMU_MAYBE_UNUSED temu_##P##ArrayPop(                     \
      temu_##P##Array *Arr)                                                    \
  {                                                                            \
    if (Arr->Size > 0)                                                         \
      Arr->Size--;                                                             \
  }

TEMU_DYN_ARRAY_TYPE(int8_t, i8)
TEMU_DYN_ARRAY_TYPE(int16_t, i16)
TEMU_DYN_ARRAY_TYPE(int32_t, i32)
TEMU_DYN_ARRAY_TYPE(int64_t, i64)

TEMU_DYN_ARRAY_TYPE(uint8_t, u8)
TEMU_DYN_ARRAY_TYPE(uint16_t, u16)
TEMU_DYN_ARRAY_TYPE(uint32_t, u32)
TEMU_DYN_ARRAY_TYPE(uint64_t, u64)
TEMU_DYN_ARRAY_TYPE(temu_Object_ *, object)

/*!
   Type tag

   The TEMU object system uses type tags to track which type is registered and
   in use at runtime. Type tags are used in for example the property
   registration functions.
 */
typedef enum temu_Type {
  teTY_Invalid, //!< Invalid value 0

  // C pointer sized integers
  teTY_Intptr,  //!< Pointer sized signed integer (intptr_t)
  teTY_Uintptr, //!< Pointer sized unsigned integer (uintptr_t)

  // Standard C floating point types
  teTY_Float,  //!< Single precision floating point value
  teTY_Double, //!< Double precision floating point value

  // Standard C fixed width integer types
  teTY_U8,  //!< 8-bit fixed width unsigned integer
  teTY_U16, //!< 16-bit fixed width unsigned integer
  teTY_U32, //!< 32-bit fixed width unsigned integer
  teTY_U64, //!< 64-bit fixed width unsigned integer
  teTY_I8,  //!< 8-bit fixed width signed integer
  teTY_I16, //!< 16-bit fixed width signed integer
  teTY_I32, //!< 32-bit fixed width signed integer
  teTY_I64, //!< 64-bit fixed width signed integer

  // Object pointer, must be saved as an object reference
  teTY_Obj, //!< Pointer to temu_Object

  // Internal pointer, points somewhere in the object itself
  // (e.g. bank resolution arrays) This can be saved as an offset...
  teTY_InternalPtr, //!< Internal pointer

  // Interface references (object and interface pointer pair)
  teTY_IfaceRef,      //!< Interface reference
  teTY_IfaceRefArray, //!< Dynamic object/interface array

  teTY_String, //!< C-string, useful for serialization
  teTY_Buffer, //!< Buffer (see Buffer.h)
  teTY_Dict,   //!< Dictionary
  teTY_Vector, //!< Vector (i.e. dynamic array)
  teTY_List,   //!< List
} temu_Type;

/*!
  Typed property reference.

  The property reference contains a type tag and a raw pointer to the
  property.
 */
typedef struct temu_Propref {
  temu_Type Typ; //!< Type of property
  void *Ptr;     //!< Pointer to property
} temu_Propref;

/*!
 * Named property reference
 */
typedef struct {
  temu_Object_ *Obj; //!< Object pointer
  const char *Name;  //!< Property name
} temu_PropName;

typedef void temu_Dict;

/*!
 * Dynamically sized array
 */
typedef struct {
  temu_Type Typ;     //!< Type of vector data
  void *VecData;     //!< Managed pointer, do not use directly
  uint32_t Size;     //!< Managed do not use directly
  uint32_t Capacity; //!< Managed do not use directly
} temu_Vector;

typedef void temu_ListNode;

/*!
 * Linked list type
 */
typedef struct {
  temu_Type Typ;       //!< Element type in list
  temu_ListNode *Head; //!< Managed pointer, do not use directly
  temu_ListNode *Tail; //!< Managed pointer, do not use directly
} temu_List;

/*!
  Generic property value.

  As properties can be of any normal type, the propval struct provides
  a sum type/tagged union which contain both the type tag and the
  property value.
 */
typedef struct temu_Propval {
  temu_Type Typ; //!< Value type
  union {
    intptr_t IntPtr;
    uintptr_t UIntPtr;

    float f;
    double d;

    uint8_t u8;
    uint16_t u16;
    uint32_t u32;
    uint64_t u64;

    int8_t i8;
    int16_t i16;
    int32_t i32;
    int64_t i64;

    temu_Object_ *Obj;
    temu_IfaceRef IfaceRef;
    temu_IfaceRefArray IfaceRefArray;
    const char *String;
    temu_Buff Buffer;
    temu_Dict *Dict;
    temu_Vector Vector;
    temu_List List;
  };
} temu_Propval;

/*!
  Generic constructor argument

  The object constructors takes an array of key/value pairs as parameters.
  The values are passed using either:

  - Registered argument in the `Class.new` command method.
  - Explicitly to `temu_createObject()`.
  - Args parameter to `object-create` global command.

  The object create function would typically scan through the array,
  finding relevant matching keys.
 */
typedef struct temu_CreateArg {
  const char *Key;  //!< Name of argument
  temu_Propval Val; //!< Value of argument
} temu_CreateArg;

#define TEMU_NULL_ARG                                                          \
  {                                                                            \
    NULL,                                                                      \
    {                                                                          \
      teTY_Invalid                                                             \
    }                                                                          \
  }

typedef void *(*temu_ObjectCreateFunc)(const char *Name, int Argc,
                                       const temu_CreateArg *Argv);
typedef void (*temu_ObjectDisposeFunc)(void *);

/*!
  Class object

  The TEMU object system is a dynamic object system.
  Classes are themselves represented as objects (instanciated by meta-classes).

  The TEMU class contains among other things the constructor/destructor.
  Internally, the hidden implementation provides more capabilities than is
  exposed by the `temu_Class` type.
 */
struct temu_Class {
  temu_Object Super;            //!< Super class of the class instance.
  void *Impl;                   //!< Internal pointer, do not touch.
  void *VTable;                 //!< Internal pointer, do not touch.
  temu_ObjectCreateFunc Create; //!< Constructor / create function for creating
                                //!< instances of the class.
  temu_ObjectDisposeFunc Dispose;    //!< Destructor / dispose function for
                                     //!< disposing instances of the class.
  const char *LoggingCategories[32]; // Named logging categories.
};

// Dictionary support. Dictionaries are data structure that contain
// named prop values. They are not meant for high performing code, but
// are useful for advanced configuration capabilities.  In practice
// dictionaries work for any type, but snapshots will not yet work
// for entries containing complex types (temu_Buff, temu_Vector and
// other dictionaries). This will be addressed in the future.

/*!
 * Creates a dictionary object
 * \result A pointer to the dictionary object
 */
TEMU_API temu_Dict *temu_dictCreate(void);

/*!
 * Delete a dictionary object
 *
 * \param Dict Pointer to the dictionary object to be deleted
 */
TEMU_API void temu_dictDispose(temu_Dict *Dict);

/*!
 * Inserts a name/value pair to a dictionary
 *
 * \param Dict Pointer to the dictionary object
 * \param Name The name to be inserted
 * \param Val The value to be inserted
 * \result 0 on success. Other values indicate failures (e.g. the key is already
 * used)
 */
TEMU_API int temu_dictInsertValue(temu_Dict *Dict, const char *Name,
                                  temu_Propval Val);

/*!
 * Retrieve a dictionary value by name
 *
 * \param Dict Pointer to the dictionary object
 * \param Name The name associated with the value to be retrieved
 * \result The value
 */
TEMU_API temu_Propval temu_dictGetValue(temu_Dict *Dict, const char *Name);

/*!
 * Erase a name/value pair from a dictionary by name
 *
 * \param Dict Pointer to the dictionary object
 * \param Name The name of the pair to be removed
 * \result 0 on success. Other values indicate errors (e.g. key was unavailable
 * in the dictionary)
 */
TEMU_API int temu_dictRemoveValue(temu_Dict *Dict, const char *Name);

/*!
 * Given a specific key of a dictionary, the next key is
 * provided by this function
 *
 * \param Dict The dictionary that contains the keys
 * \param Key The key, whose next value in the dict to be provided
 * \result The key that comes after Key, or nullptr on failure
 */
TEMU_API const char *temu_dictGetNextKey(temu_Dict *Dict, const char *Key);

// Typed vector support. The typed vectors can be described as a C++
// std::vector, however, they are expressed as a C-API here and
// supports snapshots.

/*!
 * Create a vector object
 *
 * \param Typ The type of the elements
 * \result the vector object created
 */
TEMU_API temu_Vector temu_vecCreate(temu_Type Typ);

/*!
 * Delete a vector object
 *
 * \param Vec The vector object to be deleted
 */
TEMU_API void temu_vecDispose(temu_Vector *Vec);

/*!
 * Add an element to a vector object
 *
 * \param Vec The vector, to which the element to be added
 * \param Val The element to be added
 */
TEMU_API void temu_vecPush(temu_Vector *Vec, temu_Propval Val);

/*!
 * Retrieve the array of elements of a vector
 *
 * \param Vec The vector, whose elements are requested
 * \result Returns a pointer to the first element of the vector
 */
TEMU_API void *temu_vecGetData(temu_Vector *Vec);

/*!
 * Get the number of elements of a vector
 *
 * \param Vec The vector, whose size to be retrieved
 * \result The number of elements in Vec
 */
TEMU_API size_t temu_vecGetSize(temu_Vector *Vec);

/*!
 * Create a linked-list object
 *
 * \param Typ The type to be stored in the elements of the linked list
 * \result A newly created list.
 */
TEMU_API temu_List temu_listCreate(temu_Type Typ);

/*!
 * Delete a linked-list object
 *
 * \param List The list to be deleted
 */
TEMU_API void temu_listDispose(temu_List *List);

/*!
 * Add an element to the end of a linked-list
 *
 * \param List The list, to which the element to be added
 * \param Val The element to be added
 */
TEMU_API void temu_listAppend(temu_List *List, temu_Propval Val);

/*!
 * Add an element to the beginning of a linked-list
 *
 * \param List The list, to which the element to be added
 * \param Val The element to be added
 */
TEMU_API void temu_listPrepend(temu_List *List, temu_Propval Val);

/*!
 * Removes the first element of a linked-list
 *
 * \param List The list, from which the first element to be removed
 * \result The element that was removed from the list
 */
TEMU_API temu_Propval temu_listRemoveHead(temu_List *List);

/*!
 * Removes the last element of a linked-list
 *
 * \param List The list, from which the last element to be removed
 * \result The element that was removed from the list
 */
TEMU_API temu_Propval temu_listRemoveTail(temu_List *List);

/*!
 * Get the first element of a linked-list
 *
 * \param List The list, whose  first element is requested
 * \result Pointer to the first element of the linked-list
 */
TEMU_API temu_ListNode *temu_listGetHead(temu_List *List);

/*!
 * Get the last element of a linked-list
 *
 * \param List The list, whose  last element is requested
 * \result Pointer to the last element of the linked-list
 */
TEMU_API temu_ListNode *temu_listGetTail(temu_List *List);

/*!
 * Get the next element of a linked list
 *
 * \param Node The node, whose next element is requested
 * \result The next element if available, otherwise nullptr
 */
TEMU_API temu_ListNode *temu_listGetNext(temu_ListNode *Node);

/*!
 * Get the previous element of a linked list
 *
 * \param Node The node, whose next element is requested
 * \result The previous element if available, otherwise nullptr
 */
TEMU_API temu_ListNode *temu_listGetPrev(temu_ListNode *Node);

/*!
 * Get value of a linked-list node
 *
 * \param Node The node, from which the value to be extracted
 * \result The value extracted from the node
 */
TEMU_API temu_Propval temu_listNodeGetVal(temu_ListNode *Node);

#ifdef PROP_ASSERTS_ENABLED
#define PROP_ASSERT(p, t) assert(p.Typ == t && "invalid property type")
#else
#define PROP_ASSERT(p, t)
#endif

// This is not the nicest solution,
// but we want to be compatible with C++ and C99, meaning:
// - cannot use designated initializers (C99, not C++)
// - cannot use constructors (C++, not C99)

#define PROP_VAL_INITIALIZER(typ, suffix, typetag, valtag)                     \
  static inline temu_Propval temu_makeProp##suffix(typ val)                    \
  {                                                                            \
    temu_Propval pv;                                                           \
    pv.Typ = typetag;                                                          \
    pv.valtag = val;                                                           \
    return pv;                                                                 \
  }                                                                            \
  static inline typ temu_propValue##suffix(temu_Propval pv)                    \
  {                                                                            \
    PROP_ASSERT(pv.Typ, typetag);                                              \
    typ val = pv.valtag;                                                       \
    return val;                                                                \
  }

PROP_VAL_INITIALIZER(intptr_t, IntPtr, teTY_Intptr, IntPtr)
PROP_VAL_INITIALIZER(uintptr_t, UIntPtr, teTY_Uintptr, UIntPtr)

PROP_VAL_INITIALIZER(float, Float, teTY_Float, f)
PROP_VAL_INITIALIZER(double, Double, teTY_Double, d)

PROP_VAL_INITIALIZER(uint8_t, U8, teTY_U8, u8)
PROP_VAL_INITIALIZER(uint16_t, U16, teTY_U16, u16)
PROP_VAL_INITIALIZER(uint32_t, U32, teTY_U32, u32)
PROP_VAL_INITIALIZER(uint64_t, U64, teTY_U64, u64)

PROP_VAL_INITIALIZER(int8_t, I8, teTY_I8, i8)
PROP_VAL_INITIALIZER(int16_t, I16, teTY_I16, i16)
PROP_VAL_INITIALIZER(int32_t, I32, teTY_I32, i32)
PROP_VAL_INITIALIZER(int64_t, I64, teTY_I64, i64)

PROP_VAL_INITIALIZER(temu_Object_ *, Obj, teTY_Obj, Obj)
PROP_VAL_INITIALIZER(temu_IfaceRef, IfaceRef, teTY_IfaceRef, IfaceRef)
PROP_VAL_INITIALIZER(const char *, String, teTY_String, String)

/*!  A prop writer function is provided when properties are
 *   registered, it is responsible for writing the property when
 *   temu_writeValue is called. The function can be used to introduce
 *   side-effects on writes unlike the setValue which will never have
 *   side-effects and only sets the raw value.
 */
typedef void (*temu_PropWriter)(void *Obj, temu_Propval Pv, int Idx);

/*!  A prop reader function is provided when properties are
 *   registered, it is responsible for reading the property when
 *   temu_readValue is called. The function can be used to introduce
 *   side-effects on reads unlike the getValue which will never have
 *   side-effects and only returns the raw value.
 */
typedef temu_Propval (*temu_PropReader)(void *Obj, int Idx);

/*!
  Get a reference to the named property.
  The propref is a typetag and a pointer to the raw value, as such it does
  not work with delegated properties in components, nor with pseudo properties.

  \param Obj An object of a class with the relevant property registered.
  \param PropName The name of the property to be queried.
  \result Object of the property reference
 */
TEMU_API temu_Propref temu_getPropref(const temu_Object_ *Obj,
                                      const char *PropName);

/**!
 * Generic property accessor
 *
 * This struct serves as a generic accessor for properties.
 * The intention for this is to be able to cache property accessors,
 * without having to disambiguate between properties and pseudo properties.
 *
 * The accessor is queried by the temu_getPropAccessor function,
 * and is then used by calling the functions:
 *
 * - `readProp`
 * - `writeProp`
 * - `getProp`
 * - `setProp`
 *
 * Depending on whether read, write, get or set semantics is needed.
 */
typedef struct temu_PropAccessor {
  temu_Type Typ;    //!< Type of property
  temu_Object *Obj; //!< Pointer to object containing property
  int Index;        //!< Index used in access, can be modified by user

  void *Data;             //!< Pointer to value if applicable
  temu_PropWriter Writer; //!< Writer function if applicable
  temu_PropReader Reader; //!< Reader function if applicable
  temu_PropWriter Setter; //!< Setter function if applicable
  temu_PropReader Getter; //!< Getter function if applicable

  temu_Propval (*readProp)(
      struct temu_PropAccessor *accessor); //!< Read property via accessor
  void (*writeProp)(struct temu_PropAccessor *accessor,
                    temu_Propval pv); //!< Write property via accessor
  temu_Propval (*getProp)(
      struct temu_PropAccessor *accessor); //!< Get property via accessor
  void (*setProp)(struct temu_PropAccessor *accessor,
                  temu_Propval pv); //!< Set property via accessor
} temu_PropAccessor;

/*!
  Get a reference to the named property

  This is useful as a replacement of temu_getPropref and temu_getPropName.
  The resulting data structure lets you automatically disambiguate between
  properties and pseudo properties.

  The index used will be PropIdx,
  or K if the PropName string has an array suffix `[K]`.

  \param Obj An object of a class with the relevant property registered.
  \param PropName The name of the property to be queried.
  \param PropIdx Index in property array.
  \result A struct containing prop accessor data.
 */

TEMU_API temu_PropAccessor temu_getPropAccessor(temu_Object *Obj,
                                                const char *PropName,
                                                int PropIdx);

/*!
  Get a reference to a named property.

  \param Obj An object of a class with the relevant property registered.
  \param PropName The name of the property to be queried.

  \result Prop name object, note that the returned name string is not
          the same as the passed string. The returned string has a lifetime
  bound to the class, hence it can be returned without any problems.
 */
TEMU_API temu_PropName temu_getPropName(const temu_Object_ *Obj,
                                        const char *PropName);

/*!
 * Returns property length/size if a property is an array
 *
 * \param Obj Pointer to the object of the property
 * \param PropName Name of the property
 * \result The length of the property
 */
TEMU_API int temu_getPropLength(const temu_Object_ *Obj, const char *PropName);

/*!
 * Get the dynamic length of the property
 *
 * Normally, the `temu_getPropLength()` would return 1 for a dynamic array.
 * This means that there is one dynamic array.
 * In case the length of the dynamic array is needed, this function can be used.
 *
 * \param Obj Pointer to the object of the property
 * \param PropName Name of the property
 * \result Dynamic length of property
 */
TEMU_API int temu_getPropDynLength(const temu_Object_ *Obj,
                                   const char *PropName);

/*!
 * Get the type of a property
 *
 * \param Obj Pointer to the object of the property
 * \param PropName Name of the property
 * \result Property type
 */
TEMU_API temu_Type temu_getPropType(const temu_Object_ *Obj,
                                    const char *PropName);

/*!
 * Check if the object has a named property
 *
 * \param Obj Pointer to the object of the property
 * \param PropName Name of the property
 * \result 0 if the property does not exist, otherwise the property is valid
 */
TEMU_API int temu_objectHasProp(const temu_Object_ *Obj, const char *PropName);

/*!
 * Check if the object has a named interface
 *
 * \param Obj Pointer to the object to query
 * \param IfaceName Interface name to look for
 * \result 0 if the interface does not exist, otherwise the interface exists
 */

TEMU_API int temu_objectHasIface(const temu_Object_ *Obj,
                                 const char *IfaceName);

/*!
 * Checks whether a property is a number
 *
 * \param Pv The property to be checked
 * \result 0 for false, non-zero for true
 */
TEMU_API int temu_isNumber(temu_Propval Pv);

/*!
 * Checks whether a property is a real number
 *
 * \param Pv The property to be checked
 * \result 0 for false, non-zero for true
 */
TEMU_API int temu_isReal(temu_Propval Pv);

/*!
 * Checks whether the numeric property is an integer-like
 *
 * \param Pv The property to be checked
 * \result 0 for false, non-zero for true
 */
TEMU_API int temu_isDiscrete(temu_Propval Pv);

TEMU_API int temu_isUnsigned(temu_Propval Pv);
TEMU_API int temu_isSigned(temu_Propval Pv);

/*!
 * Checks whether a property is a string
 *
 * \param Pv The property to be checked
 * \result 0 for false, non-zero for true
 */
TEMU_API int temu_isString(temu_Propval Pv);

/*!
 * Converts the given propery to integer
 *
 * \param Pv The property to be converted
 * \result Returns the property as integer
 */
TEMU_API int64_t temu_asInteger(temu_Propval Pv);

/*!
 * Converts the given property to unsigned integer
 *
 * \param Pv The property to be converted
 * \result Returns the property as an unsigned integer
 */
TEMU_API uint64_t temu_asUnsigned(temu_Propval Pv);

/*!
 * Converts the given property to double
 *
 * \param Pv The property to be converted
 * \result Returns the property as an unsigned integer
 */
TEMU_API double temu_asDouble(temu_Propval Pv);

/*!
 * Get the value of a property from an object
 *
 * \param Obj The object that contains the property
 * \param PropName Property name
 * \param Idx Index of the property. Set to 0 if the property is not an array
 * \result The value of the property
 */
TEMU_API temu_Propval temu_getValue(temu_Object_ *Obj, const char *PropName,
                                    int Idx) TEMU_NO_WRAP;

/*!
 * Get the value of a property from an object if it is an uint8
 *
 * \param Obj The object that contains the property
 * \param PropName Property name
 * \param Idx Index of the property. Set to 0 if the property is not an array
 * \result The value of the property
 */
TEMU_API uint8_t temu_getValueU8(temu_Object_ *Obj, const char *PropName,
                                 int Idx);

/*!
 * Get the value of a property from an object if it is an uint16
 *
 * \param Obj The object that contains the property
 * \param PropName Property name
 * \param Idx Index of the property. Set to 0 if the property is not an array
 * \result The value of the property
 */
TEMU_API uint16_t temu_getValueU16(temu_Object_ *Obj, const char *PropName,
                                   int Idx);

/*!
 * Get the value of a property from an object if it is an uint32
 *
 * \param Obj The object that contains the property
 * \param PropName Property name
 * \param Idx Index of the property. Set to 0 if the property is not an array
 * \result The value of the property
 */
TEMU_API uint32_t temu_getValueU32(temu_Object_ *Obj, const char *PropName,
                                   int Idx);

/*!
 * Get the value of a property from an object if it is an uint64
 *
 * \param Obj The object that contains the property
 * \param PropName Property name
 * \param Idx Index of the property. Set to 0 if the property is not an array
 * \result The value of the property
 */
TEMU_API uint64_t temu_getValueU64(temu_Object_ *Obj, const char *PropName,
                                   int Idx);

/*!
 * Get the value of a property from an object if it is an int8
 *
 * \param Obj The object that contains the property
 * \param PropName Property name
 * \param Idx Index of the property. Set to 0 if the property is not an array
 * \result The value of the property
 */
TEMU_API int8_t temu_getValueI8(temu_Object_ *Obj, const char *PropName,
                                int Idx);

/*!
 * Get the value of a property from an object if it is an int16
 *
 * \param Obj The object that contains the property
 * \param PropName Property name
 * \param Idx Index of the property. Set to 0 if the property is not an array
 * \result The value of the property
 */
TEMU_API int16_t temu_getValueI16(temu_Object_ *Obj, const char *PropName,
                                  int Idx);

/*!
 * Get the value of a property from an object if it is an int32
 *
 * \param Obj The object that contains the property
 * \param PropName Property name
 * \param Idx Index of the property. Set to 0 if the property is not an array
 * \result The value of the property
 */
TEMU_API int32_t temu_getValueI32(temu_Object_ *Obj, const char *PropName,
                                  int Idx);

/*!
 * Get the value of a property from an object if it is an int64
 *
 * \param Obj The object that contains the property
 * \param PropName Property name
 * \param Idx Index of the property. Set to 0 if the property is not an array
 * \result The value of the property
 */
TEMU_API int64_t temu_getValueI64(temu_Object_ *Obj, const char *PropName,
                                  int Idx);

/*!
 * Read a property value with side-effects
 *
 * \param Obj Object pointer
 * \param PropName Name of property to read.
 * \param Idx Index of property, set to 0 if it is not an array.
 * \result The property value corresponding to the name. In case of
 *         the property not being found, the Typ field of the returned
 *         property will be teTY_Invalid.
 */
TEMU_API temu_Propval temu_readValue(temu_Object_ *Obj, const char *PropName,
                                     int Idx) TEMU_NO_WRAP;

/*!
 * Read typed properties.
 *
 * The readValue[U|I][8|16|32|64] function reads a typed property. The
 * accessor will fail with a fatal error if the read type is not the
 * same as the property type. If the property type is unknown, use the
 * temu_readValue function instead.
 *
 * \param Obj Object pointer
 * \param PropName Name of property to read
 * \param Idx Index in property array, set to 0 if the property is not an array
 * \result The read out property value.
 */
TEMU_API uint8_t temu_readValueU8(temu_Object_ *Obj, const char *PropName,
                                  int Idx);

/*!
 * Read typed properties.
 *
 * The readValue[U|I][8|16|32|64] function reads a typed property. The
 * accessor will fail with a fatal error if the read type is not the
 * same as the property type. If the property type is unknown, use the
 * temu_readValue function instead.
 *
 * \param Obj Object pointer
 * \param PropName Name of property to read
 * \param Idx Index in property array, set to 0 if the property is not an array
 * \result The read out property value.
 */
TEMU_API uint16_t temu_readValueU16(temu_Object_ *Obj, const char *PropName,
                                    int Idx);

/*!
 * Read typed properties.
 *
 * The readValue[U|I][8|16|32|64] function reads a typed property. The
 * accessor will fail with a fatal error if the read type is not the
 * same as the property type. If the property type is unknown, use the
 * temu_readValue function instead.
 *
 * \param Obj Object pointer
 * \param PropName Name of property to read
 * \param Idx Index in property array, set to 0 if the property is not an array
 * \result The read out property value.
 */
TEMU_API uint32_t temu_readValueU32(temu_Object_ *Obj, const char *PropName,
                                    int Idx);

/*!
 * Read typed properties.
 *
 * The readValue[U|I][8|16|32|64] function reads a typed property. The
 * accessor will fail with a fatal error if the read type is not the
 * same as the property type. If the property type is unknown, use the
 * temu_readValue function instead.
 *
 * \param Obj Object pointer
 * \param PropName Name of property to read
 * \param Idx Index in property array, set to 0 if the property is not an array
 * \result The read out property value.
 */
TEMU_API uint64_t temu_readValueU64(temu_Object_ *Obj, const char *PropName,
                                    int Idx);

/*!
 * Read typed properties.
 *
 * The readValue[U|I][8|16|32|64] function reads a typed property. The
 * accessor will fail with a fatal error if the read type is not the
 * same as the property type. If the property type is unknown, use the
 * temu_readValue function instead.
 *
 * \param Obj Object pointer
 * \param PropName Name of property to read
 * \param Idx Index in property array, set to 0 if the property is not an array
 * \result The read out property value.
 */
TEMU_API int8_t temu_readValueI8(temu_Object_ *Obj, const char *PropName,
                                 int Idx);

/*!
 * Read typed properties.
 *
 * The readValue[U|I][8|16|32|64] function reads a typed property. The
 * accessor will fail with a fatal error if the read type is not the
 * same as the property type. If the property type is unknown, use the
 * temu_readValue function instead.
 *
 * \param Obj Object pointer
 * \param PropName Name of property to read
 * \param Idx Index in property array, set to 0 if the property is not an array
 * \result The read out property value.
 */
TEMU_API int16_t temu_readValueI16(temu_Object_ *Obj, const char *PropName,
                                   int Idx);

/*!
 * Read typed properties.
 *
 * The readValue[U|I][8|16|32|64] function reads a typed property. The
 * accessor will fail with a fatal error if the read type is not the
 * same as the property type. If the property type is unknown, use the
 * temu_readValue function instead.
 *
 * \param Obj Object pointer
 * \param PropName Name of property to read
 * \param Idx Index in property array, set to 0 if the property is not an array
 * \result The read out property value.
 */
TEMU_API int32_t temu_readValueI32(temu_Object_ *Obj, const char *PropName,
                                   int Idx);

/*!
 * Read typed properties.
 *
 * The readValue[U|I][8|16|32|64] function reads a typed property. The
 * accessor will fail with a fatal error if the read type is not the
 * same as the property type. If the property type is unknown, use the
 * temu_readValue function instead.
 *
 * \param Obj Object pointer
 * \param PropName Name of property to read
 * \param Idx Index in property array, set to 0 if the property is not an array
 * \result The read out property value.
 */
TEMU_API int64_t temu_readValueI64(temu_Object_ *Obj, const char *PropName,
                                   int Idx);

/*!
 * Set a raw property value without side-effects
 *
 * \param Obj Object pointer
 * \param PropName Name of property to read.
 * \param Val The value to set. It must be of the correct type.
 * \param Idx Index of property, set to 0 if it is not an array.
 */
TEMU_API void temu_setValue(temu_Object_ *Obj, const char *PropName,
                            temu_Propval Val, int Idx) TEMU_NO_WRAP;

/*!
 * Set a raw property value without side-effects
 *
 * \param Obj Object pointer
 * \param PropName Name of property to read.
 * \param Val The value to set. It must be of the correct type.
 * \param Idx Index of property, set to 0 if it is not an array.
 */
TEMU_API void temu_setValueU8(temu_Object_ *Obj, const char *PropName,
                              uint8_t Val, int Idx);

/*!
 * Set a raw property value without side-effects
 *
 * \param Obj Object pointer
 * \param PropName Name of property to read.
 * \param Val The value to set. It must be of the correct type.
 * \param Idx Index of property, set to 0 if it is not an array.
 */
TEMU_API void temu_setValueU16(temu_Object_ *Obj, const char *PropName,
                               uint16_t Val, int Idx);

/*!
 * Set a raw property value without side-effects
 *
 * \param Obj Object pointer
 * \param PropName Name of property to read.
 * \param Val The value to set. It must be of the correct type.
 * \param Idx Index of property, set to 0 if it is not an array.
 */
TEMU_API void temu_setValueU32(temu_Object_ *Obj, const char *PropName,
                               uint32_t Val, int Idx);

/*!
 * Set a raw property value without side-effects
 *
 * \param Obj Object pointer
 * \param PropName Name of property to read.
 * \param Val The value to set. It must be of the correct type.
 * \param Idx Index of property, set to 0 if it is not an array.
 */
TEMU_API void temu_setValueU64(temu_Object_ *Obj, const char *PropName,
                               uint64_t Val, int Idx);

/*!
 * Set a raw property value without side-effects
 *
 * \param Obj Object pointer
 * \param PropName Name of property to read.
 * \param Val The value to set. It must be of the correct type.
 * \param Idx Index of property, set to 0 if it is not an array.
 */
TEMU_API void temu_setValueI8(temu_Object_ *Obj, const char *PropName,
                              int8_t Val, int Idx);

/*!
 * Set a raw property value without side-effects
 *
 * \param Obj Object pointer
 * \param PropName Name of property to read.
 * \param Val The value to set. It must be of the correct type.
 * \param Idx Index of property, set to 0 if it is not an array.
 */
TEMU_API void temu_setValueI16(temu_Object_ *Obj, const char *PropName,
                               int16_t Val, int Idx);

/*!
 * Set a raw property value without side-effects
 *
 * \param Obj Object pointer
 * \param PropName Name of property to read.
 * \param Val The value to set. It must be of the correct type.
 * \param Idx Index of property, set to 0 if it is not an array.
 */
TEMU_API void temu_setValueI32(temu_Object_ *Obj, const char *PropName,
                               int32_t Val, int Idx);

/*!
 * Set a raw property value without side-effects
 *
 * \param Obj Object pointer
 * \param PropName Name of property to read.
 * \param Val The value to set. It must be of the correct type.
 * \param Idx Index of property, set to 0 if it is not an array.
 */
TEMU_API void temu_setValueI64(temu_Object_ *Obj, const char *PropName,
                               int64_t Val, int Idx);

/*!
 * Write a property value with side-effects
 *
 * \param Obj Object pointer
 * \param PropName Name of property to read.
 * \param Val The value to set. It must be of the correct type.
 * \param Idx Index of property, set to 0 if it is not an array.
 */
TEMU_API void temu_writeValue(temu_Object_ *Obj, const char *PropName,
                              temu_Propval Val, int Idx) TEMU_NO_WRAP;

/*!
 * Write a property value with side-effects
 *
 * \param Obj Object pointer
 * \param PropName Name of property to read.
 * \param Val The value to set. It must be of the correct type.
 * \param Idx Index of property, set to 0 if it is not an array.
 */
TEMU_API void temu_writeValueU8(temu_Object_ *Obj, const char *PropName,
                                uint8_t Val, int Idx);

/*!
 * Write a property value with side-effects
 *
 * \param Obj Object pointer
 * \param PropName Name of property to read.
 * \param Val The value to set. It must be of the correct type.
 * \param Idx Index of property, set to 0 if it is not an array.
 */
TEMU_API void temu_writeValueU16(temu_Object_ *Obj, const char *PropName,
                                 uint16_t Val, int Idx);

/*!
 * Write a property value with side-effects
 *
 * \param Obj Object pointer
 * \param PropName Name of property to read.
 * \param Val The value to set. It must be of the correct type.
 * \param Idx Index of property, set to 0 if it is not an array.
 */
TEMU_API void temu_writeValueU32(temu_Object_ *Obj, const char *PropName,
                                 uint32_t Val, int Idx);

/*!
 * Write a property value with side-effects
 *
 * \param Obj Object pointer
 * \param PropName Name of property to read.
 * \param Val The value to set. It must be of the correct type.
 * \param Idx Index of property, set to 0 if it is not an array.
 */
TEMU_API void temu_writeValueU64(temu_Object_ *Obj, const char *PropName,
                                 uint64_t Val, int Idx);

/*!
 * Write a property value with side-effects
 *
 * \param Obj Object pointer
 * \param PropName Name of property to read.
 * \param Val The value to set. It must be of the correct type.
 * \param Idx Index of property, set to 0 if it is not an array.
 */
TEMU_API void temu_writeValueI8(temu_Object_ *Obj, const char *PropName,
                                int8_t Val, int Idx);

/*!
 * Write a property value with side-effects
 *
 * \param Obj Object pointer
 * \param PropName Name of property to read.
 * \param Val The value to set. It must be of the correct type.
 * \param Idx Index of property, set to 0 if it is not an array.
 */
TEMU_API void temu_writeValueI16(temu_Object_ *Obj, const char *PropName,
                                 int16_t Val, int Idx);

/*!
 * Write a property value with side-effects
 *
 * \param Obj Object pointer
 * \param PropName Name of property to read.
 * \param Val The value to set. It must be of the correct type.
 * \param Idx Index of property, set to 0 if it is not an array.
 */
TEMU_API void temu_writeValueI32(temu_Object_ *Obj, const char *PropName,
                                 int32_t Val, int Idx);

/*!
 * Write a property value with side-effects
 *
 * \param Obj Object pointer
 * \param PropName Name of property to read.
 * \param Val The value to set. It must be of the correct type.
 * \param Idx Index of property, set to 0 if it is not an array.
 */
TEMU_API void temu_writeValueI64(temu_Object_ *Obj, const char *PropName,
                                 int64_t Val, int Idx);

/*!
 * Write a property value with side-effects
 *
 * \param Obj Object pointer
 * \param PropName Name of property to read.
 * \param Val The value to set. It must be of the correct type.
 * \param Idx Index of property, set to 0 if it is not an array.
 */
TEMU_API void temu_writeValueObj(temu_Object *Obj, const char *PropName,
                                 temu_Object *Val, int Idx);

/*!
 * Register a class in the TEMU object system. With these classes, the TEMU
 * object system is responsible for allocation and de-allocation.
 *
 * \param ClsName The name of the class that is to be created.
 * \param Create A constructor, the constructor is responsible for
 *               allocation and initialisation of the object.
 * \param Dispose A destructor, this function is responsible for deleting
 *                the objects of the class.
 * \result A pointer to the class object that can be further used to register
 *         properties and interfaces.
 */
TEMU_API temu_Class *temu_registerClass(const char *ClsName,
                                        temu_ObjectCreateFunc Create,
                                        temu_ObjectDisposeFunc Dispose);

#ifdef __cplusplus

/*!
 * Add a property to a class.
 *
 * \param Cls The class object
 * \param PropName Name of the property to add.
 *
 * \param Offset Offset in bytes from the start of the class to the
 *               value the property refers to.
 *
 * \param Typ Type of the property.
 *
 * \param Count Set to > 1 to indicate that the property is an array,
 *              the value is the length of the array.
 *
 * \param Wr Property write function. This function can have side-effects.
 * \param Rd Property read function. This function can have side-effects.
 * \param Doc Documentation string
 */
TEMU_API void temu_addProperty(temu_Class *Cls, const char *PropName,
                               int Offset, temu_Type Typ, int Count,
                               temu_PropWriter Wr = nullptr,
                               temu_PropReader Rd = nullptr,
                               const char *Doc = "");
#else

/*!
 * Add a property to a class.
 *
 * \param Cls The class object
 * \param PropName Name of the property to add.
 *
 * \param Offset Offset in bytes from the start of the class to the
 *               value the property refers to.
 *
 * \param Typ Type of the property.
 *
 * \param Count Set to > 1 to indicate that the property is an array,
 *              the value is the length of the array.
 *
 * \param Wr Property write function. This function can have side-effects.
 * \param Rd Property read function. This function can have side-effects.
 * \param Doc Documentation string
 */
TEMU_API void temu_addProperty(temu_Class *Cls, const char *PropName,
                               int Offset, temu_Type Typ, int Count,
                               temu_PropWriter Wr, temu_PropReader Rd,
                               const char *Doc);

#endif

/*!
 * Add property without storage
 *
 * A pseudo property does not have backing storage (and therefore no
 * offset). They exist to provide access to computed properties.
 *
 * Pseudo properties also work fine with C++ types with non-standard
 * layout.
 *
 * Pseudo properties are snapshotted if the set and get functions are
 * implemented. The set and get should implement read and write
 * without semantic effect. That is, if the property is a register,
 * then the read and write should have the effect of an actual
 * register access (including event rescheduling etc), while for a
 * get/set, only the value of the register is modified, event
 * restoration is done differently.
 *
 * \param Cls class to add pseudo property to
 * \param PropName name of pseudo property
 * \param Typ Type of property
 * \param Count number of elements, 1 for a scalar.
 * \param Wr Property write function (semantic effect)
 * \param Rd Property read function (semantic effect)
 * \param Set Property set function (non-semantic)
 * \param Get Property get function (non-semantic)
 * \param Doc Documentation string
 */
TEMU_API void temu_addPseudoProperty(temu_Class *Cls, const char *PropName,
                                     temu_Type Typ, int Count,
                                     temu_PropWriter Wr, temu_PropReader Rd,
                                     temu_PropWriter Set, temu_PropReader Get,
                                     const char *Doc);

/*!
 * temu_addLogginCategory adds a logging category to the class
 *
 * \param Cls the class
 * \param CategoryId The category id. Valid range is [8,16)
 * \param Category The category name
 * \result 0 for success, 1 indicates failure.
 */
TEMU_API int temu_addLoggingCategory(temu_Class *Cls, unsigned CategoryId,
                                     const char *Category);

/*!
 * temu_getLogginCategory adds a logging category to the class
 *
 * \param Cls the class
 * \param CategoryId The category id. Valid range is [8,16)
 * \result Pointer to the category string, NULL for failure.
 */
TEMU_API const char *temu_getLoggingCategory(temu_Class *Cls,
                                             unsigned CategoryId);

/*!
 * temu_isPseudoProperty Checks whether a property is a pseudo-property
 *
 * \param Obj The object that contains the property
 * \param PropName The property name
 * \result 0 for false, non-zero for true
 */
TEMU_API int temu_isPseudoProperty(temu_Object_ *Obj, const char *PropName);

/*!
 * temu_isNormalProperty Checks whether a property is a normal property (not a
 * pseudo-property)
 *
 * \param Obj The object that contains the property
 * \param PropName The property name
 * \result 0 for false, non-zero for true
 */
TEMU_API int temu_isNormalProperty(temu_Object_ *Obj, const char *PropName);

/*!
 * Associate the interface reference property with an interface type.
 *
 * Setting the type, prevents the connect command / function to assign
 * the connection if the target interface type is not the same as the
 * property interface reference type.
 *
 * \param Cls Class pointer
 * \param PropName Name of interface reference property
 * \param IfaceType Type required to connect the property
 */
TEMU_API void temu_requireInterface(temu_Class *Cls, const char *PropName,
                                    const char *IfaceType);

/*!
 * Add a interface reference property.
 *
 * \param Cls The class object
 * \param PropName Name of the property to add.
 *
 * \param Offset Offset in bytes from the start of the class to the
 *               value the property refers to.
 *
 * \param TypeName Interface type name.
 *
 * \param Count Set to > 1 to indicate that the property is an array,
 *              the value is the length of the array.
 * \param Flags Flags for property, 1 implies property is mandatory to set.
 * \param Wr Property write function. This function can have side-effects.
 * \param Rd Property read function. This function can have side-effects.
 * \param Doc Documentation string
 */
TEMU_API void temu_addInterfaceReference(temu_Class *Cls, const char *PropName,
                                         int Offset, const char *TypeName,
                                         int Count, unsigned Flags,
                                         temu_PropWriter Wr, temu_PropReader Rd,
                                         const char *Doc);

/*!
 * Add a interface reference property.
 *
 * \param Cls The class object
 * \param PropName Name of the property to add.
 *
 * \param TypeName Interface type name.
 *
 * \param Count Set to > 1 to indicate that the property is an array,
 *              the value is the length of the array.
 * \param Flags Flags for property, 1 implies property is mandatory to set.
 * \param Wr Property write function. This function can have side-effects.
 * \param Rd Property read function. This function can have side-effects.
 * \param Set Property set function. Non-semantic.
 * \param Get Property get function. Non-semantic.
 * \param Doc Documentation string
 */
TEMU_API void temu_addPseudoInterfaceReference(
    temu_Class *Cls, const char *PropName, const char *TypeName, int Count,
    unsigned Flags, temu_PropWriter Wr, temu_PropReader Rd, temu_PropWriter Set,
    temu_PropReader Get, const char *Doc);

/*!
 * Make interface reference property and an interface inverses
 *
 * A port is a bidirectional interface. Thus, if an interface
 * reference is connected to another objects interface, the remote
 * object will be told to issue a reverse connection. This is
 * convenient as it simplifies device connection during system
 * configuration and construction. That is, instead of calling the
 * connect two times with the inverse connections explicitly, only one
 * call is needed.
 *
 * \param C The class object
 * \param IfaceRefName the name of the interface reference property
 * \param IfaceName Name of interface that is the inverse of the
 *        interface ref property.
 * \param Doc Documentation string
 * \result 0 on success, non-zero otherwise.
 */
TEMU_API int temu_addPort(temu_Class *C, const char *IfaceRefName,
                          const char *IfaceName, const char *Doc);

/*!
 * Register interface with the named class.
 *
 * \param Cls_name Name of class to add interface to
 * \param Ifacename Interface name
 * \param IfaceType Interface type
 * \param Iface Pointer to the interface structure.
 */
#ifdef __cplusplus

/*!
 * Add an interface to a class
 *
 * \param Cls Class pointer, returned from registerClass or similar
 * \param IfaceName Name of the interface
 * \param IfaceType Name of the interface type, note that the type
 *                  names are typically defined together with the
 *                  interface C-type.
 * \param Iface Pointer to interface struct
 * \param DeprecatedParam Not used, set to zero
 * \param Doc Description of interface
 */
TEMU_API void temu_addInterface(temu_Class *Cls, const char *IfaceName,
                                const char *IfaceType, void *Iface,
                                int DeprecatedParam = 0, const char *Doc = "");

/*!
 * temu_getInterface Get an interface from an object by name
 *
 * \param Obj Pointer to the object containing the interface
 * \param IfaceName Name of the interface
 * \param Idx Index in case the interface name refers to an
 *            interface array
 * \result Pointer to the interface
 */
TEMU_API void *temu_getInterface(temu_Object_ *Obj, const char *IfaceName,
                                 int Idx = 0);

#else
TEMU_API void temu_addInterface(temu_Class *Cls,

                                const char *IfaceName, const char *IfaceType,
                                void *Iface, int DeprecatedParam,
                                const char *Doc);

TEMU_API void *temu_getInterface(temu_Object_ *Obj, const char *IfaceName,
                                 int Idx);

#endif

/*!
 * Retrieve a reference to an interface identified by its name and index
 *
 * \param Obj Pointer to the object containing the interface
 * \param IfaceName Name of the interface
 * \param Idx Interface index
 * \result The interface reference as a temu_IfaceRef object
 */
TEMU_API temu_IfaceRef temu_getInterfaceRef(temu_Object_ *Obj,
                                            const char *IfaceName, int Idx);

/*!
 * temu_getInterfaceName Get Interface Name of the Obj
 *
 * \param Obj Pointer to the object containing the interface
 * \param Iface Interface to be find w.r.t. Object
 * \result The name of the interface
 */
TEMU_API const char *temu_getInterfaceName(temu_Object *Obj, void *Iface);

/*!
 * Add an array of interfaces to a class
 *
 * Interface arrays are especially useful when e.g. multiple "network
 * ports" are available, although they can be added manually with
 * distinct names (e.g. uartaiface, uartbiface, etc), the interface
 * array registration allows for the addition of several interfaces at
 * once. Individual interfaces are then referred to with normal index
 * syntax, e.g. obj:uartiface[0].
 *
 * \param Cls Class pointer
 * \param IfaceName name of interface
 * \param IfaceType name of interface type
 * \param Iface Pointer to array of interfaces
 * \param Count Number of interfaces in array
 * \param Size Size of one interface (e.g. sizeof(uartiface[0]))
 * \param Doc Documentation comment for interface
 */
TEMU_API void temu_addInterfaceArray(temu_Class *Cls, const char *IfaceName,
                                     const char *IfaceType, void *Iface,
                                     size_t Count, size_t Size,
                                     const char *Doc);

/*!
 * Set the VTable pointer.
 *
 * Temu classes, provides the ability to, for internal objects query
 * for a VTable manually in O(1) time. In practice this can be any
 * pointer, but it is typically used to register performance sensitive
 * interfaces. E.g. the CPU and machine classes have their own VTable
 * types that refer to a number of interfaces they implement.
 *
 * \param Cls The class, for which the vtable to be set
 * \param VTable Pointer to the vtable to be used
 * \result 0 on success, non-zero otherwise.
 */
TEMU_API int temu_setVTable(temu_Class *Cls, void *VTable);

/*!
 * Get the VTable pointer for a class
 *
 * \param Cls The class, for which the vtable is to be retrieved
 * \result Pointer to the vtable in question
 */
TEMU_API void *temu_getVTableForClass(temu_Class *Cls);

/*!
 * Get the VTable pointer for an object. This only works for internal
 * objects that inherits from temu_Object.
 *
 * \param Obj The object in question
 * \result Pointer to the vtable in question
 */
TEMU_API void *temu_getVTable(const temu_Object_ *Obj);

/*!
 * Set time source object for models. Typically TS is a CPU. You can
 * only set the time source for an internal class.
 *
 * \param Obj Object to set time source in.
 * \param TS Time source object (CPU or clock model)
 */
TEMU_API void temu_setTimeSource(temu_Object_ *Obj, temu_TimeSource_ *TS);

/*
 * Qualification support
 *
 * Qualifiers can set a tag per class, which can then be used for
 * quick identification of an object's class properties. For example
 * all objects qualified as CPUs, machines and memories must have the
 * vtable set appropriatelly.
 */
#define TEMU_QUAL_NONE 0
#define TEMU_QUAL_CPU 1
#define TEMU_QUAL_MACHINE 2
#define TEMU_QUAL_MEMORY 4
#define TEMU_QUAL_COMPONENT 5
#define TEMU_QUAL_CLOCK 6

// Users can set their own class qualifiers, but should use an offset
// from the TEMU_QUAL_USER.
#define TEMU_QUAL_USER 65536

/*!
 * temu_isQualifiedAs Check if an object is qualified as Qualifier
 *
 * \param Obj The object to be checked
 * \param Qualifier The qualification to be checked
 * \result 0 for false, non-zero for true
 */
TEMU_API int temu_isQualifiedAs(const temu_Object_ *Obj, unsigned Qualifier);

/*!
 * Predicate for identifying CPU classes. In release without assert
 * builds this runs in O(1) time.
 *
 * \param Obj The object to test
 * \result 0 for false, non-zero for true
 */
TEMU_API int temu_isCpu(const temu_Object_ *Obj);

/*!
 * Predicate for identifying CPU classes. In release without assert
 * builds this runs in O(1) time..
 *
 * \param Obj The object to test
 * \result 0 for false, non-zero for true
 */
TEMU_API int temu_isMachine(const temu_Object_ *Obj);

/*!
 * temu_isMemory Checks whether an object is a memory object
 *
 * \param Obj The object to test
 * \result 0 for false, non-zero for true
 */
TEMU_API int temu_isMemory(const temu_Object_ *Obj);

/*!
 * temu_isComponent Checks if an object is a component
 *
 * \param Obj The object to test
 * \result 0 for false, non-zero for true
 */
TEMU_API int temu_isComponent(const temu_Object_ *Obj);

/*!
 * Bless the class so that the isCpu predicate returns 1. There are
 * certain assumptions that CPU classes must meet. These assumptions
 * are not stable yet, so do not use this in user code.
 *
 * \param Cls The class to be blessed
 */
TEMU_API void temu_qualifyAsCpu(temu_Class *Cls);

/*!
 * Bless the class so that the isMachine predicate returns 1. There
 * are certain assumptions that machine classes must meet. These
 * assumptions are not stable yet, so do not use this in user code.
 *
 * \param Cls The class to be blessed
 */
TEMU_API void temu_qualifyAsMachine(temu_Class *Cls);

/*!
 * Bless the class so that the isMemory predicate returns 1
 *
 * \param Cls The class to be blessed
 */
TEMU_API void temu_qualifyAsMemory(temu_Class *Cls);

/*!
 * Bless the class so that the isQualified predicate returns 1.
 * Qualifiers with the MSb cleared are reserved for TEMU internals.
 *
 * \param Cls The class to be blessed
 */
TEMU_API void temu_qualifyAs(temu_Class *Cls, unsigned Qualifier);

/*! Erase all classes, interfaces, properties and objects registered
 *  in the object system.
 */
TEMU_API void temu_objsysClear(void);

/*! Erase all objects, but do not delete classes.
 */
TEMU_API void temu_objsysClearObjects(void);

/*! Create object with class and name.
 *
 * \param ClsName Name of class used for instantiation
 * \param ObjName Name of object in object system
 *
 * \param Args Named argument pairs to pass into constructor. The list
 *             of args should be terminated by a
 *             TEMU_NULL_ARG. In case no arguments are passed,
 *             pass in NULL.
 *
 * \result Allocated object. Result is NULL in case of: class does not exist,
 *         object already allocated with the given name, out of memory.
 */
TEMU_API temu_Object_ *temu_createObject(const char *ClsName,
                                         const char *ObjName,
                                         const temu_CreateArg *Args);

/*! Dispose object. The function will call the Objects dispose
 *  function.
 *
 * \param Obj The object to delete.
 */
TEMU_API void temu_disposeObject(temu_Object_ *Obj);

/*!
 * Get the class pointer for the named class
 *
 * \param ClsName The name of the class
 * \result Pointer to the class if found, otherwise NULL
 */
TEMU_API temu_Class *temu_classForName(const char *ClsName);

/*!
 * temu_nameForClass Get the class name from its object
 *
 * \param Cls Pointer to the object of the class in question
 * \result The name of the class as a C-string
 */
TEMU_API const char *temu_nameForClass(temu_Class *Cls);

/*!
 * Get the class for the given object.
 *
 * \param Obj Pointer to the object
 * \result Pointer to the class type object
 */
TEMU_API temu_Class *temu_classForObject(const temu_Object_ *Obj);

/*!
 * Query whether the class has a specific command implemented
 *
 * \param Cls class to query for a command.
 * \param CmdName Command name class is expected to implement.
 * \result 0 if command is not implemented by class, 1 if implemented.
 */
TEMU_API int temu_classHasCommand(const temu_Class *Cls, const char *CmdName);

typedef void temu_TypeObject;

typedef struct {
  const char *Name; //!< Name of property
  temu_Type Typ;    //!< Type tag
  size_t Count;     //!< Number of elements in property
  uintptr_t Offset; //!< Offset from struct start
  temu_TypeObject *TypeObj;
} temu_PropInfo;

/*!
 * Get property info for class
 *
 * It is possible to extract low level property info data using this
 * function. This can be useful if integrating in another simulator or
 * if integrating the system in e.g. a GUI in which you need to
 * provide object introspection.
 *
 * The function can be called with the PI array set to NULL to return
 * the number of properties in the class.
 *
 * You can read out all property fields using the following sequence:
 * \code{.c}
 *   temu_PropInfo PI; unsigned i = 0; int r;
 *   while ((r = temu_propInfoForClass(C, i, 1, &PI)) == 1) {
 *     // ...
 *     i += r;
 *   }
 * \endcode
 *
 * \param Cls The class to inspect
 * \param PIIndex The property index to read from.
 * \param PICount Size of PI array in number of entries.
 * \Param PI Pointer to an array of prop info objects to fill in.
 * \result Number of read PI entries. In case PI is NULL, the total
 *         number of PIs for the class.
 */
TEMU_API int temu_propInfoForClass(temu_Class *Cls, unsigned PIIndex,
                                   unsigned PICount, temu_PropInfo *PI);

/*!
 * Get object for name
 *
 * \param Name The name of the object
 * \result Pointer to the object
 */
TEMU_API temu_Object_ *temu_objectForName(const char *Name);

/*!
 * Get the name of an object
 *
 * \param Obj Pointer to the object
 * \result Name of the object as C-string
 */
TEMU_API const char *temu_nameForObject(const temu_Object_ *Obj);

/*!
 * Get name for interface
 *
 * \param Obj Pointer to the object that contains the interface
 * \param Iface Pointer to the interface
 * \result The name of the interface as C-string
 */
TEMU_API const char *temu_nameForInterface(const temu_Object_ *Obj,
                                           const void *Iface);

/*!
 * Get index for interface
 *
 * \param Obj Pointer to the object that contains the interface
 * \param Iface Pointer to the interface
 * \result Index of interface if it is in an iface array.
 *         Otherwise it returns -1. Consequently the function only have valid
 *         result for interface arrays.
 */
TEMU_API int temu_indexForInterface(const temu_Object_ *Obj, const void *Iface);

/*!
 * Get type name for interface
 *
 * \param Obj Pointer to the object that contains the interface
 * \param Iface Pointer to the interface
 * \result The name of the type as C-string
 */
TEMU_API const char *temu_typenameForInterface(const temu_Object_ *Obj,
                                               const void *Iface);

/*!
 * Load a plugin. When a plugin is loaded the temu_pluginInit function
 * will be called. This function should create and define any classes
 * that the plugin provides.
 *
 * The function loads plugins following the operating system's loading
 * rules. This means in particular that first, the plugin will be
 * found in the RPATH location of libTEMUSupport (which refers to the
 * TEMU lib directory). Secondly, it will look in the LD_LIBRARY_PATH
 * and other standard load directories.
 *
 * In practice, it will look for lib<Plugin>.so in the TEMU plugin
 * paths, or libTEMU<Plugin>.so in the same paths. It will then
 * fallback on attempting to load libTEMU<Plugin>.so from the system
 * load paths. If the plugin name contains a slash '/' it will be
 * treated as a path and not a plugin name.
 *
 * \param PluginName A path or plugin name
 * \result 0 on success, non-zero otherwise.
 */
TEMU_API int temu_loadPlugin(const char *PluginName);

/*!
 * Load a plugin. When a plugin is loaded the temu_pluginInit function
 * will be called. This function should create and define any classes
 * that the plugin provides.
 * 
 * The plugin will be made available globally for the linker
 * (e.g. using RTLD_GLOBAL).
 * 
 * This is not recommended for normal TEMU plugins.
 *
 * The function loads plugins following the operating system's loading
 * rules. This means in particular that first, the plugin will be
 * found in the RPATH location of libTEMUSupport (which refers to the
 * TEMU lib directory). Secondly, it will look in the LD_LIBRARY_PATH
 * and other standard load directories.
 *
 * In practice, it will look for lib<Plugin>.so in the TEMU plugin
 * paths, or libTEMU<Plugin>.so in the same paths. It will then
 * fallback on attempting to load libTEMU<Plugin>.so from the system
 * load paths. If the plugin name contains a slash '/' it will be
 * treated as a path and not a plugin name.
 *
 * \param PluginName A path or plugin name
 * \result 0 on success, non-zero otherwise.
 */
TEMU_API int temu_loadPluginGlobal(const char *PluginName);

/*!
 * temu_pluginPathAppend Add a path to the list of paths, where T-emu searches
 * for plugins
 *
 * \param Path The path to append
 */
TEMU_API void temu_pluginPathAppend(const char *Path);

/*!
 * temu_pluginPathRemove Remove a path from the list of paths, where T-emu
 * searches for plugins
 *
 * \param Path to remove
 */
TEMU_API void temu_pluginPathRemove(const char *Path);

/*!
 * temu_pluginPathPrint Print the list of paths, where T-emu searches for
 * plugins
 */
TEMU_API void temu_pluginPathPrint(void);

/*!
 * Get a string representing the type tag.
 *
 * \param Typ The type, whose name is required
 * \result The name as a C-string
 */
TEMU_API const char *temu_typeToName(temu_Type Typ);

/* NOTE: The getProcessors, getProcsessorCount, getComponents and
   getComponentCount functions are experimental and unstable. Do not
   rely on these for the moment.
*/
/*!
 * temu_getProcessors Get the list of processors in the current simulation
 *
 * \result An array of processor pointers, the length can be obtained from
 * temu_getProcessorCount()
 */
TEMU_API temu_Object_ **temu_getProcessors(void);

/*!
 * temu_getProcessorCount Get the number of processors in the current simulation
 *
 * \result The number of processors in the current simulation
 */
TEMU_API size_t temu_getProcessorCount(void);

/*!
 * temu_getComponents Get the number of components in the current simulation
 *
 * \result An array of the pointers to the components in the current simulation,
 *  the length of the array can be obtained from temu_getComponentCount()
 */
TEMU_API temu_Component **temu_getComponents(void);

/*!
 * temu_getComponentCount Get the number of components in the current simulation
 *
 * \result The number of components in the current simulation
 */
TEMU_API size_t temu_getComponentCount(void);

/*!

  Connect an interface property in object A to the interface in object
  B. If A.PropName is an interface array, B,B:IfaceName will be
  appended to it. For dynamic arrays, it is pushed to the end, for
  static arrays, it is placed in the first null slot (unless, the
  indexing syntax is used in the property name).

  \param A The object to connect.
  \param PropName The name of the property in object A
  \param B The object to connect to.
  \param IfaceName The name of the interface in object B
  \result 0 on success, otherwise non-zero
 */
TEMU_API int temu_connect(temu_Object_ *A, const char *PropName,
                          temu_Object_ *B, const char *IfaceName);

/*!
  Serialise the simulation state to the given file name.

  The serialisation writes a JSON formatted file and calls the
  serialise procedure in the (optional) ObjectIface if it is
  implemented.

  The serialisation interface can for example be used to write out
  memory buffers to separate files as such data should not be written
  to a JSON file.

  \param FileName The filename, to which the JSON data is to be stored
  \result 0 on success, otherwise non-zero
 */
TEMU_API int temu_serialiseJSON(const char *FileName);

/*!
  De-serialise the simulation state.

  The function clears the whole internal object system, reads the JSON
  file FileName and creates all the objects.

  If a class implements the ObjectIface, the (optional) deserialise
  procedure will be called.

  \param FileName The filename that contains the JSON data
  \result 0 on success, otherwise non-zero
 */
TEMU_API int temu_deserialiseJSON(const char *FileName);

/*!
  De-serialise the simulation state.

  This function directly restores properties on already created objects.
  It does not clear the objects system.
  Thus pointers to objects remains stable after a snapshot restore.

  If a class implements the ObjectIface, the (optional) deserialise
  procedure will be called.

  \param FileName The filename that contains the JSON data
  \result 0 on success, otherwise non-zero
 */
TEMU_API int temu_inlineDeserialiseJSON(const char *FileName);

/*!
 * Serialise an explicit property, this can be called from the
 * serialisiation interface. Note that the availability of pseudo
 * properties makes this function and the serialise interface more or
 * less obsolete.
 *
 * \param Ctxt The context object that has the property (same as
 *             passed to serialise function in the interface)
 * \param Name The name of the property
 * \param Typ The type of the property
 * \param Count 0 if the property is not an array, otherwise the
 *              element number in the property
 * \param Data Serialized data
 */
TEMU_API void temu_serialiseProp(void *Ctxt, const char *Name, temu_Type Typ,
                                 int Count, void *Data);

/*!
 * temu_deserialiseProp Deserialize a serialized property
 *
 * \param Ctxt An opaque context, this is the same context as is
 *             passed to the deserialise interface function.
 *
 * \param Obj Object being deserialised
 * \param Name The name of the property
 */
TEMU_API void temu_deserialiseProp(void *Ctxt, temu_Object_ *Obj,
                                   const char *Name);

/*!
 * Get number of entries for the serialized property
 *
 * \param Ctxt Context passed to deserialise function
 * \param Name Name of property / key in the serialized file
 * \result Length of the property in elements. Negative on error.
 */
TEMU_API int temu_snapshotGetLength(void *Ctxt, const char *Name);
/*!
 * Get number of entries for the serialized property
 *
 * \param Ctxt Context passed to deserialise function
 * \param Name Name of property / key in the serialized file
 * \result Length of the property in elements. Negative on error.
 */
TEMU_API int temu_checkpointGetLength(void *Ctxt, const char *Name);

/*!
 * Get value for named snapshot value
 *
 * \param Ctxt Context is passed to deserialise interface function
 * \param Name Name of the saved value
 * \param Idx Index of the saved value (if array)
 * \result Property value with the contents saved to the snapshot
 */
TEMU_API temu_Propval temu_snapshotGetValue(void *Ctxt, const char *Name,
                                            int Idx);

/*!
 * Get value for named snapshot value
 *
 * \param Ctxt Context is passed to deserialise interface function
 * \param Name Name of the saved value
 * \param Idx Index of the saved value (if array)
 * \result Property value with the contents saved to the snapshot
 */
TEMU_API temu_Propval temu_checkpointGetValue(void *Ctxt, const char *Name,
                                              int Idx);

/*! Check object system sanity.
 *
 * The function traverses all objects and their properties and ensures that
 * all (scalar) interface properties are connected.
 * An object can override the default check by implementing the checkSanity
 * function in the ObjectIface.
 *
 * \param Report set to 1 to have the function report connectivity issues on
 *        stdout, 2 to report on stderr, and 0 to not report at all.
 * \result Returns 0 if the object system is connected.
 *       Returns non-zero if the object system is not properly connected.
 */
TEMU_API int temu_checkSanity(int Report);

//! Print the current object graph as a dot file
//!
//! Pass NULL (or nullptr) to generate to stdout. And set Display to
//! non-zero to automatically display the generated dot-file.
//!
//! \param Path Destination file to write the graph to. NULL indicates
//!             stdout.
//!
//! \param Display If non-zero, the function will attempt to display
//!                the graph by running it through dot and .
//!
//! \result 0 if the generation was successful. Non-zero in case of
//!         failure.
TEMU_API int temu_generateObjectGraph(const char *Path, int Display);

TEMU_API int temu_isValidObjectName(const char *Name);
TEMU_API int temu_isValidClassName(const char *Name);
TEMU_API int temu_isValidInterfaceName(const char *Name);
TEMU_API int temu_isValidPropertyName(const char *Name);

/*!
 * Generic object interface The object interface provides generic
 * functionality such as serialisation and sanity checking support.
 */
typedef struct {
  /*! Optional function
   * Called after an object has been written to the snapshot, this
   * function can write out additional properties to the snapshot
   * and take other actions.
   *
   * Note that with the pseudoproperties supporting snapshotting,
   * this function is very rarely needed.
   */
  void (*serialise)(void *Obj, const char *BaseName, void *Ctxt);

  /*! Optional function
   * Called after an object has been restored from a snapshot, this
   * function can read additional properties to the snapshot
   * and take other actions.
   *
   * Note that with the pseudoproperties supporting snapshotting,
   * this function is very rarely needed.
   */

  void (*deserialise)(void *Obj, const char *BaseName, void *Ctxt);

  /*!
   * Return zero if the object is connected as expected, return
   * non-zero if the object is not fully connected the default check
   * will ensure that all the Interface references are connected, but
   * does not care about optional interfaces or
   */
  int (*checkSanity)(void *Obj, int Report); // Optional

  /*!
   * Optional function
   *
   * Called when the time source has been set on the object
   * The function can for example post initial events.
   */
  void (*timeSourceSet)(void *Obj);

  /*!
   * Optional function
   *
   * Pretty prints the object to stdout, called by the object-print
   * command.
   */
  void (*printObject)(void *Obj);
} temu_ObjectIface;
#define TEMU_OBJECT_IFACE_TYPE "ObjectIface"
TEMU_IFACE_REFERENCE_TYPE(temu_Object)

/*!
 * Call function on every object in the object system
 * The function is called with the object pointer and the argument.
 * NOTE: This function is experimental.
 *
 * \param Func Function to call
 * \param Arg Argument passed as the last parameter to Func
 */
TEMU_API void temu_foreachObject(void (*Func)(temu_Object_ *, void *),
                                 void *Arg);

/*!
 * Call function on every class in the object system
 * The function is called with the metaclass pointer and the argument.
 * NOTE: This function is experimental.
 *
 * \param Func Function to call
 * \param Arg Argument passed as the last parameter to Func
 */
TEMU_API void temu_foreachClass(void (*Func)(temu_Class *, void *), void *Arg);

/*!
 * Call function on every processor in the object system
 * The function is called with the CPU pointer and the argument.
 * NOTE: This function is experimental.
 *
 * \param Func Function to call
 * \param Arg Argument passed as the last parameter to Func
 */
TEMU_API void temu_foreachProcessor(void (*Func)(temu_Object_ *, void *),
                                    void *Arg);

/*!
 * Call function on every property in a class.
 * The function is called with the class pointer, property name  and the
 * argument. NOTE: This function is experimental.
 *
 * \param C Class to iterate properties
 * \param Func Function to call
 * \param Arg Argument passed as the last parameter to Func
 */

TEMU_API void temu_foreachProperty(
    temu_Class *C, void (*Func)(temu_Class *, const char *, void *), void *Arg);

/*!
 * Call function on every interface in a class.
 * The function is called with the class pointer, interface name  and the
 * argument. NOTE: This function is experimental.
 *
 * \param C Class to iterate properties
 * \param Func Function to call
 * \param Arg Argument passed as the last parameter to Func
 */

TEMU_API void temu_foreachInterface(
    temu_Class *C, void (*Func)(temu_Class *, const char *, void *), void *Arg);

TEMU_API void *temu_registerInterfaceType(const char *Name);
TEMU_API void *temu_getInterfaceType(const char *Name);

//! Add a scalar (one element) property without setters / getters
//! \param Cls Class pointer
//! \param Name Name of property
//! \param offset Offset from object pointer
//! \param T Property type
//! \param Doc Documentation string
//! \result 0 if the property was added
TEMU_API int temu_addScalarProperty(temu_Class *Cls, const char *Name,
                                    int offset, temu_Type T, const char *Doc);
//! Add a fixed length array property without setters / getters
//! \param Cls Class pointer
//! \param Name Name of property
//! \param offset Offset from object pointer
//! \param T Property type
//! \param NumElems Array length
//! \param Doc Documentation string
//! \result 0 if the property was added
TEMU_API int temu_addArrayProperty(temu_Class *Cls, const char *Name,
                                   int offset, temu_Type T, int NumElems,
                                   const char *Doc);

//! Add a scalar (one element) pseudo-property
//! \param Cls Class pointer
//! \param Name Name of property
//! \param T Property type
//! \param Doc Documentation string
//! \result 0 if the property was added
TEMU_API int temu_addScalarPseudoProperty(temu_Class *Cls, const char *Name,
                                          temu_Type T, const char *Doc);

//! Add an array pseudo-property
//! \param Cls Class pointer
//! \param Name Name of property
//! \param T Property type
//! \param NumElems Array length
//! \param Doc Documentation string
//! \result 0 if the property was added
TEMU_API int temu_addArrayPseudoProperty(temu_Class *Cls, const char *Name,
                                         temu_Type T, int NumElems,
                                         const char *Doc);

//! Write property (with side effects)
//! \param Obj Object pointer
//! \param Name Name of property
//! \param idx Index in property array (set to 0 if scalar)
//! \param PV Pointer to propval
//! \result 0 if the property was written
TEMU_API int temu_writeProp(temu_Object_ *Obj, const char *Name, int idx,
                            temu_Propval *PV);

//! Read property (with side effects)
//! \param Obj Object pointer
//! \param Name Name of property
//! \param idx Index in property array (set to 0 if scalar)
//! \result The read value, set to teTY_Invalid for failures
TEMU_API temu_Propval temu_readProp(temu_Object_ *Obj, const char *Name,
                                    int idx);

//! Get property (without side effects)
//! \param Obj Object pointer
//! \param PropName Name of property
//! \param Idx Index in property array (set to 0 if scalar)
//! \result The property value converted to unsigned
TEMU_API uint64_t temu_getValueUnsigned(temu_Object_ *Obj, const char *PropName,
                                        int Idx);
//! Get property (without side effects)
//! \param Obj Object pointer
//! \param PropName Name of property
//! \param Idx Index in property array (set to 0 if scalar)
//! \result The property value converted to signed
TEMU_API int64_t temu_getValueSigned(temu_Object_ *Obj, const char *PropName,
                                     int Idx);

//! Get property (without side effects)
//! \param Obj Object pointer
//! \param PropName Name of property
//! \param Idx Index in property array (set to 0 if scalar)
//! \result The property value converted to double
TEMU_API double temu_getValueDouble(temu_Object_ *Obj, const char *PropName,
                                    int Idx);
//! Read property (with side effects)
//! \param Obj Object pointer
//! \param PropName Name of property
//! \param Idx Index in property array (set to 0 if scalar)
//! \result The property value converted to unsigned
TEMU_API uint64_t temu_readValueUnsigned(temu_Object_ *Obj,
                                         const char *PropName, int Idx);
//! Read property (with side effects)
//! \param Obj Object pointer
//! \param PropName Name of property
//! \param Idx Index in property array (set to 0 if scalar)
//! \result The property value converted to signed
TEMU_API int64_t temu_readValueSigned(temu_Object_ *Obj, const char *PropName,
                                      int Idx);
//! Read property (with side effects)
//! \param Obj Object pointer
//! \param PropName Name of property
//! \param Idx Index in property array (set to 0 if scalar)
//! \result The property value converted to double
TEMU_API double temu_readValueDouble(temu_Object_ *Obj, const char *PropName,
                                     int Idx);

//! Set unsigned property (without side effects)
//! \param Obj Object pointer
//! \param PropName Name of property
//! \param Val The new property value
//! \param Idx Index in property array (set to 0 if scalar)
TEMU_API void temu_setValueUnsigned(temu_Object_ *Obj, const char *PropName,
                                    uint64_t Val, int Idx);
//! Set signed property (without side effects)
//! \param Obj Object pointer
//! \param PropName Name of property
//! \param Val The new property value
//! \param Idx Index in property array (set to 0 if scalar)
TEMU_API void temu_setValueSigned(temu_Object_ *Obj, const char *PropName,
                                  int64_t Val, int Idx);
//! Set floating point property (without side effects)
//! \param Obj Object pointer
//! \param PropName Name of property
//! \param Val The new property value
//! \param Idx Index in property array (set to 0 if scalar)
TEMU_API void temu_setValueDouble(temu_Object_ *Obj, const char *PropName,
                                  double Val, int Idx);
//! Write unsigned property (with side effects)
//! \param Obj Object pointer
//! \param PropName Name of property
//! \param Val The new property value
//! \param Idx Index in property array (set to 0 if scalar)
TEMU_API void temu_writeValueUnsigned(temu_Object_ *Obj, const char *PropName,
                                      uint64_t Val, int Idx);
//! Write signed property (with side effects)
//! \param Obj Object pointer
//! \param PropName Name of property
//! \param Val The new property value
//! \param Idx Index in property array (set to 0 if scalar)
TEMU_API void temu_writeValueSigned(temu_Object_ *Obj, const char *PropName,
                                    int64_t Val, int Idx);
//! Write floating point property (with side effects)
//! \param Obj Object pointer
//! \param PropName Name of property
//! \param Val The new property value
//! \param Idx Index in property array (set to 0 if scalar)
TEMU_API void temu_writeValueDouble(temu_Object_ *Obj, const char *PropName,
                                    double Val, int Idx);

/*!
 * Create a signed propval of the given type
 * \param T Type ID, must be a signed integer type tag
 * \param I Integer value
 * \result Property value with the given type and value.
 */
TEMU_API temu_Propval temu_signedPropval(temu_Type T, int64_t I);
/*!
 * Create an unsigned propval of the given type
 * \param T Type ID, must be an unsigned integer type tag
 * \param U Integer value
 * \result Property value with the given type and value.
 */
TEMU_API temu_Propval temu_unsignedPropval(temu_Type T, uint64_t U);
/*!
 * Create a floating point propval of the given type
 * \param T Type ID, must be a teTY_Float or teTY_Double
 * \param V Value
 * \result Property value with the given type and value.
 */
TEMU_API temu_Propval temu_floatingPointPropval(temu_Type T, double V);

/*!
 * Check if object has a command
 * \param Obj Object pointer
 * \param CmdName Name of command
 * \result 0 in case command does not exist. Otherwise 1.
 */
TEMU_API int temu_objectHasCmd(const temu_Object_ *Obj, const char *CmdName);

#ifdef __cplusplus
}
#endif

#endif /*!  TEMU_OBJSYS_C_H */
