//===-- temu-c/Target/ARM.h - ARM Architecture Interfaces -------*- C++ -*-===//
//
// TEMU: The Terma Emulator
// (c) Terma 2016
// Authors: Mattias Holm <maho (at) terma.com>
//
//===----------------------------------------------------------------------===//


#ifndef TEMU_TARGET_ARM_H
#define TEMU_TARGET_ARM_H
#include <stdint.h>
#include "temu-c/Support/Objsys.h"

typedef enum {
  teARM_Usr, //!< User        0b10000 0x10 16
  teARM_Fiq, //!< Fast irq    0b10001 0x11 17
  teARM_Irq, //!< Interrupt   0b10010 0x12 18
  teARM_Svc, //!< Supervisor  0b10011 0x13 19
  teARM_Mon, //!< Monitor     0b10110 0x14 20
  teARM_Abt, //!< Abort       0b10111 0x15 21
  teARM_Hyp, //!< Hypervisor  0b11010 0x1a 26
  teARM_Und, //!< Undefined   0b11011 0b1b 27
  teARM_Sys, //!< System      0b11111 0b1f 31
} temu_ARMMode;

typedef enum {
  teARM_ARM, //!< Processor is in ARM mode
  teARM_Thumb, //!< Processor is in Thumb mode
  teARM_Jazelle, //!< Processor is in Jazelle mode
  teARM_ThumbEE, //!< Processor is in Thumb EE mode
} temu_ARMExecMode;


typedef struct {
  uint32_t (*getBankedRegister)(void *Obj, temu_ARMMode Mode, unsigned Reg);
  temu_ARMMode (*getMode)(void *Obj);
  temu_ARMExecMode (*getExecMode)(void *Obj);
  void (*setExecMode)(void *Obj, temu_ARMExecMode Mode);

  uint32_t (*getAPSR)(void *Obj);
  void (*setAPSR)(void *Obj, uint32_t Value);

  uint32_t (*getFpscr)(const void *Cpu);
  void (*setFpscr)(void *Cpu, uint32_t Value);

  uint32_t (*getFpexc)(const void *Cpu);
  void (*setFpexc)(void *Cpu, uint32_t Value);

  uint32_t (*getFpinst)(const void *Cpu, int idx);
  void (*setFpinst)(void *Cpu, int idx, uint32_t Value);
} temu_ARMCpuIface;
#define TEMU_ARM_CPU_IFACE_TYPE "ARMCpu"
TEMU_IFACE_REFERENCE_TYPE(temu_ARMCpu);

typedef struct {
  // Return 1 if cp operation is accepted, 0 if not accepted
  uint8_t (*accepts)(void *Obj, uint32_t cp, uint32_t instr);

  uint32_t (*getOneWord)(void *Obj, uint32_t cp, uint32_t instr);
  // Return t in lower half, t2 in high half
  uint64_t (*getTwoWords)(void *Obj, uint32_t cp, uint32_t instr);
  void (*sendOneWord)(void *Obj, uint32_t t, uint32_t cp, uint32_t instr);
  void (*sendTwoWords)(void *Obj, uint32_t t2, uint32_t t, uint32_t cp, uint32_t instr);
} temu_ARMCoProcessorIface;

#define TEMU_ARM_COPROC_IFACE_TYPE "ARMCoProcessor"
TEMU_IFACE_REFERENCE_TYPE(temu_ARMCoProcessor);

#endif /* ! TEMU_TARGET_ARM_H */
