/*
 * A simple rtems program to show how to use configuration templates
 */

#include <rtems.h>
/* configuration information */

#define CONFIGURE_INIT

#include <bsp.h> /* for device driver prototypes */

rtems_task Init( rtems_task_argument argument); /* forward declaration needed */

/* configuration information */

#define CONFIGURE_APPLICATION_NEEDS_CONSOLE_DRIVER
#define CONFIGURE_APPLICATION_NEEDS_CLOCK_DRIVER

#define CONFIGURE_MAXIMUM_TASKS             4

#define CONFIGURE_RTEMS_INIT_TASKS_TABLE
#define CONFIGURE_LIBIO_MAXIMUM_FILE_DESCRIPTORS 32

#define CONFIGURE_EXTRA_TASK_STACKS         (3 * RTEMS_MINIMUM_STACK_SIZE)


#include <rtems/confdefs.h>

/* If --drvmgr was enabled during the configuration of the RTEMS kernel */
#ifdef RTEMS_DRVMGR_STARTUP
 #ifdef LEON3
  /* Add Timer and UART Driver for this example */
  #ifdef CONFIGURE_APPLICATION_NEEDS_CLOCK_DRIVER
   #define CONFIGURE_DRIVER_AMBAPP_GAISLER_GPTIMER
  #endif
  #ifdef CONFIGURE_APPLICATION_NEEDS_CONSOLE_DRIVER
   #define CONFIGURE_DRIVER_AMBAPP_GAISLER_APBUART
  #endif

  /* OPTIONAL FOR GRLIB SYSTEMS WITH APBUART AS SYSTEM CONSOLE.
   *
   * Assign a specific UART the system console (syscon=1) and debug output
   * debug_uart_index. Note that the default is to have APBUART0 as
   * system/debug console, one must override it by setting syscon=0 on the
   * first APBUART, then setting syscon=1 on APBUART that should be system
   * console.
   *
   * Note also that the debug console does not have to be on the same UART
   * as the system console.
   *
   * Determine if console driver should do be interrupt driven (mode=1)
   * or polling (mode=0).
   *
   * Example below:
   *   APBUART[0] in Interrupt mode
   *   APBUART[1] in Interrupt mode, System console and Debug console
   *   APBUART[N>=2] in Polling mode (not configured)
   */
  #if 0
   #include <drvmgr/ambapp_bus.h>
   /* APBUART0 */
   struct drvmgr_key grlib_drv_res_apbuart0[] =
   {
        {"mode", KEY_TYPE_INT, {(unsigned int)1}},
        {"syscon", KEY_TYPE_INT, {(unsigned int)0}},
        KEY_EMPTY
   };
   /* APBUART1 */
   struct drvmgr_key grlib_drv_res_apbuart1[] =
   {
        {"mode", KEY_TYPE_INT, {(unsigned int)1}},
        {"syscon", KEY_TYPE_INT, {(unsigned int)1}},
        KEY_EMPTY
   };
   /* LEON3 System with driver configuration for 2 APBUARTs, the
    * the rest of the AMBA device drivers use their defaults.
    */
   struct drvmgr_bus_res grlib_drv_resources =
   {
        .next = NULL,
        .resource = {
        {DRIVER_AMBAPP_GAISLER_APBUART_ID, 0, &grlib_drv_res_apbuart0[0]},
        {DRIVER_AMBAPP_GAISLER_APBUART_ID, 1, &grlib_drv_res_apbuart1[0]},
        RES_EMPTY
        }
   };

   /* Override defualt debug UART assignment.
    * 0 = Default APBUART. APBUART[0], but on MP system CPU0=APBUART0,
    *     CPU1=APBUART1...
    * 1 = APBUART[0]
    * 2 = APBUART[1]
    * 3 = APBUART[2]
    * ...
    */
   int debug_uart_index = 2; /* second UART -- APBUART[1] */
  #endif
 #endif

 #include <drvmgr/drvmgr_confdefs.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>

typedef struct {
  uint32_t Ctrl;
  uint32_t RxMaxLen;
  uint32_t TxDesc;
  uint32_t RxDesc;
  uint32_t Addr;
  uint32_t Padding[3];
} grspw2_dma_t;

typedef struct {
  uint32_t Ctrl;
  uint32_t Status;
  uint32_t DefAddr;
  uint32_t ClockDiv;
  uint32_t DestKey;
  uint32_t TimeCode;
  uint32_t Padding0[2];
  grspw2_dma_t Dma[4];
} grspw2_dev_t;

typedef struct {
  uint32_t PacketInfo;
  uint32_t PacketAddress;
} grspw2_rxdma_desc_t;

typedef struct {
  uint32_t PacketInfo;
  uint32_t HeaderAddr;
  uint32_t DataLen;
  uint32_t DataAddr;
} grspw2_txdma_desc_t;

// Memory allocation for tx and rx descriptors.
static grspw2_rxdma_desc_t rx_descs[128] __attribute__ ((aligned (1024)));
static grspw2_txdma_desc_t tx_descs[64] __attribute__ ((aligned (1024)));

#define GRSPW2_CTRL_LD (1 << 0)
#define GRSPW2_CTRL_LS (1 << 1)
#define GRSPW2_CTRL_AS (1 << 2)

#define GRSPW2_TXDESC_EN (1 << 12)

#define GRSPW2_DMACTRL_TE (1 << 0)

const char*
LinkStateName(uint32_t link_state)
{
  const char* linkStateNames[] = {
    "ErrorReset",
    "ErrorWait",
    "Ready",
    "Started",
    "Connecting",
    "Run"};
  if (link_state < sizeof(linkStateNames)/sizeof(linkStateNames[0])) {
    return linkStateNames[link_state];
  }
  return "Unknown";
}

uint32_t LinkState(volatile grspw2_dev_t *grspw2dev)
{
  return (grspw2dev->Status >> 21) & 0x7;
}

void
PrintLinkState(volatile grspw2_dev_t *grspw2dev)
{
  printf("OBSW: Link state: %s\n",
         LinkStateName((grspw2dev->Status >> 21) & 0x7));
}

void
AssertState(volatile grspw2_dev_t *grspw2_1, uint32_t state)
{
  if (LinkState(grspw2_1) != state) {
    printf("[ERROR] Expected state: %s, actual: %s\n",
           LinkStateName(LinkState(grspw2_1)), LinkStateName(state));
  }
}

void
TestStates(volatile grspw2_dev_t *grspw2_1)
{
  printf("Set link disable, auto start and link start => Rmain Ready.\n");
  AssertState(grspw2_1, 2);
  grspw2_1->Ctrl |= GRSPW2_CTRL_LD;
  grspw2_1->Ctrl |= (GRSPW2_CTRL_AS | GRSPW2_CTRL_LS);
  AssertState(grspw2_1, 2);

  printf("Unset link disable (auto start and link start = 1) => Go run.\n");
  grspw2_1->Ctrl &= ~(GRSPW2_CTRL_LD);
  AssertState(grspw2_1, 5);

  printf("Set link disable => Go ready.\n");
  //grspw2_1->Ctrl &= ~GRSPW2_CTRL_AS;
  grspw2_1->Ctrl |= GRSPW2_CTRL_LD;
  AssertState(grspw2_1, 2);

  printf("Set link start => Stay Ready\n");
  grspw2_1->Ctrl |= GRSPW2_CTRL_LS;
  AssertState(grspw2_1, 2);

  printf("Unset link disable => Go Run\n");
  grspw2_1->Ctrl &= ~(GRSPW2_CTRL_LD);
  AssertState(grspw2_1, 5);
}

rtems_task
Init(rtems_task_argument ignored)
{
  // Devices address mapping of the ut700
  volatile grspw2_dev_t *grspw2_1 = (grspw2_dev_t*)0x80000a00;
  volatile grspw2_dev_t *grspw2_2 = (grspw2_dev_t*)0x80000b00;
  volatile grspw2_dev_t *grspw2_3 = (grspw2_dev_t*)0x80000c00;
  volatile grspw2_dev_t *grspw2_4 = (grspw2_dev_t*)0x80000d00;

  printf("OBSW: Link enable/disable\n");

  TestStates(grspw2_1);

  // Check the link state.
  PrintLinkState(grspw2_1);

  // Start the link.
  printf("=================================================================== \n");
  printf("OBSW: Starting link\n");
  grspw2_1->Ctrl &= ~(GRSPW2_CTRL_LD);
  grspw2_1->Ctrl |= GRSPW2_CTRL_LS | GRSPW2_CTRL_AS;
  AssertState(grspw2_1, 5);

  printf("OBSW: Sending a SpaceWire message through Grspw2 (1)\n");

  // Set up descriptors at
  grspw2_1->Dma[0].TxDesc = (uint32_t)tx_descs;
  grspw2_1->Dma[0].RxDesc = (uint32_t)rx_descs;
  printf("OBSW: Tx descriptors at: 0x%x\n", (uint32_t)tx_descs);
  printf("OBSW: Rx descriptors at: 0x%x\n", (uint32_t)rx_descs);

  // Run the spacewire.
  const char message[] = "hello world";
  tx_descs[0].PacketInfo = GRSPW2_TXDESC_EN;
  tx_descs[0].HeaderAddr = 0;
  tx_descs[0].DataLen = sizeof(message)/sizeof(message[0]);
  tx_descs[0].DataAddr = (uint32_t)message;

  printf("OBSW: Tx descriptors at: 0x%x\n", grspw2_1->Dma[0].TxDesc);
  printf("OBSW: Rx descriptors at: 0x%x\n", grspw2_1->Dma[0].RxDesc);

  printf("OBSW: Descriptor.Ctrl:  %d\n", tx_descs[0].PacketInfo);
  printf("OBSW: Descriptor.HeaderAddr:  %d\n", tx_descs[0].HeaderAddr);
  printf("OBSW: Descriptor.DataLen:  %d\n", tx_descs[0].DataLen);
  printf("OBSW: Descriptor.DataAddr: 0x%x\n", tx_descs[0].DataAddr);

  // Enable transmission.
  grspw2_1->Dma[0].Ctrl = GRSPW2_DMACTRL_TE;

  sleep(1);
  printf("OBSW: Send complete\n");

  exit( 0 );
}
