//===------------------------------------------------------------*- C++ -*-===//
//
// TEMU: The Terma Emulator
// (c) Terma 2020
// Authors: Mattias Holm <maho (at) terma.com>
//
//===----------------------------------------------------------------------===//

#ifndef TEMU_ETHERNET_H
#define TEMU_ETHERNET_H

#include "temu-c/Support/Bitmanip.h"
#include "temu-c/Support/Buffer.h"
#include "temu-c/Support/Objsys.h"

#include <stdint.h>
#include <string.h>

#ifdef __cplusplus
extern "C" {
#endif

/*
  The ethernet device interface is EXPERIMENTAL and UNSTABLE at the
  moment.

  MAC addresses are 6 bytes, as convention here, we encode these in a
  uint64_t field, where the two MSBs are set to 0. The, MAC address
  will be in host endianess.

  Ethernet simulation is based on the transmission of frames. Frames
  do not need to have valid CRCs, but the CRC data bytes should be
  allocated for the frames. This speeds up simulation significantly.

  CRC errors can instead be efficiently injected by flipping the
  relevant bit in the flags field.

  Ethernet is not a point to point protocol (although in practice,
  these days it is P2P thanks to switches which are collission free),
  but we could immagine some one simulating an ethernet link via coax
  in which case there may be multiple nodes connected to the same
  cable. It is possible to tap the ethernet object by adding an
  ethernet device to it that listens to all addresses (i.e. it
  notifies the bus model about being "promiscuous").

  ETH support is split using multiple levels, i.e. a MAC and a PHY
  controller. The PHY controller implements the PHY interface and
  usually also the MII interface (there is a generic MII device that
  can be used which implements both PHY and MII). PHY is used for
  connecting to other PHY controllers, when sending a frame, the MAC
  will (using e.g. DMA) get the frame data, assemble a temu_EthFrame
  and send this to the PHY controller, the PHY controller in turn
  forwards the frame to the destination PHY. In addition, there is an
  MIIBus model which lets a single master control multiple PHY
  devices. The MIIBus model implements the MII interface only and
  simply routes the calls to the correct attached PHY device.

  So, the use is typically:

    MAC -> PHY -> ETH -> PHY -> MAC
 */

// Flag bits
#define TEMU_ETH_CRC_ERR 1

#define TEMU_ETH_ETH_CRC_NOT_SET (1 << 1)
#define TEMU_ETH_IP_CRC_NOT_SET (1 << 2)
#define TEMU_ETH_UDP_CRC_NOT_SET (1 << 3)
#define TEMU_ETH_TCP_CRC_NOT_SET (1 << 4)
#define TEMU_ETH_NON_STANDARD_PREAMBLE (1 << 5)
#define TEMU_ETH_PREAMBLE_LENGTH_MASK (0xf << 6)
#define TEMU_ETH_PREAMBLE_LENGTH_SHIFT 6

typedef struct {
  uint32_t Flags;       //!< Flags used for error injection
  temu_Buff Data;       //!< ETH frame data
  uint8_t Preamble[15]; //!< Preamble bits, normally 0x[aa aa aa aa aa aa aa]
  uint8_t Sfd;          //!< Start frame delimiter, normally 0xab
} temu_EthFrame;

typedef struct {
  uint8_t MAC[6];
} temu_EthMAC;

static inline temu_EthMAC
temu_ethMacToStruct(uint64_t mac)
{
  temu_EthMAC result;
  result.MAC[0] = mac >> 40;
  result.MAC[1] = mac >> 32;
  result.MAC[2] = mac >> 24;
  result.MAC[3] = mac >> 16;
  result.MAC[4] = mac >> 8;
  result.MAC[5] = mac >> 0;
  return result;
}
static inline uint64_t
temu_ethMacFromStruct(temu_EthMAC mac)
{
  uint64_t result = 0;
  result |= ((uint64_t)mac.MAC[0]) << 40;
  result |= ((uint64_t)mac.MAC[1]) << 32;
  result |= ((uint64_t)mac.MAC[2]) << 24;
  result |= ((uint64_t)mac.MAC[3]) << 16;
  result |= ((uint64_t)mac.MAC[4]) << 8;
  result |= ((uint64_t)mac.MAC[5]) << 0;
  return result;
}
static inline void
temu_ethSetDestMAC(temu_EthFrame *Frame, uint64_t MAC)
{
  temu_EthMAC macAddr = temu_ethMacToStruct(MAC);
  uint8_t *Data = temu_buffWritableData(&Frame->Data);
  memcpy(&Data[0], &macAddr.MAC[0], 6);
}

static inline void
temu_ethSetSourceMAC(temu_EthFrame *Frame, uint64_t MAC)
{
  temu_EthMAC macAddr = temu_ethMacToStruct(MAC);
  uint8_t *Data = temu_buffWritableData(&Frame->Data);
  memcpy(&Data[6], &macAddr.MAC[0], 6);
}

static inline void
temu_ethSetDestAndSourceMAC(temu_EthFrame *Frame, uint64_t DestMAC,
                            uint64_t SrcMAC)
{
  temu_EthMAC destMacAddr = temu_ethMacToStruct(DestMAC);
  temu_EthMAC srcMacAddr = temu_ethMacToStruct(SrcMAC);
  uint8_t *Data = temu_buffWritableData(&Frame->Data);
  memcpy(&Data[0], &destMacAddr.MAC[0], 6);
  memcpy(&Data[6], &srcMacAddr.MAC[0], 6);
}

typedef struct temu_PHYIface temu_PHYIface;
typedef struct temu_MACIface temu_MACIface;
typedef struct temu_EthernetIface temu_EthernetIface;
TEMU_IFACE_REFERENCE_TYPE(temu_MAC);
TEMU_IFACE_REFERENCE_TYPE(temu_PHY);
TEMU_IFACE_REFERENCE_TYPE(temu_Ethernet);

struct temu_MACIface {
  void (*connected)(
      void *Dev); // Called when link is connected (i.e. cable inserted)
  void (*disconnected)(
      void *Dev); // Called when link is disconnected (i.e. cable removed)
  void (*up)(void *Dev);   // Called when link goes up
  void (*down)(void *Dev); // Called when link goes down

  int (*receive)(void *Dev, temu_EthFrame *Frame); // Receive frame
  uint64_t (*getMAC)(void *Dev);                   // Get MAC address
  void (*setMAC)(void *Dev, uint64_t MAC);         // Set MAC address
};
#define TEMU_MAC_IFACE_TYPE "temu::MACIface"

struct temu_PHYIface {
  void (*connected)(
      void *Dev,
      temu_EthernetIfaceRef
          Link); // Called when link is connected (i.e. cable inserted)
  void (*disconnected)(
      void *Dev,
      temu_EthernetIfaceRef
          Link); // Called when link is disconnected (i.e. cable removed), note
                 // that the PHY must remove the MACs from the link
  void (*up)(void *Dev);   // Called when link goes up
  void (*down)(void *Dev); // Called when link goes down

  int (*send)(void *Dev, temu_EthFrame *Frame);    // Send frame from MAC
  int (*receive)(void *Dev, temu_EthFrame *Frame); // Receive frame

  // PHY should clear bits not related to its capabilities in the status and
  // ext status registers
  uint32_t (*autoNegotiate)(void *Obj, uint32_t Caps); // Do auto-negotiatio
  void (*autoNegotiateDone)(void *Obj, uint32_t Caps);
};
#define TEMU_PHY_IFACE_TYPE "temu::PHYIface"

// These bits should be transferred using auto-negotiation requests
// Matches MII status reg (reg 1), but needs to be shifted 8 bits up
#define TEMU_ETH_100BASE_T4 (1 << (15 - 8))
#define TEMU_ETH_100BASE_X_FD (1 << (14 - 8))
#define TEMU_ETH_100BASE_X_HD (1 << (13 - 8))
#define TEMU_ETH_10BASE_FD (1 << (12 - 8))
#define TEMU_ETH_10BASE_HD (1 << (11 - 8))
#define TEMU_ETH_100BASE_T2_FD (1 << (10 - 8))
#define TEMU_ETH_100BASE_T2_HD (1 << (9 - 8))

// Matches MII extended status register (reg 15)
#define TEMU_ETH_1000BASE_X_FD (1 << 15)
#define TEMU_ETH_1000BASE_X_HD (1 << 14)
#define TEMU_ETH_1000BASE_T_FD (1 << 13)
#define TEMU_ETH_1000BASE_T_HD (1 << 12)

// Interface for the actual link, a link can be a switch, hub, or a
// direct connection, note that ETH is not seen as a Point to Point
// connection for the reason of supporting HUBs, splitters etc.
struct temu_EthernetIface {
  void (*connect)(void *Obj,
                  temu_PHYIfaceRef Dev); // Called to connect a device
  void (*disconnect)(void *Obj,
                     temu_PHYIfaceRef Dev); // Called to disconnect link

  // Ensure the device receives messages for this MAC address
  // Note, do not forget to add broadcast and multicast bits
  void (*addMAC)(void *Obj, temu_PHYIfaceRef Dev, uint64_t MAC);
  void (*removeMAC)(void *Obj, temu_PHYIfaceRef Dev, uint64_t MAC);
  void (*setPromiscuous)(void *Obj, temu_PHYIfaceRef Dev, int PromiscuousMode);

  // Do auto-negotiation, this results in queries on all attached devices
  // that determines the shared link speed
  uint32_t (*autoNegotiate)(void *Obj, uint32_t Caps);

  // Send frame to virtual ethernet cable return non zero if frame is
  // accepted, link will route the frame to relevant devices
  int (*send)(void *Obj, temu_EthFrame *Frame);
};
#define TEMU_ETHERNET_IFACE_TYPE "temu::EthernetIface"

// I.e. Eth::connect() -> MAC::connected()
// I.e. Phy::send() -> Eth::receive()

// MDIO interface
// The MDIO interface is for creating devices connected to an MDIO bus
// or to simulate an MDIO, note that we do not actually
// simulate MDIO per see, but do support mulpiple PHY controllers to
// be attached to the same bus.
typedef struct {
  // The readReg returns negative if the call failed, it can fail if
  // the access is to a bus and the PHY_ID is invalid or if the
  // address is invalid. A successful means that the result can be
  // safely be converted to a uint16_t.
  int32_t (*readReg)(void *Obj, unsigned PHY_ID, uint16_t Addr);
  int (*writeReg)(void *Obj, unsigned PHY_ID, uint16_t Addr, uint16_t Value);
} temu_MDIOIface;
#define TEMU_MDIO_IFACE_TYPE "temu::MDIOIface"
TEMU_IFACE_REFERENCE_TYPE(temu_MDIO);

// Interface for adding a delay object in the connection
// between eth link and eth device
struct temu_EthDelayIface {
  void (*addDelay)(temu_Object *link, temu_Object *device, temu_Object *delay,
                   double seconds); // Called to connect a delay object
};
#define TEMU_ETH_DELAY_IFACE_TYPE "temu::EthDelayIface"

#define TEMU_ETH_MIN_PAYLOAD 46
#define TEMU_ETH_MAX_PAYLOAD 1500
#define TEMU_ETH_LAYER1_HEADER_LENGTH (7 + 1 + 12 + 2)
#define TEMU_ETH_LAYER2_HEADER_LENGTH (6 + 6 + 2)
#define TEMU_ETH_CRC_LENGTH 4
#define TEMU_ETH_INTERPACKET_GAP 12
#define TEMU_ETH_PAYLOAD_OFFSET (6 + 6 + 2)
#define TEMU_ETH_MAX_FRAME_LENGTH 1518
#define TEMU_ETH_802_1Q_TAG_BYTES 4
#define TEMU_ETH_802_1Q_MAX_FRAME_LENGTH 1522

// An assortment of ethertype constants, non exhaustive.
// See: https://www.iana.org/assignments/ieee-802-numbers/ieee-802-numbers.xhtml
// for a more complete list.
#define TEMU_ETHTYPE_IPV4 0x0800
#define TEMU_ETHTYPE_ARP 0x0806
#define TEMU_ETHTYPE_WAKE_ON_LAN 0x0842
#define TEMU_ETHTYPE_SRP 0x22ea
#define TEMU_ETHTYPE_RARP 0x8035
#define TEMU_ETHTYPE_802_1Q 0x8100
#define TEMU_ETHTYPE_IPV6 0x86dd
#define TEMU_ETHTYPE_PTP 0x88f7
#define TEMU_ETHTYPE_TTE_CTRL 0x891d

// Note, this excludes 802.1Q field
#define TEMU_ETH_IPV4_VERS_IHL_OFFSET (TEMU_ETH_PAYLOAD_OFFSET + 0)
#define TEMU_ETH_IPV4_DSCP_ECN_OFFSET (TEMU_ETH_PAYLOAD_OFFSET + 1)
#define TEMU_ETH_IPV4_TOTAL_LENGTH_OFFSET (TEMU_ETH_PAYLOAD_OFFSET + 2)
#define TEMU_ETH_IPV4_FRAGMENTATION_INFO_OFFSET (TEMU_ETH_PAYLOAD_OFFSET + 6)
#define TEMU_ETH_IPV4_TTL_OFFSET (TEMU_ETH_PAYLOAD_OFFSET + 8)
#define TEMU_ETH_IPV4_PROTOCOL_OFFSET (TEMU_ETH_PAYLOAD_OFFSET + 9)
#define TEMU_ETH_IPV4_CRC_OFFSET (TEMU_ETH_PAYLOAD_OFFSET + 10)
#define TEMU_ETH_IPV4_SRC_IP_OFFSET (TEMU_ETH_PAYLOAD_OFFSET + 12)
#define TEMU_ETH_IPV4_DST_IP_OFFSET (TEMU_ETH_PAYLOAD_OFFSET + 16)

// Long list of allocated IP protocols
#define TEMU_IP_PROT_HOPOPT 0x00
#define TEMU_IP_PROT_ICMP 0x01
#define TEMU_IP_PROT_IGMP 0x02
#define TEMU_IP_PROT_GGP 0x03
#define TEMU_IP_PROT_IP_IN_IP 0x04
#define TEMU_IP_PROT_ST 0x05
#define TEMU_IP_PROT_TCP 0x06
#define TEMU_IP_PROT_CBT 0x07
#define TEMU_IP_PROT_EGP 0x08
#define TEMU_IP_PROT_IGP 0x09
#define TEMU_IP_PROT_BBN_RCC_MON 0x0a
#define TEMU_IP_PROT_NVP_II 0x0b
#define TEMU_IP_PROT_PUP 0x0c
#define TEMU_IP_PROT_ARGUS 0x0d
#define TEMU_IP_PROT_EMCON 0x0e
#define TEMU_IP_PROT_XNET 0x0f
#define TEMU_IP_PROT_CHAOS 0x10
#define TEMU_IP_PROT_UDP 0x11
#define TEMU_IP_PROT_MUX 0x12
#define TEMU_IP_PROT_DCN_MEAS 0x13
#define TEMU_IP_PROT_HMP 0x14
#define TEMU_IP_PROT_PRM 0x15
#define TEMU_IP_PROT_XNS_IDP 0x16
#define TEMU_IP_PROT_TRUNK_1 0x17
#define TEMU_IP_PROT_TRUNK_2 0x18
#define TEMU_IP_PROT_LEAF_1 0x19
#define TEMU_IP_PROT_LEAF_2 0x1a
#define TEMU_IP_PROT_RDP 0x1b
#define TEMU_IP_PROT_IRTP 0x1c
#define TEMU_IP_PROT_ISO_TP4 0x1d
#define TEMU_IP_PROT_NETBLT 0x1e
#define TEMU_IP_PROT_MFE_NSP 0x1f
#define TEMU_IP_PROT_MERIT_INP 0x20
#define TEMU_IP_PROT_DCCP 0x21
#define TEMU_IP_PROT_3CP 0x22
#define TEMU_IP_PROT_IDPR 0x23
#define TEMU_IP_PROT_XTP 0x24
#define TEMU_IP_PROT_DDP 0x25
#define TEMU_IP_PROT_IDPR_CMTP 0x26
#define TEMU_IP_PROT_TP_PLUS_PLUS 0x27
#define TEMU_IP_PROT_IL 0x28
#define TEMU_IP_PROT_IPV6_ENCAP 0x29
#define TEMU_IP_PROT_SDRP 0x2a
#define TEMU_IP_PROT_IPV6_ROUTE 0x2b
#define TEMU_IP_PROT_IPV6_FRAG 0x2c
#define TEMU_IP_PROT_IDRP 0x2d
#define TEMU_IP_PROT_RSVP 0x2e
#define TEMU_IP_PROT_GRE 0x2f
#define TEMU_IP_PROT_DSR 0x30
#define TEMU_IP_PROT_BNA 0x31
#define TEMU_IP_PROT_ESP 0x32
#define TEMU_IP_PROT_AH 0x33
#define TEMU_IP_PROT_I_NLSP 0x34
#define TEMU_IP_PROT_SW_IP_E 0x35
#define TEMU_IP_PROT_NARP 0x36
#define TEMU_IP_PROT_MOBILE 0x37
#define TEMU_IP_PROT_TLSP 0x38
#define TEMU_IP_PROT_SKIP 0x39
#define TEMU_IP_PROT_IPV6_ICMP 0x3a
#define TEMU_IP_PROT_IPV6_NO_NXT 0x3b
#define TEMU_IP_PROT_IPV6_OPTS 0x3c

#define TEMU_IP_PROT_CFTP 0x3e

#define TEMU_IP_PROT_SAT_EXPAK 0x40
#define TEMU_IP_PROT_KRYPTOLAN 0x41
#define TEMU_IP_PROT_RVD 0x42
#define TEMU_IP_PROT_IPPC 0x43

#define TEMU_IP_PROT_SAT_MON 0x45
#define TEMU_IP_PROT_VISA 0x46
#define TEMU_IP_PROT_IPCU 0x47
#define TEMU_IP_PROT_CPNX 0x48
#define TEMU_IP_PROT_CPHB 0x49
#define TEMU_IP_PROT_WSN 0x4a
#define TEMU_IP_PROT_PVP 0x4b
#define TEMU_IP_PROT_BR_SAT_MON 0x4c
#define TEMU_IP_PROT_SUN_ND 0x4d
#define TEMU_IP_PROT_WB_MON 0x4e
#define TEMU_IP_PROT_WB_EXPAK 0x4f

#define TEMU_IP_PROT_ISO_IP 0x50
#define TEMU_IP_PROT_VMTP 0x51
#define TEMU_IP_PROT_SECURE_VMTP 0x52
#define TEMU_IP_PROT_VINES 0x53
#define TEMU_IP_PROT_IPTM 0x54
#define TEMU_IP_PROT_NSFNTET_IGP 0x55
#define TEMU_IP_PROT_DGP 0x56
#define TEMU_IP_PROT_TCF 0x57
#define TEMU_IP_PROT_EIGRP 0x58
#define TEMU_IP_PROT_OSPF 0x59
#define TEMU_IP_PROT_SPRITE_RPC 0x5a
#define TEMU_IP_PROT_LARP 0x5b
#define TEMU_IP_PROT_MTP 0x5c
#define TEMU_IP_PROT_AX_25 0x5d
#define TEMU_IP_PROT_OS 0x5e
#define TEMU_IP_PROT_MICP 0x5f

#define TEMU_IP_PROT_SCC_SP 0x60
#define TEMU_IP_PROT_ETHERIP 0x61
#define TEMU_IP_PROT_ENCAP 0x62

#define TEMU_IP_PROT_GMTP 0x64
#define TEMU_IP_PROT_IFMP 0x65
#define TEMU_IP_PROT_PNNI 0x66
#define TEMU_IP_PROT_PIM 0x67
#define TEMU_IP_PROT_ARIS 0x68
#define TEMU_IP_PROT_SCPS 0x69
#define TEMU_IP_PROT_QNX 0x6a
#define TEMU_IP_PROT_AN 0x6b
#define TEMU_IP_PROT_IP_COMP 0x6c
#define TEMU_IP_PROT_SNP 0x6d
#define TEMU_IP_PROT_COMPAQ_PEER 0x6e
#define TEMU_IP_PROT_IPX_IN_IP 0x6f

#define TEMU_IP_PROT_VRRP 0x70
#define TEMU_IP_PROT_PGM 0x71

#define TEMU_IP_PROT_L2TP 0x73
#define TEMU_IP_PROT_DDX 0x74
#define TEMU_IP_PROT_IATP 0x75
#define TEMU_IP_PROT_STP 0x76
#define TEMU_IP_PROT_SRP 0x77
#define TEMU_IP_PROT_UTI 0x78
#define TEMU_IP_PROT_SMP 0x79
#define TEMU_IP_PROT_SM 0x7a
#define TEMU_IP_PROT_PTP 0x7b
#define TEMU_IP_PROT_IS_IS_OVER_IPV4 0x7c
#define TEMU_IP_PROT_FIRE 0x7d
#define TEMU_IP_PROT_CRTP 0x7e
#define TEMU_IP_PROT_CRUDP 0x7f

#define TEMU_IP_PROT_SSCOPMCE 0x80
#define TEMU_IP_PROT_IPLT 0x81
#define TEMU_IP_PROT_SPS 0x82
#define TEMU_IP_PROT_PIPE 0x83
#define TEMU_IP_PROT_SCTP 0x84
#define TEMU_IP_PROT_FC 0x85
#define TEMU_IP_PROT_RSVP_E2E_IGNORE 0x86
#define TEMU_IP_PROT_MOBILITY_HEADER 0x87
#define TEMU_IP_PROT_UDP_LITE 0x88
#define TEMU_IP_PROT_MPLS_IN_IP 0x89
#define TEMU_IP_PROT_MANET 0x8a
#define TEMU_IP_PROT_HIP 0x8b
#define TEMU_IP_PROT_SHIM6 0x8c
#define TEMU_IP_PROT_WESP 0x8d
#define TEMU_IP_PROT_ROHC 0x8e
#define TEMU_IP_PROT_ETHERNET 0x8f

#define TEMU_IP_PROT_AGGFRAG 0x90
#define TEMU_IP_PROT_NSH 0x91
#define TEMU_IP_PROT_HOMA 0x92
#define TEMU_IP_PROT_BIT_EMU 0x93

#define TEMU_IP_PROT_EXPERIMENTAL_0 0xfd
#define TEMU_IP_PROT_EXPERIMENTAL_1 0xfe

// Relative to start of TCP/UDP headers
#define TEMU_TCP_CRC_OFFSET 16
#define TEMU_UDP_CRC_OFFSET 6

// Flags bit in the flags field in TCP header
#define TEMU_TCP_CWR_FLAG (1 << 7)
#define TEMU_TCP_ECE_FLAG (1 << 6)
#define TEMU_TCP_URG_FLAG (1 << 5)
#define TEMU_TCP_ACK_FLAG (1 << 4)
#define TEMU_TCP_PSH_FLAG (1 << 3)
#define TEMU_TCP_RST_FLAG (1 << 2)
#define TEMU_TCP_SYN_FLAG (1 << 1)
#define TEMU_TCP_FIN_FLAG (1 << 0)

static inline uint64_t
temu_ethGetDestMAC(temu_EthFrame *Frame)
{
  uint64_t MAC = 0;
  const uint8_t *Data = temu_buffReadableData(&Frame->Data);

  for (int i = 0; i < 6; ++i) {
    MAC <<= 8;
    MAC |= Data[i];
  }
  return MAC;
}

static inline uint64_t
temu_ethGetSourceMAC(temu_EthFrame *Frame)
{
  uint64_t MAC = 0;
  const uint8_t *Data = temu_buffReadableData(&Frame->Data);

  for (int i = 6; i < 12; ++i) {
    MAC <<= 8;
    MAC |= Data[i];
  }
  return MAC;
}
/*!
 * Get length / EtherType field.
 * By ethernet standard, the size field which is 16 bit, will be a length if it
 * is <= 1500 and an ethertype tag if it is >= 1536.
 */
static inline uint16_t
temu_ethGetEthTypeSizeField(temu_EthFrame *Frame)
{
  uint16_t Size = 0;
  const uint8_t *Data = temu_buffReadableData(&Frame->Data);

  for (int i = 12; i < 14; ++i) {
    Size <<= 8;
    Size |= Data[i];
  }

  if (Size == TEMU_ETHTYPE_802_1Q) {
    Size = 0;
    for (int i = 12 + TEMU_ETH_802_1Q_TAG_BYTES;
         i < 14 + TEMU_ETH_802_1Q_TAG_BYTES; ++i) {
      Size <<= 8;
      Size |= Data[i];
    }
  }

  return Size;
}

static inline bool
temu_ethIs8021Q(temu_EthFrame *Frame)
{
  uint16_t TagStart = 0;
  const uint8_t *Data = temu_buffReadableData(&Frame->Data);

  for (int i = 12; i < 14; ++i) {
    TagStart <<= 8;
    TagStart |= Data[i];
  }

  return TagStart == TEMU_ETHTYPE_802_1Q;
}

static inline uint16_t
temu_eth8021QTagBytes(temu_EthFrame *Frame)
{
  return temu_ethIs8021Q(Frame) ? TEMU_ETH_802_1Q_TAG_BYTES : 0;
}

static inline uint16_t
temu_ethGetLength(temu_EthFrame *Frame)
{
  // Should compute number of bits / bytes for the whole frame
  // including headers
  return temu_buffLen(&Frame->Data);
}

/*!
 * Get a readable pointer to the payload.
 * Note that the first bytes will often be an LLC header, it may not be your
 * actual data payload.
 */
static inline const uint8_t *
temu_ethGetPayload(temu_EthFrame *Frame)
{
  const uint8_t *Data = temu_buffReadableData(&Frame->Data);
  return &Data[TEMU_ETH_PAYLOAD_OFFSET + temu_eth8021QTagBytes(Frame)];
}
static inline uint8_t *
temu_ethGetWritablePayload(temu_EthFrame *Frame)
{
  uint8_t *Data = temu_buffWritableData(&Frame->Data);
  return &Data[TEMU_ETH_PAYLOAD_OFFSET + temu_eth8021QTagBytes(Frame)];
}

// Return actual payload size (including padding)
static inline const uint16_t
temu_ethGetPayloadAndPaddingSize(temu_EthFrame *Frame)
{
  return temu_ethGetLength(Frame) - TEMU_ETH_PAYLOAD_OFFSET -
         TEMU_ETH_CRC_LENGTH - temu_eth8021QTagBytes(Frame);
}

#define ETH_BCAST_ADDR UINT64_C(0xffffffffffff)

static inline int
temu_ethIsBroadcastFrame(temu_EthFrame *Frame)
{
  uint64_t MAC = temu_ethGetDestMAC(Frame);
  if (MAC == ETH_BCAST_ADDR) {
    return 1;
  }

  return 0;
}

// Return 1 if frame is multicast (OR BROADCAST)
static inline int
temu_ethIsMulticastFrame(temu_EthFrame *Frame)
{
  const uint8_t *Data = temu_buffReadableData(&Frame->Data);

  if (Data[0] & 1)
    return 1;
  return 0;
}

// Len field specify ethertype if larger or equal to this value
#define ETH_ETHERTYPE_MIN 1536

#define ETH_CRC_MAGIC 0xC704DD7B
#define ETH_CRC_POLY_NORM 0x04C11DB7
#define ETH_CRC_POLY_REV 0xEDB88320
#define ETH_CRC_POLY_REV_RECIP 0x82608EDB

#define ETH_CRC_POLY_LE 0xedb88320
#define ETH_CRC_POLY_BE 0x04c11db6

#ifdef __cplusplus
}
#endif
#endif /* !TEMU_ETHERNET_H */
