//===------------------------------------------------------------*- C++ -*-===//
//
// TEMU: The Terma Emulator
// (c) Terma 2021
// Authors: Daria Vorotnikova <davo (at) terma.com>
//
//===----------------------------------------------------------------------===//

#ifndef TEMU_BUS_SPI_H
#define TEMU_BUS_SPI_H
#include <assert.h>
#include <stddef.h>
#include <stdint.h>

#include "temu-c/Memory/Memory.h"
#include "temu-c/Models/Device.h"
#include "temu-c/Support/Logging.h"
#include "temu-c/Support/Objsys.h"

#define teSPI_WRSR 0x01  // write status register
#define teSPI_WRITE 0x02 // write command
#define teSPI_READ 0x03  // read command
#define teSPI_WDI 0x04   // write disable
#define teSPI_STAT 0x05  // read status register
#define teSPI_WEN 0x06   // write enable

// -------------------------
// This header contains names which include master/slave terminology. The TEMU
// project does not use such terms normally, however given that the hardware
// specification, this header deals with, use these terms they are reused here
// for consistency.
// Would the hardware specification change the terminology, the
// header will be updated in compliance with the TEMU version number policy.
// ------------------------

typedef struct {
  uint8_t charLength;
  uint8_t chipNum;
} temu_SpiDevConfig;

typedef struct {
  temu_Object Super;
  temu_SpiDevConfig DevParams;
} temu_SpiSlaveDevice;

typedef struct {
  void (*send)(temu_Object *, uint8_t *Data, uint32_t size);
  void (*raiseAlert)(temu_Object *, uint8_t chipNum);
  void (*lowerAlert)(temu_Object *, uint8_t chipNum);
} temu_SpiMasterDeviceIface;
TEMU_IFACE_REFERENCE_TYPE(temu_SpiMasterDevice)
#define TEMU_SPI_MASTER_IFACE_TYPE "temu::SpiMasterDeviceIface"

typedef struct {
  void (*send)(temu_Object *, uint8_t *Data, uint32_t size, uint32_t configSize,
               uint32_t responseSize);
} temu_SpiSlaveDeviceIface;
TEMU_IFACE_REFERENCE_TYPE(temu_SpiSlaveDevice)
#define TEMU_SPI_SLAVE_DEV_IFACE_TYPE "temu::SpiSlaveDeviceIface"

typedef struct {
  temu_Object Super;
  temu_SpiSlaveDeviceIfaceRefArray SpiSlaveDevices;
  temu_SpiMasterDeviceIfaceRef SpiMasterDevice;
  uint8_t CurrentChipNum;
} temu_SpiBus;

typedef struct {
  void (*raiseAlert)(temu_Object *, uint8_t chipNum);
  void (*lowerAlert)(temu_Object *, uint8_t chipNum);
  void (*sendToMaster)(temu_Object *, uint8_t *Data, uint32_t size);
  void (*sendToSlave)(temu_Object *, uint8_t *Data, uint32_t size,
                      uint32_t configSize, uint8_t chipNum,
                      uint32_t responseSize);
} temu_SpiBusIface;
TEMU_IFACE_REFERENCE_TYPE(temu_SpiBus)
#define TEMU_SPI_BUS_IFACE_TYPE "temu::SpiBusIface"

static inline void
temu_spiDeviceRegister(temu_Class *C)
{
  temu_addProperty(C, "chipNumber",
                   offsetof(temu_SpiSlaveDevice, DevParams.chipNum), teTY_U8,
                   1, // Number of elements (1 = scalar)
                   NULL, NULL, "eSPI device chip id");
  temu_addProperty(C, "charLength",
                   offsetof(temu_SpiSlaveDevice, DevParams.charLength), teTY_U8,
                   1, // Number of elements (1 = scalar)
                   NULL, NULL, "Character length in bits per character.");
}

static inline void
temu_spiBusRegister(temu_Class *C)
{
  temu_addProperty(C, "spiMasterDevice", offsetof(temu_SpiBus, SpiMasterDevice),
                   teTY_IfaceRef,
                   1, // Number of elements (1 = scalar)
                   NULL, NULL, "eSpi master communication Iface");
  temu_requireInterface(C, "spiMasterDevice", TEMU_SPI_MASTER_IFACE_TYPE);

  temu_addProperty(C, "spiSlaveDevices", offsetof(temu_SpiBus, SpiSlaveDevices),
                   teTY_IfaceRefArray, 1, NULL, NULL, "eSPI slave devices");

  temu_addProperty(C, "currentChipNumber",
                   offsetof(temu_SpiBus, CurrentChipNum), teTY_U8,
                   1, // Number of elements (1 = scalar)
                   NULL, NULL, "Last chosen chip");
}

typedef struct temu_SpiRomIface {
  void (*getData)(void *obj, uint64_t offset, size_t *length, uint8_t **data);
} temu_SpiRomIface;
TEMU_IFACE_REFERENCE_TYPE(temu_SpiRom);
#define TEMU_SPI_ROM_IFACE_TYPE "temu::SpiRomIface"

#endif // !TEMU_BUS_SPI_H