//===-- temu-c/Component.h - TEMU Component Support ------------*- C++ -*-===//
//
// TEMU: The Terma Emulator
// (c) Terma 2016
// Authors: Mattias Holm <maho (at) terma.com>
//
//===----------------------------------------------------------------------===//

//------------------------------------------------------------------------------
// WARNING!!! WARNING!!!    EXPERIMENTAL API   WARNING!!!  WARNING!!! WARNING!!!
//
// BE WARNED THAT THIS API IS EXPERIMENTAL, UNSTABLE AND NOT WELL TESTED.
// IN PARTICULAR, THE API MAY CRASH ON OCCASIONS.
//------------------------------------------------------------------------------
#ifndef TEMU_SUPPORT_COMPONENT_H
#define TEMU_SUPPORT_COMPONENT_H

#include "temu-c/Support/Attributes.h"
#include "temu-c/Support/Objsys.h"

#ifdef __cplusplus
extern "C" {
#endif

#ifndef TEMU_COMPONENT_DEFINED
#define TEMU_COMPONENT_DEFINED
typedef struct temu_Component temu_Component;
#endif // !TEMU_COMPONENT_DEFINED

/*!
 * Register a new component class in
 *
 * \param CompClass Name of the component to be registered
 * \param Create The constructor function that allocates the component
 * \param Dispose The destructor function
 * \result Pointer to the new class
 */
TEMU_API temu_Class *temu_registerComponent(const char *CompClass,
                                            temu_ObjectCreateFunc Create,
                                            temu_ObjectDisposeFunc Dispose);

/*!
 * Allocate a component object.
 *
 * The component create shall be called in the component
 * constructor/create function (the create function passed to the
 * registerComponent function).  It will allocate an opaque component
 * object with the relevant name.
 *
 * The returned component is what the component constructor should
 * return.
 *
 * \param Name Name of component
 * \result Component pointer
 */
TEMU_API temu_Component_ *temu_componentCreate(const char *Name);

/*!
 * Deallocate a component object
 *
 * The component dispose shall be called by the component
 * destructor/dispose function registered in the registerComponent
 * call.
 *
 * Note that a component is seen as owning all the objects created
 * with createComponentObject, and subsequently, all objects created
 * will be recursively deleted when deleting the component.
 *
 * \param Comp Component to dispose.
 */
TEMU_API void temu_componentDispose(void *Comp); // Default destructor

/*!
 * Add delegate interface to component
 *
 * Delegate interfaces are interfaces in objects internal in the
 * component. It is possible to use the connect function to attach
 * directly to a component delegated interface (although, in practice
 * the connection is done to the internal object's interface).
 *
 * Note that delegate interface do not support interface arrays, and
 * can only expose a single interface instance at present.
 *
 * \param Comp the component to add the delegate interface to.
 * \param Name name of the delegate interface
 * \param Iface Interface reference which the delegated interface is
 *              resolved to
 */
TEMU_API void temu_componentAddDelegateIface(temu_Component_ *Comp,
                                             const char *Name,
                                             temu_IfaceRef Iface);

/*!
 * Add delegate property to component
 *
 * Delegate property are properties in objects internal in the
 * component. It is possible to use the connect function to connect
 * from a delegated property directly using the component instead of
 * the underlying object.
 *
 * \param Comp the component to add the delegate interface to.
 * \param Name name of the delegate property
 * \param Obj Object to which the property resolves to.
 * \param PropName Name of property in Obj
 */

TEMU_API void temu_componentAddDelegateProp(temu_Component_ *Comp,
                                            const char *Name, temu_Object_ *Obj,
                                            const char *PropName);

/*!
 * Query the component for a delegated interface
 *
 * Normally this function is not needed for the end user, however
 * some usecases can be seen for exposing this. It returns the
 * IfaceRef that has been added as a delegate interface. The main use
 * is to redelegate delegated interfaces in a component of components.
 *
 * \param Comp The component to query the IfaceRef from
 * \param Name Name of the interface reference.
 * \result Interface reference associated with Name, if none is found,
 *                   iref type will be teTY_Invalid.
 */
TEMU_API temu_IfaceRef temu_componentGetDelegateIface(temu_Component *Comp,
                                                      const char *Name);

/*!
 * Query the compontent for a delegated property
 *
 * Normally this should not be called by the user. But is useful in
 * case the user constructs components of components in which case a
 * component can redelegate delegated properties.
 *
 * \param Comp The componten to query
 * \param Name Name of delegated property
 * \result Object name pair with the target object and property name.
 */
TEMU_API temu_PropName temu_componentGetDelegateProp(temu_Component *Comp,
                                                     const char *Name);

/*!
 * Create an object in a component.
 *
 * This function works as the temu_createObject, except that the
 * object created is inserted in the component.
 *
 * The function is indended to be used in the component constructor.
 *
 * Object names are uniqued using the objnamefmt paramters and
 * following varargs. Plus, that the name of the component itself is
 * prefixed as '[compname]-'.
 *
 * \param Comp The component under which the object is to be created.
 * \param Class class of the created object
 * \param Args NULL terminated array of create arguments for the
 *             constructor.
 * \param ObjNameFmt Name of created object, but it allows for printf
 *                   style string names, simplifying the creation of
 *                   multiple objects of the same type.
 * \result Pointer to created object.
 */
TEMU_API temu_Object_ *temu_createComponentObject(temu_Component_ *Comp,
                                                  const char *Class,
                                                  const temu_CreateArg *Args,
                                                  const char *ObjNameFmt, ...);

/*!
 * Get named object in component
 * \param Comp Component pointer
 * \param Name String without the component prefix (i.e. local name)
 * \result NULL if the object is not a member in the component
 */
TEMU_API temu_Object_ *temu_componentGetObject(temu_Component_ *Comp,
                                               const char *Name);

/*!
 * Get number of processors in component
 * \param Comp Component pointer
 * \result Number of processors in component.
 */
TEMU_API size_t temu_componentGetNumberOfProcessors(temu_Component *Comp);

/*!
 * Get processor in component
 * \param Comp Component pointer
 * \param index Index of processor in component
 * \result NULL if index is out of bounds.
 */
TEMU_API temu_Object *temu_componentGetProcessor(temu_Component *Comp,
                                                 unsigned index);

// Call function on every component
/*!
 * Iterate over all components and call a function on them
 *
 * \param Func The function to be called on each component, with the signature
 * *Func)(temu_Component*,void*) \param Arg The second argument to be passed to
 * the function, for passing context
 */
TEMU_API void temu_foreachComponent(void (*Func)(temu_Component *, void *),
                                    void *Arg);

/*!
 * Iterate over all root components and call a function on them
 *
 * \param Func The function to be called on each component, with the signature
 * *Func)(temu_Component*,void*) \param Arg The second argument to be passed to
 * the function, for passing context
 */
TEMU_API void temu_foreachRootComponent(void (*Func)(temu_Component *, void *),
                                        void *Arg);


/*!
 * Get component context
 * \param comp Component pointer
 * \result Context pointer
 */
TEMU_API void *temu_componentGetContext(temu_Component *comp);

/*!
 * set component context
 * \param comp Component pointer
 * \param ctxt Context pointer
 */
TEMU_API void temu_componentSetContext(temu_Component *comp, void *ctxt);


#ifdef __cplusplus
}
#endif

#endif /* !TEMU_SUPPORT_COMPONENT_H */
