//===-- temu-c/Cpu.h - Micro-processor functions ----------------*- C++ -*-===//
//
// TEMU: The Terma Emulator
// (c) Terma 2015
// Authors: Mattias Holm <maho (at) terma.com>
//
//===----------------------------------------------------------------------===//

#ifndef TEMU_SUPPORT_CPU
#define TEMU_SUPPORT_CPU
#include "temu-c/Support/Attributes.h"
/*!
  \file Cpu.h

  Wrappers functions for the CPU interfaces.

  The functions declared in this file exists for convenience
  purposes. They will in general look up the relevant interface by
  name via a double map lookup.

  The functions here provide an easy way invoke functions in the
  CpuIface without having to query for the interface and then calling
  the function.

  If the emulator is integrated in a simulator, the recommended
  approach is to first construct the system, and then query relevant
  interfaces, caching them as needed. Note that system configurations
  should not normally change after the construction phase, so any
  cached object-interface pairs will be stable.

  \warning A general rule is that the functions defined here calls
  abort in-case the interface cannot be found for the relevant object.
*/

#include <stdint.h>

#ifdef __cplusplus
extern "C" {
#endif

/*!
  Get the clock frequency for the CPU

  \warning In case the Cpu does not implement the CpuIface the program
  will abort. Do not use this function in performance critical code!!!

  \param Cpu The CPU object
  \result The configured clock frequency in Hz.
 */
TEMU_API uint64_t temu_cpuGetFreq(void *Cpu);

/*!
  Set the program counter

  The program counter will be set to the supplied value.

  \note For targets with delay slots (SPARC, MIPS etc), the function
  will also set the next PC to PC + sizeof(instruction).

  \warning In case the Cpu does not implement the CpuIface the program
  will abort. Do not use this function in performance critical code!!!

  \param Cpu The CPU object
  \param Pc The program counter.
 */
TEMU_API void temu_cpuSetPc(void *Cpu, uint64_t Pc);

/*!
  Get the program counter

  The program counter will be returned.

  \warning In case the Cpu does not implement the CpuIface the program
  will abort. Do not use this function in performance critical code!!!

  \param Cpu The CPU object
  \result The value of the program counter register
 */
TEMU_API uint64_t temu_cpuGetPc(void *Cpu);

/*!
  Reset the processor.

  Resetting the CPU will result in a reset cascade where all connected
  devices are also reset.

  \warning In case the Cpu does not implement the CpuIface the program
  will abort. Do not use this function in performance critical code!!!

  \param Cpu The CPU object

  \param ResetType The type of reset, by convention 0 means cold
                   reset, and 1 indicates a warm reset.
 */
TEMU_API void temu_cpuReset(void *Cpu, int ResetType);

/*!
  Run the processor for a number of cycles

  The function runs the processor for a number of cycles. If you wish
  to run the processor with another time unit, you can compute the
  cycles from the clock frequency of the emulated processor.

  In case the processor halts or enters idle mode and there are no
  pending events the function will return early.

  \warning In case the Cpu does not implement the CpuIface the program
  will abort. Do not use this function in performance critical code!!!

  \param Cpu The CPU object
  \param Cycles The number of cycles to run the processor for.
  \result The number of executed cycles.
 */
TEMU_API uint64_t temu_cpuRun(void *Cpu, uint64_t Cycles);

/*!
  Run the processor for a number of steps

  This function is different from temu_cpuRun, which runs for a
  time. The steps here indicates instructions executed (including
  trapping instructions). This can be contrasted to the run function
  which may advance the cycle counter by more than one for an
  instruction (depending on the timing models).

  The function may return early in case the processor halts its
  execution or has entered idle mode and there are no events pending.

  \warning In case the Cpu does not implement the CpuIface the program
  will abort. Do not use this function in performance critical code!!!

  \param Cpu The CPU object
  \param Steps The number of steps to run the processor for.
  \result The number of executed steps.
 */
TEMU_API uint64_t temu_cpuStep(void *Cpu, uint64_t Steps);

/*!
  Uses the MMU to translate the given virtual address Va
  to the physical address in the emulator
 
  \param Cpu the CPU object
  \param Va the virtual address to be translated
  \param flags flags for the translation (CPU specific)
  \param physAddressResult the result in a uint64_t pointer
  \return error code. 0 on success, non-zero otherwise,
 */
TEMU_API int temu_cpuTranslateAddress(void *Cpu, uint64_t Va, uint32_t flags,
                                      uint64_t *physAddressResult);



/*!
  Uses the an explicit page table pointer to translate a virtual address
  to physical address.

  The routine is dependent on the processor having hardware managed page tables.
  Hence this works e.g. for SPARC and ARM processors, but not for PowerPC. 
 
  \param cpu the CPU object
  \param va the virtual address to be translated
  \param flags flags for the translation (CPU specific)
  \param physAddressResult the result in a uint64_t pointer
  \param pageTableRoot The page table to use for the translation
  \return error code. 0 on success, non-zero otherwise,
 */
TEMU_API int
temu_cpuTranslateAddressWithRootPointer(void *cpu, uint64_t va, uint32_t flags,
                                        uint64_t *physAddressResult,
                                        uint64_t pageTableRoot);

/*!
 * Gets the register in the processor
 * \param Cpu CPU pointer
 * \param Reg Register number
 * \result Register value
 */
TEMU_API uint64_t temu_cpuGetReg(void *Cpu, unsigned Reg);

/*!
 * Set the register in the processor
 * \param Cpu CPU pointer
 * \param Reg Register number
 * \param Value Value to write to register
 */
TEMU_API void temu_cpuSetReg(void *Cpu, unsigned Reg, uint64_t Value);

/*!
 * Get a 32 bit floating point register
 * \param Cpu CPU pointer
 * \param Reg FPU register number
 * \result Host float with the content of the FPU register
 */
TEMU_API float temu_cpuGetFpr32(void *Cpu, unsigned Reg);

/*!
 * Get a 32 bit floating point register
 * \param Cpu CPU pointer
 * \param Reg FPU register number
 * \result Floating point register contents
 */
TEMU_API uint32_t temu_cpuGetFpr32Bits(void *Cpu, unsigned Reg);

/*!
 * Set floating point register value
 * \param Cpu CPU pointer
 * \param Reg Floating point register number
 * \param Value Floating point value to set
 */
TEMU_API void temu_cpuSetFpr32(void *Cpu, unsigned Reg, float Value);

/*!
 * Set 32 bit floating point register value
 * \param Cpu CPU pointer
 * \param Reg Floating point register number
 * \param Value Floating point value to set
 */

TEMU_API void temu_cpuSetFpr32Bits(void *Cpu, unsigned Reg, uint32_t Value);

/*!
 * Get 64 bit floating point register as double
 * \param Cpu CPU pointer
 * \param Reg FPU register number
 * \result FPU register value
 */
TEMU_API double temu_cpuGetFpr64(void *Cpu, unsigned Reg);

/*!
 * Get 64 bit floating point register contents
 * \param Cpu CPU pointer
 * \param Reg FPU register number
 * \result FPU register value
 */
TEMU_API uint64_t temu_cpuGetFpr64Bits(void *Cpu, unsigned Reg);

/*!
 * Set FPU register value
 * \param Cpu CPU pointer
 * \param Reg FPU register number
 * \param Value Value to set to register
 */
TEMU_API void temu_cpuSetFpr64(void *Cpu, unsigned Reg, double Value);

/*!
 * Set FPU register value
 * \param Cpu CPU pointer
 * \param Reg FPU register number
 * \param Value Value to set to register
 */
TEMU_API void temu_cpuSetFpr64Bits(void *Cpu, unsigned Reg, uint64_t Value);

/*!
 * Enable traps on processor
 * \param Cpu CPU pointer
 */
TEMU_API void temu_cpuEnableTraps(void *Cpu);

/*!
 * Disable traps on processor
 * \param Cpu CPU pointer
 */
TEMU_API void temu_cpuDisableTraps(void *Cpu);

/*!
 * Raise a trap
 * The function simulates a trap being raised at the current PC.
 *
 * \param Cpu Processor pointer
 * \param Trap Trap ID. This is target dependent, for the SPARC this
 *             is the TT value of the trap.
 *
 * \param Flags set flag 1 to enable longjmp trap (this is useful in
 *              MMIO handlers to force a trap while a core is
 *              running). Set to 0 if called from outside the core
 *              or in e.g. an event handler.
 */
TEMU_API void temu_cpuRaiseTrap(void *Cpu, int Trap, unsigned Flags);

/* Sparc specific functions */

typedef void (*temu_SparcAsrHandler)(void *Cpu, uint32_t Instr);
// typedef void (*temu_SparcAsiHandler)(void *Cpu, temu_MemTransaction *MT);

/*!
 * Get how many register windows are supported by the CPU
 * \param Cpu SPARC CPU pointer
 * \result Number of register windows
 */
TEMU_API int temu_sparcGetWindowCount(void *Cpu);

/*!
 * Get a windowed SPARC register
 * \param Cpu SPARC CPU pointer
 * \param Window Window number (-1 for current window)
 * \param Reg Register number within window
 * \result Register value
 */
TEMU_API uint32_t temu_sparcGetWindowedReg(void *Cpu, int Window, unsigned Reg);

/*!
 * Set a windowed SPARC register
 * \param Cpu SPARC CPU pointer
 * \param Window Window number (-1 for current window)
 * \param Reg Register number within window
 * \param Value Register value
 */
TEMU_API void temu_sparcSetWindowedReg(void *Cpu, int Window, unsigned Reg,
                                       uint32_t Value);

/*!
 * Set the Y register of the CPU,
 * \param Cpu SPARC CPU pointer
 * \param Value Value to write to the Y register
 */
TEMU_API void temu_sparcSetY(void *Cpu, uint64_t Value);

/*!
 * Get the Y register of the CPU,
 * \param Cpu SPARC CPU pointer
 * \result Current Y register value
 */
TEMU_API uint64_t temu_sparcGetY(void *Cpu);

/*!
 * Set the ASR register
 * \param Cpu SPARC CPU pointer
 * \param Reg ASR register ID
 * \param Value Value to write
 */
TEMU_API void temu_sparcSetAsr(void *Cpu, unsigned Reg, uint64_t Value);

/*!
 * Get ASR register value
 * \param Cpu SPARC CPU pointer
 * \param Reg ASR register ID
 * \result ASR register value
 */
TEMU_API uint64_t temu_sparcGetAsr(void *Cpu, unsigned Reg);

/*!
 * Install the handler to be called when an ASR is written
 * \param Cpu SPARC CPU pointer
 * \param Asr ASR register ID
 * \param Handler Function to call.
 */
TEMU_API void temu_sparcSetAsrWriter(void *Cpu, unsigned Asr,
                                     temu_SparcAsrHandler Handler);

/*!
 * Install the handler to be called when an ASR is read
 * \param Cpu SPARC CPU pointer
 * \param Asr ASR register ID
 * \param Handler Function to call.
 */
TEMU_API void temu_sparcSetAsrReader(void *Cpu, unsigned Asr,
                                     temu_SparcAsrHandler Handler);

/*!
 * Set the Processor State Register
 * \param Cpu SPARC CPU pointer
 * \param Value Value to write
 */
TEMU_API void temu_sparcSetPsr(void *Cpu, uint32_t Value);

/*!
 * Get the Processor State Register
 * \param Cpu SPARC CPU pointer
 * \result PSR value
 */
TEMU_API uint32_t temu_sparcGetPsr(void *Cpu);

/*!
 * Set the Trap Base Register
 * \param Cpu SPARC CPU pointer,
 * \param Value Value for TBR register
 */
TEMU_API void temu_sparcSetTbr(void *Cpu, uint32_t Value);

/*!
 * Get Trap Base Register value
 * \param Cpu SPARC CPU pointer
 * \result TBR value
 */
TEMU_API uint32_t temu_sparcGetTbr(void *Cpu);

/*!
 * Set window invalidation mask register
 * \param Cpu SPARC CPU pointer
 * \param Value value to set in WIM
 */
TEMU_API void temu_sparcSetWim(void *Cpu, uint32_t Value);

/*!
 * Get window invalidation mask register
 * \param Cpu SPARC CPU pointer
 * \result WIM value
 */
TEMU_API uint32_t temu_sparcGetWim(void *Cpu);

/*!
 * Set the nPC value (next program counter)
 *
 * NOTE: When you use the setPC function, the NPC is also
 * updated to PC+4. Use this to explicitly set NPC.
 *
 * \param Cpu SPARC CPU pointer
 * \param Value Value to write to NPC
 */
TEMU_API void temu_sparcSetNPc(void *Cpu, uint32_t Value);

/*!
 * Get the nPC value
 * \param Cpu SPARC CPU pointer
 * \result nPC value
 */
TEMU_API uint32_t temu_sparcGetNPc(void *Cpu);


/*!
 * Set the ARM processor state register
 * \param Cpu ARM CPU pointer
 * \param Value Value to set the APSR to
 */
TEMU_API void temu_armSetAPSR(void *Cpu, uint32_t Value);
/*!
 * Get the ARM processor state register
 * \param Cpu ARM CPU pointer
 * \result APSR value
 */
TEMU_API uint32_t temu_armGetAPSR(void *Cpu);
/*!
 * Set the ARM FPSCR register
 * \param Cpu ARM CPU pointer
 * \param Value FPSCR value
 */
TEMU_API void temu_armSetFPSCR(void *Cpu, uint32_t Value);
/*!
 * Get the ARM FPSCR register
 * \param Cpu ARM CPU pointer
 * \result FPSCR value
 */
TEMU_API uint32_t temu_armGetFPSCR(void *Cpu);

/*!
 * Set the ARM FPEXC register
 * \param Cpu ARM CPU pointer
 * \param Value FPEXC value
 */
TEMU_API void temu_armSetFPEXC(void *Cpu, uint32_t Value);
/*!
 * Get the ARM FPEXC register
 * \param Cpu ARM CPU pointer
 * \result FPEXC value
 */
TEMU_API uint32_t temu_armGetFPEXC(void *Cpu);

/*!
 * Set the ARM FPINST register
 * \param Cpu ARM CPU pointer
 * \param Idx FPINST register number
 * \param Value FPINST value
 */

TEMU_API void temu_armSetFPINST(void *Cpu, int Idx, uint32_t Value);
/*!
 * Get the ARM FPINST register
 * \param Cpu ARM CPU pointer
 * \param Idx FPINST register number
 * \result FPINST value
 */
TEMU_API uint32_t temu_armGetFPINST(void *Cpu, int Idx);

/*!
 * Get the ARM execution mode
 * \param Cpu ARM CPU pointer
 * \result Execution mode
 */
TEMU_API unsigned temu_armGetExecMode(void *Cpu);
/*!
 * Set the ARM execution mode
 * \param Cpu ARM CPU pointer
 * \param Mode Execution mode
 */
TEMU_API void temu_armSetExecMode(void *Cpu, unsigned Mode);

/*!
 * Set the PowerPC CR register
 * \param Cpu Processor pointer
 * \param Value New CR value
 */
TEMU_API void temu_ppcSetCrReg(void *Cpu, uint32_t Value);
/*!
 * Get the PowerPC CR register
 * \param Cpu Processor pointer
 * \result CR value
 */
TEMU_API uint32_t temu_ppcGetCrReg(void *Cpu);
/*!
 * Set the PowerPC XER register
 * \param Cpu Processor pointer
 * \param Value New XER value
 */
TEMU_API void temu_ppcSetXerReg(void *Cpu, uint32_t Value);
/*!
 * Get the PowerPC XER register
 * \param Cpu Processor pointer
 * \result XER value
 */
TEMU_API uint32_t temu_ppcGetXerReg(void *Cpu);

/*!
 * Set the PowerPC MSR register
 * \param Cpu Processor pointer
 * \param Value New MSR value
 */
TEMU_API void temu_ppcSetMsrReg(void *Cpu, uint64_t Value);
/*!
 * Get the PowerPC MSR register
 * \param Cpu Processor pointer
 * \result MSR value
 */
TEMU_API uint64_t temu_ppcGetMsrReg(void *Cpu);
/*!
 * Set the PowerPC reserve address used by `lwarx` and `stwcx`
 * \param Cpu Processor pointer
 * \param reserveAddress New reserved address value
 */
TEMU_API void temu_ppcSetReserveAddress(void *Cpu, uint64_t reserveAddress);
/*!
 * Get the PowerPC reserve address used by `lwarx` and `stwcx`
 * \param Cpu Processor pointer
 * \result The reserved address value
 */
TEMU_API uint64_t temu_ppcGetReserveAddress(void *Cpu);
/*!
 * Clear the PowerPC reserve address used by `lwarx` and `stwcx`
 * \param Cpu Processor pointer
 */
TEMU_API void temu_ppcClearAddressReservation(void *Cpu);
/*!
 * Check if PowerPC reserve address is set `lwarx` and `stwcx`
 * \param Cpu Processor pointer
 * \result 1 if reservation is set, 0 if cleared.
 */
TEMU_API int temu_ppcIsReservationBitSet(void *Cpu);
/*!
 * Set the PowerPC CTR register
 * \param Cpu Processor pointer
 * \param Value New CTR value
 */
TEMU_API void temu_ppcSetCtrReg(void *Cpu, uint32_t Value);
/*!
 * Get the PowerPC CTR register
 * \param Cpu Processor pointer
 * \result The CTR value
 */
TEMU_API uint32_t temu_ppcGetCtrReg(void *Cpu);
/*!
 * Set the PowerPC FPSCR register
 * \param Cpu Processor pointer
 * \param Value New FPSCR value
 */
TEMU_API void temu_ppcSetFpscrReg(void *Cpu, uint32_t Value);
/*!
 * Get the PowerPC FPSCR register
 * \param Cpu Processor pointer
 * \result The FPSCR value
 */
TEMU_API uint32_t temu_ppcGetFpscrReg(void *Cpu);
/*!
 * Set the PowerPC LR register
 * \param Cpu Processor pointer
 * \param Value New LR value
 */
TEMU_API void temu_ppcSetLrReg(void *Cpu, uint64_t Value);
/*!
 * Get the PowerPC LR register
 * \param Cpu Processor pointer
 * \result The LR value
 */
TEMU_API uint64_t temu_ppcGetLrReg(void *Cpu);
/*!
 * Set the PowerPC SPR register
 * \param Cpu Processor pointer
 * \param RegId SPR register number
 * \param Value New SPR value
 */
TEMU_API void temu_cpuSetSpr(void *Cpu, unsigned RegId, uint64_t Value);
/*!
 * Get the PowerPC SPR register
 * \param Cpu Processor pointer
 * \param RegId SPR register number
 * \result Value of SPR specified in `RegId`.
 */
TEMU_API uint64_t temu_cpuGetSpr(void *Cpu, unsigned RegId);

// Machine functions

/*!
  Get the Machine for a given CPU
  \param Cpu Pointer to the CPU object
  \result Pointer to the machine object
 */
TEMU_API void *temu_cpuGetMachine(void *Cpu);

/*!
  Reset the Machine

  Resetting the Machine will result in a reset cascade where all connected
  devices and processors are also reset.

  \warning In case the Machine does not implement the MachineIface the
  program will abort. Do not use this function in performance critical
  code!!!

  \param Machine The Machine object

  \param ResetType The type of reset, by convention 0 means cold
                   reset, and 1 indicates a warm reset.
 */
TEMU_API void temu_machineReset(void *Machine, int ResetType);

/*!
  Run the machine for a number of nanoseconds

  The function runs the machine and for a number of nanoseconds. This
  will run all the CPUs in the system.

  In case a processor halts or enters idle mode, the function returns
  early.

  \warning In case the machine does not implement the MachineIface the
  program will abort. Do not use this function in performance critical
  code!!!

  \param Machine The machine object
  \param NanoSecs The number of nanosecs to run each processor for.
  \result The number of executed nanoseconds.
 */
TEMU_API uint64_t temu_machineRun(void *Machine, uint64_t NanoSecs);

typedef void (*temu_SafeCb)(void *);

/*!
  Post a callback in a time source

  The posting will be thread-safe and the callback will be excuted by
  the main thread (the one calling the cpu or machine run / step
  functions). The main thread is the thread calling cpu run if there
  is only a single cpu in the system, and the thread calling machine
  run if there are multiple CPUs in the system.

  The callback will be executed as soon as possible, when the even
  queue is checked for events. Which in practice mean:

  - At the end of the current time quanta for multi-cpu systems.

  - At regular intervals for single CPU systems (a CPU runs a
    null-event periodically. Note that the callback will be
    called as soon as either a normal or the null event is executed.

  The posting of this callback is thread-safe but not async / signal
  safe. I.e. do not do this from signal handler or async io callbacks.

  When the event is executed, it is safe to do most emulator
  operations, including posting of events, calling API functions, etc.

  The callback must however not:

  - Manipulate the object system meta state (i.e. creating or
    modifying classes)

  - Replace signal handlers (e.g. SIGINT)

  - Run or step the emulator in any way (e.g. by calling machineRun(),
    cpuRun() or using the CpuIface or MachineIface interfaces.
  
  \param Obj Time source
  \param Cb Callback
  \param Arg Argument passed to callback
 */
TEMU_API void temu_postCallback(void *Obj, temu_SafeCb Cb, void *Arg);

/*!
 * Get processor statistics
 * 
 * \param Obj Processor pointer
 * \param Stat Statistics ID
 * \result Statistics value
 */
TEMU_API uint64_t temu_cpuGetStat(void *Obj, int Stat);
/*!
 * Clear processor statistics
 * 
 * \param Obj Processor pointer
 * \param Stat Statistics ID
 */
TEMU_API void temu_cpuResetStat(void *Obj, int Stat);

#ifdef __cplusplus
}
#endif

#endif /* ! TEMU_SUPPORT_CPU */
