//===-- temu-c/Init.h - TEMU Library Initialisation  -----------*- C++ -*-===//
//
// TEMU: The Terma Emulator
// (c) Terma 2015
// Authors: Mattias Holm <maho (at) terma.com>
//
//===----------------------------------------------------------------------===//

#ifndef TEMU_SUPPORT_INIT
#define TEMU_SUPPORT_INIT
#include "temu-c/Support/Attributes.h"
#ifdef __cplusplus
extern "C" {
#endif

/*!
 * Initialisation of TEMU support library.
 *
 * Call this function before any other use of TEMU. The function does
 * not return failures, but halts on failed initialisation.
 *
 * The function should be called as early as possible in the
 * application's execution. Preferably in the main function (though
 * there is no strict requirement for this).
 *
 * The initialisation function is at present not thread safe, call it
 * in the main thread only!
 *
 */
TEMU_API void temu_initSupportLib(void);

/*!
 * Create the temu config directories (typically ~/.config/temu/)
 *
 * This function is invoked by the init function normally, but it is
 * useful in some cases to create the config dirs without calling
 * init.
 */
TEMU_API void temu_initConfigDirs(void);


/*!
 * Init only license system.
 *
 * Initialise the license manager only. This can be used to check for
 * whether the license is valid before proceeding with rest of the
 * system initialisation. The function does not terminate the
 * application on invalid license, but returns a non-zero value.
 *
 * \result 0 if the license is valid, other values indicate errors.
 */
TEMU_API int temu_initLicense(void);

/*!
 * Initialise path finding support
 *
 * This function initialises auxillary path support needed by TEMU
 * for locating supporting tools (e.g. stuff in libexec) and
 * automatically appending of scripting search paths.
 *
 * In the future, this will likely be merged into initSupportLib for a
 * unified init function, but this will break API compatibility, so
 * this change is deferred to TEMU 3.
 *
 * Call the Function passing Argv0 of whatever you would do to invoke
 * the temu command line tool. E.g. if temu is in your path and this
 * is the installation you wish to use, the parameter should be
 * "temu". However, if you wish to use a custom installation, pass the
 * full path to the temu program. The base path can contain ~ for the
 * current home dir and ~foo for the home dir of user foo.
 *
 * Note that it is not an error to omit this initialisation, but
 * omitting it will imply that temu cannot launch UI models such as
 * the ConsoleUI model.
 *
 * \param Argv0 Name of TEMU executable (typically passed as argv[0]
 *              to main)
 */
TEMU_API void temu_initPathSupport(const char *Argv0);

#ifdef __cplusplus
}
#endif

#endif /* !TEMU_SUPPORT_INIT */
