//===-- temu-c/Loader.h - Binary image loading ------------------*- C++ -*-===//
//
// TEMU: The Terma Emulator
// (c) Terma 2015
// Authors: Mattias Holm <maho (at) terma.com>
//
//===----------------------------------------------------------------------===//
#ifndef TEMU_LOADER_H
#define TEMU_LOADER_H

#include "temu-c/Support/Attributes.h"
#include <stdint.h>

#ifdef __cplusplus
extern "C" {
#endif

/*!
  Loads a binary image to the memory object.

  The function will autodetect the file type (based on extensions and
  magic headers in the file).

  Binary images will be placed at address 0.

  The image can be one of the supported file formats.

  \param mem Memory object, must conform to the mem interface.
  \param file Path to file to load.
  \result 0 on success, other values indicates errors.
 */
TEMU_API int temu_loadImage(void *mem, const char *file);

/*!
  Loads a binary image to the memory object.

  The function will autodetect the file type (based on extensions and
  magic headers in the file).

  Binary images will be placed at address 0.

  The image can be one of the supported file formats.

  This specific variant, will return the start address of the binary,
  the StartAddr is typically embedded in the ELF or SREC files. The
  function therefore allows you to set the program counter to the
  correct start address after the function has been called.

  \param Mem Memory object, must conform to the mem interface.
  \param FileName Path to file to load.
  \param StartAddr Pointer to location to store the start address.
  \result 0 on success, 1 for successful load but without valid StartAddr,
    negative values indicate error.
 */

TEMU_API int temu_loadImageAndStartAddr(void *Mem, const char *FileName,
                                        uint64_t *StartAddr TEMU_NONNULL);

/*!
  Loads a binary image to the memory object.

  The function will autodetect the file type (based on extensions and
  magic headers in the file).

  Binary images will be placed at address 0.

  The image can be one of the supported file formats.

  This specific variant, will for file formats that have physical
  load addresses as well as virtual prefer the physical addresses.
  This is the case for e.g. ELF files.
  Other files are loaded with whatever address they define.

  \param Mem Memory object, must conform to the mem interface.
  \param FileName Path to file to load.
  \param StartAddr Pointer to location to store the start address.
  \result 0 on success, 1 for successful load but without valid StartAddr,
    negative values indicate error.
 */
TEMU_API int temu_loadImagePhys(void *Mem, const char *FileName,
                                uint64_t *StartAddr TEMU_NONNULL);

/*!
  Loads a raw binary image to the memory object at the given address.

  The function will assume that the file is a raw binary.
  This way you can load an ELF file as is to memory
  (e.g. where it is expected by the boot loader).

  \param mem Memory object, must conform to the mem interface.
  \param file Path to file to load.
  \param pa Physical address where to load the image.
  \result 0 on success, other values indicates errors.
 */
TEMU_API int temu_loadBinaryImage(void *mem, const char *file, uint64_t pa);

typedef void temu_Symtab;

/*!
  Loads an ELF file without going through filetype detection

  \param mem Memory object, must conform to the mem interface.
  \param file Path to file to load.
  \param pa Unused argument
  \result 0 on success, other values indicates errors
 */
TEMU_API int temu_loadElfImage(void *mem, const char *file, uint64_t pa);

/*!
  Loads an ELF file without going through filetype detection

  \param Mem Memory object, must conform to the mem interface.
  \param FileName Path to file to load.
  \param StartAddr Pointer where to place the start address of the elf
                   file
  \result 0 on success, 1 for successful load but without valid StartAddr,
    negative values indicate error.
 */

TEMU_API int temu_loadElfImageAndStartAddr(void *Mem, const char *FileName,
                                           uint64_t *StartAddr TEMU_NONNULL);

/*!
  Loads an ELF file without going through filetype detection.
  Unlike the normal ELF image loader, this uses the physical addresses
  in the ELF program headers to load the file.

  \param Mem Memory object, must conform to the mem interface.
  \param FileName Path to file to load.
  \param StartAddr Pointer where to place the start address of the elf
                   file
  \result 0 on success, 1 for successful load but without valid StartAddr,
    negative values indicate error.
 */

TEMU_API int temu_loadElfImagePhys(void *Mem, const char *FileName,
                                   uint64_t *StartAddr TEMU_NONNULL);

/*!
  Return non-zero if file is an ELF file

  \param file Path to file to check for ELFiness.
  \result 0 if file is not an ELF file, 1 if the file is an ELF file.
 */
TEMU_API int temu_imageIsELF(const char *file);

/*!
  Return non-zero if file has DWARF data

  \param file Path to file to check for DWARF-info.
  \result 0 if file does not contain DWARF, 1 if the file contains DWARF.
 */
TEMU_API int temu_elfHasDwarf(const char *file);

/*!
  Loads the symbol table from a given ELF file.

  The symbol table is allocated and the pointer of the allocated
  symbol table is placed in Sym. The user is responsible for
  disposing the symbol table with temu_disposeSymtab().

  \param FileName Name of ELF file
  \param Sym Address of your symtab pointer.
  \result 0 on success, other values indicates errors.
 */
TEMU_API int temu_loadSymtab(const char *FileName,
                             temu_Symtab **Sym TEMU_NONNULL);

/*!
  Get the function name associated with the given address

  The address can be any address within the range of the function,
  i.e. from the start to othe end of the function. Thus this serves as
  a way to find the name of the current function given a PC value.

  The function first searches for local symbols (e.g. static symbols
  in C). It then searches for global symbols and last for weak
  symbols.

  If a local symbol is found, the LocalFile pointer will point at a
  string with the file name it is associated to, otherwise the local
  file pointer will be set to null.

  The returned pointers are valid until the symbol table is disposed.

  \param Sym Symboltable

  \param LocalFile The pointer it refers to will be set to NULL if the
         symbol found is global.

  \param Symbol The pointer will be set to NULL if a function cannot
         be found at the address.

  \param Addr Virtual address to find a function name for.
*/
TEMU_API void temu_symtabGetFuncName(temu_Symtab *Sym TEMU_NONNULL,
                                     const char **LocalFile TEMU_NONNULL,
                                     const char **Symbol TEMU_NONNULL,
                                     uint64_t Addr);

/*!
  Get the object name associated with the given address

  The address can be any address within the range of the object. Thus
  this serves as a way to find the name of a global or static object
  given an address.

  The function first searches for local symbols (e.g. static symbols
  in C). It then searches for global symbols and last for weak
  symbols.

  If a local symbol is found, the LocalFile pointer will point at a
  string with the file name it is associated to, otherwise the local
  file pointer will be set to NULL.

  The returned pointers are valid until the symbol table is disposed.

  \param Sym Symboltable

  \param LocalFile The pointer it refers to will be set to NULL if the
         symbol found is global.

  \param Symbol The pointer will be set to NULL if an object cannot
         be found at the address.

  \param Addr Virtual address to find an object name for.
 */

TEMU_API void temu_symtabGetObjName(temu_Symtab *Sym TEMU_NONNULL,
                                    const char **LocalFile TEMU_NONNULL,
                                    const char **Symbol TEMU_NONNULL,
                                    uint64_t Addr);

/*!
  Get the range of a global function

  On success, the Addr and the Size references will be set to the starting
  virtual address of the function and the size in bytes respectively.

  \param Sym The Symbol table, from which the function is to be retrieved
  \param FuncName The function, whose range is required
  \param Addr On success, gives the address of the function
  \param Size On success, gives the size of the function start from Addr
  \result 0 on success (the function exists), other values indicate falures.
 */
TEMU_API int temu_symtabGetGlobalFuncRange(temu_Symtab *Sym TEMU_NONNULL,
                                           const char *FuncName TEMU_NONNULL,
                                           uint64_t *Addr TEMU_NONNULL,
                                           uint64_t *Size);

/*!
  Get the range of a local/static function

  On success, the Addr and the Size references will be set to the starting
  virtual address of the function and the size in bytes respectively.

  \param Sym The Symbol table, from which the function to be retrieved
  \param FileName The file/compilation unit that contains the function
  \param FuncName The function, whose range is required
  \param Addr On success, gives the address of the function
  \param Size On success, gives the size of the function start from Addr

  \result 0 on success (the function exists in the given file), other
          values indicate falures
 */
TEMU_API int temu_symtabGetLocalFuncRange(temu_Symtab *Sym TEMU_NONNULL,
                                          const char *FileName TEMU_NONNULL,
                                          const char *FuncName TEMU_NONNULL,
                                          uint64_t *Addr TEMU_NONNULL,
                                          uint64_t *Size TEMU_NONNULL);

/*!
  Get the range of a global object

  On success, the Addr and the Size references will be set to the starting
  virtual address of the object and the size in bytes respecivelly.

  \param Sym The Symbol table, from which the object to be retrieved
  \param ObjectName Name of the object, whose address range is to be obtained
  \param Addr On success, gives the address of the object
  \param Size On success, gives the size of the object start from Addr

  \result 0 on success (the object exists), other values indicate falures
 */
TEMU_API int temu_symtabGetGlobalObjRange(temu_Symtab *Sym TEMU_NONNULL,
                                          const char *ObjectName TEMU_NONNULL,
                                          uint64_t *Addr TEMU_NONNULL,
                                          uint64_t *Size TEMU_NONNULL);

/*!
  Get the range of a local/static object

  On success, the Addr and the Size references will be set to the starting
  virtual address of the object and the size in bytes respecivelly.

  \param Sym The Symbol table, from which the object to be retrieved
  \param FileName The file/compilation unit that contains the function
  \param ObjectName The object, whose range is required
  \param Addr On success, gives the address of the object
  \param Size On success, gives the size of the object start from Addr

  \result 0 on success (the object exists), other values indicate falures
 */
TEMU_API int temu_symtabGetLocalObjRange(temu_Symtab *Sym TEMU_NONNULL,
                                         const char *FileName TEMU_NONNULL,
                                         const char *ObjectName TEMU_NONNULL,
                                         uint64_t *Addr TEMU_NONNULL,
                                         uint64_t *Size TEMU_NONNULL);

/*!
  Dispose (deallocate) the symbol table
  \param Sym Symbol table pointer to free.
 */
TEMU_API void temu_disposeSymtab(temu_Symtab *Sym);

/*!
  Loads an SREC file without going through filetype detection.

  \param mem Memory object, must conform to the mem interface.
  \param file Path to file to load.
  \param pa Unused argument
  \result 0 on success, other values indicates errors
 */
TEMU_API int temu_loadSrecImage(void *mem, const char *file, uint64_t pa);

/*!
  Loads an SREC file without going through filetype detection.

  \param mem Memory object, must conform to the mem interface.
  \param fileName Path to file to load.
  \param startAddr Start address, will be set if result is 0
  \result 0 on success, 1 for successful load but without valid StartAddr,
    negative values indicate error.
 */
TEMU_API int temu_loadSrecImageAndStartAddr(void *mem, const char *fileName,
                                            uint32_t *startAddr TEMU_NONNULL);

typedef struct {
  uint64_t Address;
  int64_t Size;
} temu_SrecRegion;

/*!
  Loads an SREC file without going through filetype detection.

  \param mem Memory object, must conform to the mem interface.
  \param file Path to file to load.
  \param regions Regins to write to file, end region should be {x,-1}
  \result 0 on success, other values indicates errors
 */
TEMU_API int temu_storeSrecImage(void *mem, const char *file,
                                 temu_SrecRegion regions[]);

#ifdef __cplusplus
}
#endif

#endif // ! TEMU_LOADER_H
