//===-- temu-c/Notifications.h - Untimed Event API --------------*- C++ -*-===//
//
// TEMU: The Terma Emulator
// (c) Terma 2015
// Authors: Mattias Holm <maho (at) terma.com>
//
//===----------------------------------------------------------------------===//

#ifndef TEMU_NOTIFICATIONS_H
#define TEMU_NOTIFICATIONS_H

#include <stdint.h>
#include "temu-c/Support/Attributes.h"
#include "temu-c/Support/Objsys.h"

// The following interfaces are for non-timed events (or
// notifications). Notification 0 is reserved as a special
// no-notification handler connected id. Notifications have a name and
// are associated with an object. The subscriber can request a
// notification with a null object (i.e. will accept all identical
// notifications, or a given object, which means the notification
// handler will be called only for some objects.  There is currently
// no way to depublish a notification, but the standard approach is to
// set the notification id to 0.
#ifdef __cplusplus
extern "C" {
#endif

typedef void (*temu_NotificationHandler)(void *Arg, temu_Object_ *Source, void *NotInfo);

/*! Publish a notification source
 * A notification source is identified by an event name and an object pointer
 * \param NotName Name of the notification
 * \param Obj Pointer to the object
 * \result Notification ID of the published event.
 */
TEMU_API int64_t temu_publishNotification(const char *NotName, temu_Object_ *Obj);

/*! Install notification functions for the given event generated by source
 *
 * \param NotName Name of the notification
 * \param Source If source is NULL, the event subscriber will be
 *               notified by all the sources for the given name.
 * \param Arg Context argument to be passed to the callback notification function
 * \param NotFunc The callback function on notification
 */
TEMU_API int temu_subscribeNotification(const char *NotName, temu_Object_ *Source, void *Arg,
                               temu_NotificationHandler NotFunc);

/*! Remove notification handler for the given name and source.
 * Note that this function runs in O(N) time. It is not meant for
 * being used in performance critical code.
 * \param NotName Name of the notification
 * \param Source If source is NULL, the event subscriber will be
 *               notified by all the sources for the given name.
 * \param NotFunc The callback function to be called on notification
 */
TEMU_API int temu_unsubscribeNotification(const char *NotName, temu_Object_ *Source,
                                 temu_NotificationHandler NotFunc);

/*! Remove notification handler for the given name, source and arg.
 * Note that this function runs in O(N) time. It is not meant for
 * being used in performance critical code.
 * \param NotName Name of the notification
 * \param Source If source is NULL, the event subscriber will be
 *               notified by all the sources for the given name.
 * \param NotFunc The callback function to be called on notification
 * \param Arg Context argument to be passed to the callback notification function
 * \result zero on success, otherwise non-zero
 */
TEMU_API int temu_unsubscribeNotificationArg(const char *NotName, temu_Object_ *Source,
                                             temu_NotificationHandler NotFunc,
                                             void *Arg);


/*! Call event subscriber, EvInfo is a per event specific struct
 * the event handler must cast this to the appropriate type.
 * \param Id Notification ID
 * \param NotInfo Ponter to pass to notification handlers
 */
TEMU_API void temu_notify(int64_t Id, void *NotInfo);

/*! Quick inline call macro that do not call the function for an invalid
 * event id. This saves the function call to the notify
 * function, minimising the cost for unset event handlers.
 * \param Id Notification ID
 * \param NotInfo Pointer to pass to notification handlers
 */
static inline void
temu_notifyFast(int64_t *Id, void *NotInfo)
{
  if (*Id) {
    temu_notify(*Id, NotInfo);
  }
}

#ifdef __cplusplus
}
#endif

#endif /*  ! TEMU_EVENTS_H */
