//===-- temu-c/Register.h - TEMU Register Introspection ---------*- C++ -*-===//
//
// TEMU: The Terma Emulator
// (c) Terma 2018-2019
// Authors: Mattias Holm <maho (at) terma.com>
//
//===----------------------------------------------------------------------===//

#ifndef TEMU_REGISTER_H
#define TEMU_REGISTER_H
#include "temu-c/Memory/Memory.h"
#include "temu-c/Support/Attributes.h"
#include "temu-c/Support/Objsys.h"
#include <stdint.h>
#include <stdlib.h>

/*
 * EXPERIMENTAL API
 * API Changes:
 *   2019-07-09: Rework introspection API, no more temu_Field objects returned.
 *               Add index argument to set/get/read/write.
 *   2018-06-11: Remove wr and rd arguments from temu_addField
 *               Replace the separate flag params with a flag bit field.
 */

// Registers are normal properties suitable for device registers (must
// be of an integer type), but have addtitional meta-data attached to
// them. Primarily you can associate fields with a register, a
// register field may even have individual accessors which can be
// invoked to set single bits.

#ifdef __cplusplus
extern "C" {
#endif

#define TEMU_FIELD_WR (1 << 0)         // Writeable field
#define TEMU_FIELD_WARM_RESET (1 << 1) // Field is subject to warm reset

typedef struct {
  const char *Name;
  const char *Desc;
  uint64_t Mask;
  uint64_t ResetValue;
  unsigned Writable : 1;
  unsigned Resettable : 1;
  unsigned ClearOnSet : 1;
} temu_FieldInfo;

#define TEMU_FIELD_INFO_NULL ((temu_FieldInfo){NULL, NULL, 0, 0})

typedef struct {
  const char *Name;
  const char *Desc;
  size_t NumRegs;

  unsigned DeviceOffset;
  unsigned Stride;
  uint64_t ColdResetValue;

  uint64_t WarmResetMask;
  uint64_t WarmResetValue;

  uint64_t ReadMask;
  uint64_t WriteMask;

  size_t NumFields;
  temu_FieldInfo *Field;
} temu_RegisterInfo;

typedef struct {
  const char *Name;
  const char *Desc;
  size_t NumRegisters;
  const char **RegNames;
} temu_RegisterBankInfo;

typedef struct {
  size_t NumRegisterBanks;
  const char **BankNames;
} temu_ModelRegInfo;

typedef struct temu_RegisterBank temu_RegisterBank;
typedef struct temu_Register temu_Register;
typedef struct temu_Field temu_Field;

/*!
 * Adds a register bank to a TEMU class
 * \param C Pointer to the TEMU class
 * \param Name Name of the register
 * \param MemAccessIface Pointer to the register memory access interface
 * \result Pointer to the new register bank
 */
TEMU_API temu_RegisterBank *
temu_addRegisterBank(temu_Class *C, const char *Name,
                     temu_MemAccessIface *MemAccessIface);

/*!
 * Retrieves the register bank name from its
 * object \param Bank Pointer to the register bank name \result Bank name as a
 * C-string
 */
TEMU_API const char *temu_getRegisterBankName(temu_RegisterBank *Bank);

/*!
 * Add register property to class
 *
 * Adds a register with the given name to a class. It returns a reference
 * to a meta register which can be used to add fields.
 *
 * \param Bank Register bank to add register to
 * \param Name Name of register (must be a valid C-identifier)
 * \param Offset Offset to storage element in the device struct.
 * \param Typ Type of register, note that registers are limited to
 *        unsigned integer types with fixed width.
 * \param Count Number of registers (normally 1)
 * \param Wr Register write function
 * \param Rd Register read function
 * \param Doc Documentation string for register.
 *
 * \param DeviceOffset Offset of register in memory system. This is
 *        the same offset that is used in the memory transaction
 *        interface.
 *
 * \param Stride In case the register is an array of registers, then
 *        the stride is used for the offset in physical address space
 *        between each register.
 * \result A reference to a meta register which can be used to add fields.
 */
TEMU_API temu_Register *
temu_addRegister(temu_RegisterBank *Bank, const char *Name, int Offset,
                 temu_Type Typ, int Count, temu_PropWriter Wr,
                 temu_PropReader Rd, const char *Doc,
                 // Offset of register from device mapping (or bank mapping)
                 // in memory space
                 uint32_t DeviceOffset, uint32_t Stride);

TEMU_API temu_Register *temu_addRegisterWithFields(
    temu_RegisterBank *Bank, const char *Name, int Offset, temu_Type Typ,
    int Count, temu_PropWriter Wr, temu_PropReader Rd, const char *Doc,
    // Offset of register from device mapping (or
    // bank mapping) in memory space
    uint32_t DeviceOffset, uint32_t Stride, temu_FieldInfo Fields[]);

/*!
 * Add field to meta register
 *
 * Adds a field to the meta register.
 *
 * \param R Meta register pointer.
 * \param Name Name of of field (must be a C-compatible identifier)
 * \param Mask Mask identifying the bits in the register corresponding
 *        to the field. Mask must contain consecutive bits only.
 * \param Reset Reset value.
 * \param Flags Flags used: TEMU_FIELD_WR means field is writable,
 *        TEMU_FIELD_WARM_RESET means the field is subject to reset actions
 *        also on warm resets.
 * \param Doc Documentation string
 */

TEMU_API void temu_addField(temu_Register *R, const char *Name, uint64_t Mask,
                            uint64_t Reset, uint64_t Flags, const char *Doc);

/*!
 * Get a register bank from a class by name
 * \param C Pointer to the TEMU class
 * \param Name Name of the register bank
 * \result Pointer to the register bank
 */
TEMU_API temu_RegisterBank *temu_getRegisterBank(temu_Class *C,
                                                 const char *Name);

/*!
 * Get list of class register banks
 * The returned object is thread local. It will be destroyed on the next call in
 * the current thread.
 *
 * \param C The class object.
 * \result Thread local mobel reg info object with a list of all register banks.
 */
TEMU_API const temu_ModelRegInfo *temu_getModelRegisterBankInfo(temu_Class *C);

/*!
 * Get register info for named register
 * The returned object is thread local. It will be destroyed on the next call in
 * the current thread.
 *
 * \param C The class object.
 * \param RegName Register name.
 * \result Thread local register info object.
 */

TEMU_API const temu_RegisterInfo *temu_getRegisterInfo(temu_Class *C,
                                                       const char *RegName);

TEMU_API const temu_RegisterBankInfo *
temu_getRegisterBankInfo(temu_Class *C, const char *RegBankName);

/*!
 * Get a register from a register bank by name
 * \param Bank Pointer to the register bank object
 * \param Name Name of the register
 * \result Pointer to the register object
 */
TEMU_API temu_Register *temu_getRegister(temu_RegisterBank *Bank,
                                         const char *Name);

/*!
 * Retrieve register name from pointer
 * \param R Pointer to the register
 * \result The name of the register
 */
TEMU_API const char *temu_getRegisterName(temu_Register *R);

/*!
 * Get the documentation of a register as a string
 * \param R Pointer to the register
 * \result The documentation of the register as a C-string
 */
TEMU_API const char *temu_getRegisterDocs(temu_Register *R);

/*!
 * Retrieve the value of a field in a register
 * \param Obj The object of the class the has the register
 * \param RegName Register name
 * \param RegIdx Register index
 * \param FieldName Field name
 * \return The value of the field
 */

TEMU_API uint64_t temu_getFieldValue(temu_Object *Obj, const char *RegName,
                                     unsigned RegIdx, const char *FieldName);

/*!
 * Retrieve the value of a field in a register with
 *        side-effects.
 * \param Obj The object of the class the has the register
 * \param RegName Register name
 * \param RegIdx Register index
 * \param FieldName Field name
 * \return The value of the field
 */
TEMU_API uint64_t temu_readFieldValue(temu_Object *Obj, const char *RegName,
                                      unsigned RegIdx, const char *FieldName);

/*!
 * Set a field's value in a register
 * \param Obj The object that contains the register
 * \param RegName Register name
 * \param RegIdx Register index
 * \param FieldName Field name
 * \param Value The value to be set
 * \result zero on success, otherwise non-zero value
 */

TEMU_API int temu_setFieldValue(temu_Object *Obj, const char *RegName,
                                unsigned RegIdx, const char *FieldName,
                                uint64_t Value);

/*!
 * Set a field's value in a register with side-effects
 * \param Obj The object that contains the register
 * \param RegName Register name
 * \param RegIdx Register index
 * \param FieldName Field name
 * \param Value The value to be set
 * \result zero on success, otherwise non-zero value
 */

TEMU_API int temu_writeFieldValue(temu_Object *Obj, const char *RegName,
                                  unsigned RegIdx, const char *FieldName,
                                  uint64_t Value);

/*!
 * Returns the cold reset value of the register (computed from the field info
 * data). Cold reset is also known as a hard reset, and implies that the power
 * has been off for some time.
 *
 * \param Reg Pointer to the register
 * \return The cold reset value
 */
TEMU_API uint64_t temu_getRegisterColdResetValue(temu_Register *Reg);

/*!
 * Returns the warm reset value of the register
 * \param Reg Pointer to the register
 * \return The warm reset value
 */
TEMU_API uint64_t temu_getRegisterWarmResetValue(temu_Register *Reg);

/*!
 * Get the current write mask of a register
 * \param Reg Pointer to the register
 * \return The value of the mask
 */
TEMU_API uint64_t temu_getRegisterWriteMask(temu_Register *Reg);

/*!
 * Get the current read mask of a register
 * \param Reg Pointer to the register
 * \return The value of the mask
 */
TEMU_API uint64_t temu_getRegisterReadMask(temu_Register *Reg);

#ifdef __cplusplus
} // extern c
#endif

#include "temu-c/Support/NewRegisterAPI.h"

#endif /* ! TEMU_REGISTER_H */
