//===-------------------------------------------------------*- TableGen -*-===//
//
// TEMU: The Terma Emulator
// (c) Terma 2022
// Authors: Mattias Holm <maho(at)terma.com>
//
//===----------------------------------------------------------------------===//

#ifndef TEMU_CODE_PATTERNS_H
#define TEMU_CODE_PATTERNS_H
#include "temu-c/Support/Objsys.h"
#include <stdint.h>

// NOTICE: UNSTABLE API
//   This API is in early form and is seen as UNSTABLE.
//   Unstable means it can change without notice
//   (even if the rest of the system follows proper semantic versioning).
//   In most cases, UNSTABLE APIs are still kept properly versioned,
//   but there is no firm guarantee for this.
//
//   The API is used internally for all IDLE mode patterns subject to detection.
typedef enum {
  tePA_Idle,        //!< Enter idle when running pattern
  tePA_Skip,        //!< Skip over code pattern (currently not supported)
  tePA_Call,        //!< Call function but run code after function returns (currently not supported)
  tePA_Invalid,     //!< Invalid action, do not use! Internal API use only.
} temu_PatternAction;

typedef struct {
  uint32_t Instruction; //!< Instruction value
  uint32_t Mask;        //!< Instruction mask (use to ignore register numbers etc)
} temu_CodePatternEntry;

/*!
 * Code pattern structure
 *
 * The code pattern structure exists to make it possible to define a static list of code patterns.
 * The patterns should be installed on a processor using the code pattern interface.
 * 
 * The code pattern can be used to attach custom actions,
 * either at a specific physical address
 * or when a sequence of instructions have been matched.
 * 
 * TEMU uses this mechanism internally to detect idle-loops.
 * 
 * A pattern is matched if the physical address of the start matches,
 * and all instructions in the pattern match.
 * Zero length patterns are legal if the physical address mask is non-zero.
 * Normally, skip length and pattern length should be the same.
 * 
 * It is possible to skip backwards.
 * It is also possible to skip more or less than the matched instructions.
 *
 * When the pattern is installed, it is injected into a trie.
 * The injected structure is after that no longer used and can be disposed.
 * 
 * Patterns are matched at decode time, for both interpreter
 * and binary translator.
 * 
 * For the interpreter, a pseudo instruction will be inserted to carry out the action.
 * 
 * In the case of processors with branch delay slots (e.g. SPARC),
 * the pattern is only triggered if the nPC follows the PC.
 * 
 * Pattern entries match individual instruction chunks,
 * i.e. for Thumb2 an entry matches a 16-bit part,
 * meaning two entries will be needed for a 32-bit instruction.
 * 
 * Note that processors are expected to implement one pattern interface per instruction set.
 * 
 * Patterns are not matched if they cross page boundaries.
 */
typedef struct {
  uint64_t PhysicalAddress;     //!< Physical address matched in the decoder (currently ignored)
  uint64_t PhysicalAddressMask; //!< Physical address mask matched in the 
                                //!< decoder, set to 0 to ignore address (currently ignored)
  temu_PatternAction Action;    //!< Action to take on this match

  //! Function to call for actions tePA_Call and tePA_CallAndSkip, first
  //! argument is the processor pointer (currently ignored)
  void (*Callback)(void *, void *);
  void *CallbackData;     //!< Data passed in second parameter to callback (currently ignored)
  int Parameter;          //!< Number of instructions to skip (in case of tePA_Skip), or tag for idle
  unsigned PatternLength; //!< Number of pattern entries
  const temu_CodePatternEntry *Pattern; // Pattern to match
} temu_CodePattern;

typedef struct {
  void (*installPattern)(void *obj, temu_CodePattern *pattern);
} temu_CodePatternIface;
#define TEMU_CODE_PATTERN_IFACE_TYPE "temu::CodePatternIface"
TEMU_IFACE_REFERENCE_TYPE(temu_CodePattern);

#endif // !TEMU_CODE_PATTERNS_H
