//===------------------------------------------------------------*- C++ -*-===//
//
// TEMU: The Terma Emulator
// (c) Terma 2015
// Authors: Mattias Holm <maho (at) terma.com>
//
//===----------------------------------------------------------------------===//

#ifndef TEMU_BUS_AMBA_H
#define TEMU_BUS_AMBA_H
/*!
 * AMBA Plug-and-play emulation support.
 *
 * This file define structures, constants and interfaces that should
 * be used in models which need to implement the AHB and APB
 * plug-and-play information.
 */

#include "temu-c/Support/Objsys.h"
#include <stdint.h>

#ifdef __cplusplus
extern "C" {
#endif


// Vendor constants
static const uint32_t teAMBA_VendorGaisler = 0x01;
static const uint32_t teAMBA_VendorPender =  0x02;
static const uint32_t teAMBA_VendorEsa =  0x04;
static const uint32_t teAMBA_VendorAstrium =  0x06;
static const uint32_t teAMBA_VendorOpenChip =  0x07;
static const uint32_t teAMBA_VendorOpenCores =  0x08;
static const uint32_t teAMBA_VendorVarious =  0x09;
static const uint32_t teAMBA_VendorDlr =  0x0a;
static const uint32_t teAMBA_VendorEonic =  0x0b;
static const uint32_t teAMBA_VendorParisTech =  0x0c;
static const uint32_t teAMBA_VendorRadionor =  0x0f;
static const uint32_t teAMBA_VendorGleichmann =  0x10;
static const uint32_t teAMBA_VendorMenta =  0x11;
static const uint32_t teAMBA_VendorSun =  0x13; // Oracle ?
static const uint32_t teAMBA_VendorMovidia =  0x14;
static const uint32_t teAMBA_VendorOrbita =  0x17;
static const uint32_t teAMBA_VendorSiemens =  0x1a;
static const uint32_t teAMBA_VendorActel =  0xac;
static const uint32_t teAMBA_VendorCaltech =  0xca;
static const uint32_t teAMBA_VendorEmbeddit =  0xea;


//! AHB bus plug and play record
typedef struct {
  uint32_t IdentReg;
  uint32_t UserDef[3];
  uint32_t Bar[4];
} temu_AhbPnpInfo;

//! AHB bus plug and play interface
//!
//! A device providing plug and play info for the AHB bridge, should
//! implement this interface. The recommended way is to put a
//! temu_AhbPnpInfo struct inside your model struct and then return a
//! pointer to this one. This way the PNP info can be changed on a per
//! object basis.
typedef struct temu_AhbIface {
  temu_AhbPnpInfo* (*getAhbPnp)(void *Obj);
} temu_AhbIface;
#define TEMU_AHB_IFACE_TYPE "AhbIface"
TEMU_IFACE_REFERENCE_TYPE(temu_Ahb);


//! APB bus plug and play record
typedef struct {
  uint32_t ConfigWord;
  uint32_t Bar;
} temu_ApbPnpInfo;


//! APB bus plug and play interface
//!
//! A device providing plug and play info for the APB bridge, should
//! implement this interface. The recommended way is to put a
//! temu_ApbPnpInfo struct inside your model struct and then return a
//! pointer to this one. This way the PNP info can be changed on a per
//! object basis.
typedef struct temu_ApbIface {
  temu_ApbPnpInfo* (*getApbPnp)(void *Obj);
} temu_ApbIface;
#define TEMU_APB_IFACE_TYPE "ApbIface"
TEMU_IFACE_REFERENCE_TYPE(temu_Apb);



static inline void
temu_ahbSetVendor(temu_AhbPnpInfo *Ahb, uint32_t Vendor)
{
  Ahb->IdentReg = (Ahb->IdentReg & 0x00ffffff) | (Vendor << 24);
}


static inline void
temu_ahbSetDeviceId(temu_AhbPnpInfo *Ahb, uint32_t Device)
{
  Ahb->IdentReg = (Ahb->IdentReg & 0xff000fff) | (Device << 12);
}

static inline void
temu_ahbSetVersion(temu_AhbPnpInfo *Ahb, uint32_t Version)
{
  Ahb->IdentReg = (Ahb->IdentReg & 0xfffffc1f) | (Version << 5);
}

static inline void
temu_ahbSetIrq(temu_AhbPnpInfo *Ahb, uint32_t Irq)
{
  Ahb->IdentReg = (Ahb->IdentReg & 0xffffffe0) | Irq;
}

static inline uint32_t
temu_ahbGetIrq(temu_AhbPnpInfo *Ahb)
{
  return Ahb->IdentReg & 0x1f;
}

static inline void
temu_ahbSetUserDef(temu_AhbPnpInfo *Ahb, int Id, uint32_t User)
{
  Ahb->UserDef[Id] = User;
}

static inline void
temu_ahbSetAddr(temu_AhbPnpInfo *Ahb, int Id, uint32_t Addr)
{
  Ahb->Bar[Id] = (Ahb->Bar[Id] & 0x000fffff) | (Addr & 0xfff00000);
}

static inline void
temu_ahbSetCP(temu_AhbPnpInfo *Ahb, int Id, uint32_t CP)
{
  Ahb->Bar[Id] = (Ahb->Bar[Id] & 0xfff0ffff) | (CP << 16);
}

static inline void
temu_ahbSetMask(temu_AhbPnpInfo *Ahb, int Id, uint32_t Mask)
{
  Ahb->Bar[Id] = (Ahb->Bar[Id] & 0xffff000f) | (Mask << 4);
}

static inline void
temu_ahbSetType(temu_AhbPnpInfo *Ahb, int Id, uint32_t Type)
{
  Ahb->Bar[Id] = (Ahb->Bar[Id] & 0xfffffff0) | Type;
}



static inline void
temu_apbSetVendorId(temu_ApbPnpInfo *Apb, uint32_t Vendor)
{
  Apb->ConfigWord = (Apb->ConfigWord & 0x00ffffff) | (Vendor << 24);
}

static inline uint32_t
temu_apbGetVendorId(temu_ApbPnpInfo *Apb)
{
  return (Apb->ConfigWord >> 24) & 0xff;
}

static inline void
temu_apbSetDeviceId(temu_ApbPnpInfo *Apb, uint32_t Device)
{
  Apb->ConfigWord = (Apb->ConfigWord & 0xff000fff) | (Device << 12);
}

static inline uint32_t
temu_apbGetDeviceId(temu_ApbPnpInfo *Apb)
{
  return (Apb->ConfigWord >> 12) & 0xfff;
}


static inline void
temu_apbSetVersion(temu_ApbPnpInfo *Apb, uint32_t Version)
{
  Apb->ConfigWord = (Apb->ConfigWord & 0xfffffc1f) | (Version << 5);
}

static inline uint32_t
temu_apbGetVersion(temu_ApbPnpInfo *Apb)
{
  return (Apb->ConfigWord >> 5) & 0x1f;
}


static inline void
temu_apbSetIrq(temu_ApbPnpInfo *Apb, uint32_t Irq)
{
  Apb->ConfigWord = (Apb->ConfigWord & 0xffffffe0) | Irq;
}

static inline uint32_t
temu_apbGetIrq(temu_ApbPnpInfo *Apb)
{
  return Apb->ConfigWord & 0x1f;
}

static inline void
temu_apbSetAddr(temu_ApbPnpInfo *Apb, uint32_t Addr)
{
  Apb->Bar = (Apb->Bar & 0x000fffff) | ((Addr & 0x000fff00) << 12);
}

static inline void
temu_apbSetCP(temu_ApbPnpInfo *Apb, uint32_t CP)
{
  Apb->Bar = (Apb->Bar & 0xfff0ffff) | (CP << 16);
}

static inline void
temu_apbSetMask(temu_ApbPnpInfo *Apb, uint32_t Mask)
{
  Apb->Bar = (Apb->Bar & 0xffff000f) | (Mask << 4);
}

static inline void
temu_apbSetType(temu_ApbPnpInfo *Apb, uint32_t Type)
{
  Apb->Bar = (Apb->Bar & 0xfffffff0) | Type;
}

#ifdef __cplusplus
}
#endif

#endif /* ! TEMU_BUS_SERIAL_H */
