//===-- temu-c/Debugger.h - TEMU Debugger API -------------------*- C++ -*-===//
//
// TEMU: The Terma Emulator
// (c) Terma 2023
// Authors: Mattias Holm <maho (at) terma.com>
//
//===----------------------------------------------------------------------===//

#ifndef TEMU_DEBUGGER_H
#define TEMU_DEBUGGER_H

#include "temu-c/Support/Objsys.h"
#include <stdint.h>

#ifdef __cplusplus
extern "C" {
#endif

typedef struct temu_TimeSource temu_TimeSource;
typedef struct temu_Object temu_Object;
typedef struct temu_MemTransaction temu_MemTransaction;
/*
 * Debugger functions and interfaces
 *
 * These functions and interfaces define an interface that can be used to
 * attach debugger servers for e.g. the GDB RSP protocol.
 *
 * The functions and interfaces should be seen as unstable.
 * Note that that means the API may change, not that they are broken.
 *
 * These interfaces are intended to be used to interface debuggers
 * with the scheduler.
 */

/*!
 * Return current processor number
 *
 * Returns -1 when running, or stopped CPU number after breakpoint hit.
 */
int temu_debuggerGetCurrentProcessor();

/*!
 * Stop the scheduler from a debugger
 *
 * This function is intended to be used by the GDB server.
 * It stops the scheduler as soon as possible, without returning
 * from the temu_run*() functions.
 *
 * The function is blocking until the core has been stopped.
 * If the core is currently not running, this function waits
 * until someone calls temu_run*().
 */
void temu_debuggerStop();

/*!
 * Resume the scheduler from a debugger
 *
 * Resumes the scheduler after a stop or breakpoint.
 * Note the function returns immediately.
 * The debugger server is expected to listen for events
 * through the debugger interface instead.
 */
void temu_debuggerCont();

/*!
 * Step the given processor.
 *
 * The debugger must be attached, and the function will block until
 * the scheduler is running using temu_run*() functions.
 */
void temu_debuggerStep(temu_TimeSource *cpu, int steps);

typedef enum {
  teBS_Ignore,
  teBS_Handle,
} temu_BreakState;

typedef enum {
  teBA_Stop,
  teBA_Resume,
} temu_BreakAction;

// The breakpoint functions here are called as:
//
// While running, the hit function is called, this function may be called
// in parallel from different processors.
// The handler should return the action to be taken.
// E.g. if we want to stop at this location, or not.
// Subsequently, if the breakpoint results in in a teBS_Handle,
// the corresponding handle function will be called in a thread safe manner.
// Note that in the case of parallel execution, multiple breakpoint may be
// "hit". before handling them.
// The interface is seen as unstable since debugging support is evolving.

typedef struct {
  temu_BreakState (*breakpointHit)(void *obj, temu_TimeSource *cpu, int cpuId,
                                   temu_MemTransaction *mt);
  temu_BreakState (*watchpointReadHit)(void *obj, temu_TimeSource *cpu,
                                       int cpuId, temu_MemTransaction *mt);
  temu_BreakState (*watchpointWriteHit)(void *obj, temu_TimeSource *cpu,
                                        int cpuId, temu_MemTransaction *mt);

  temu_BreakState (*ioReadHit)(void *obj, temu_TimeSource *cpu, int cpuId,
                               temu_Object *ioDevice, temu_MemTransaction *mt);
  temu_BreakState (*ioWriteHit)(void *obj, temu_TimeSource *cpu, int cpuId,
                                temu_Object *ioDevice, temu_MemTransaction *mt);

  temu_BreakAction (*handleBreakpoint)(void *obj, temu_TimeSource *cpu,
                                       int cpuId, temu_MemTransaction *mt);
  temu_BreakAction (*handleReadWatchpoint)(void *obj, temu_TimeSource *cpu,
                                           int cpuId, temu_MemTransaction *mt);
  temu_BreakAction (*handleWriteWatchpoint)(void *obj, temu_TimeSource *cpu,
                                            int cpuId, temu_MemTransaction *mt);

  temu_BreakAction (*handleIoRead)(void *obj, temu_TimeSource *cpu, int cpuId,
                                   temu_Object *ioDevice,
                                   temu_MemTransaction *mt);
  temu_BreakAction (*handleIoWrite)(void *obj, temu_TimeSource *cpu, int cpuId,
                                    temu_Object *ioDevice,
                                    temu_MemTransaction *mt);

  void (*stopped)(void *obj);
  void (*stepped)(void *obj, temu_TimeSource *cpu, int cpuId);
} temu_DebuggerIface;
#define TEMU_DEBUGGER_IFACE_TYPE "temu::DebuggerIface"
TEMU_IFACE_REFERENCE_TYPE(temu_Debugger)

typedef struct {
  void (*attachDebugger)(void *obj, void *debugger, temu_DebuggerIface *debugIface);
  void (*detachDebugger)(void *obj);
} temu_DebuggerAttachIface;
#define TEMU_DEBUGGER_ATTACH_IFACE_TYPE "temu::DebuggerAttachIface"
TEMU_IFACE_REFERENCE_TYPE(temu_DebuggerAttach)

/*!
 * Attach a debugger to the current scheduler
 *
 * This attaches a debugger to the scheduler,
 * ensuring it is notified about events such as breakpoints and watchpoints.
 */
void temu_debuggerAttach(void *debugger, temu_DebuggerIface *debuggerIface);

/*!
 * Notify debugger infrastructure that a remote connection has been made
 */
void temu_debuggerConnected();

/*!
 * Detach the debugger from the current scheduler
 *
 * This attaches a debugger to the scheduler,
 * ensuring it is notified about events such as breakpoints and watchpoints.
 */

void temu_debuggerDetach(void *debugger);

#ifdef __cplusplus
}
#endif

#endif // !TEMU_DEBUGGER_H