#include "temu-c/Support/Init.h"
#include "temu-c/Support/Objsys.h"
#include "temu-c/Memory/Memory.h"
#include "temu-c/Target/Cpu.h"
#include "temu-c/Support/Loader.h"

#include <stdio.h>

int
main(int argc, const char *argv[argc])
{
  temu_CreateArg Args = TEMU_NULL_ARG;

  // Init support library, this will check your license
  // the program will terminate if you do not have a valid license
  temu_initSupportLib();

  // Look up the temu command and setup the plugin paths based on
  // its location.
  temu_initPathSupport("temu");

  // Load the plugins needed
  temu_loadPlugin("libTEMULeon2.so");
  temu_loadPlugin("libTEMULeon2SoC.so");
  temu_loadPlugin("libTEMUMemory.so");
  temu_loadPlugin("libTEMUConsole.so");

  // Create needed objects, no arguments are given (see init above)
  void *Cpu = temu_createObject("Leon2", "cpu0", &Args);
  void *L2SoC = temu_createObject("Leon2SoC", "leon2soc0", &Args);
  void *MemSpace = temu_createObject("MemorySpace", "mem0", &Args);
  void *Rom = temu_createObject("Rom", "rom0", &Args);
  void *Ram = temu_createObject("Ram", "ram0", &Args);
  void *Console = temu_createObject("Console", "tty0", &Args);

  // Allocate space for ROM and RAM
  temu_writeValueU64(Rom, "size", 0x80000, 0);
  temu_writeValueU64(Ram, "size", 0x80000, 0);

  // Map in ROM, RAM and the IO modules in the memory space
  temu_memoryMap(MemSpace, 0x00000000, 0x80000, Rom, 0);
  temu_memoryMap(MemSpace, 0x40000000, 0x80000, Ram, 0);
  temu_memoryMap(MemSpace, 0x80000000, 0x100, L2SoC, 0);

  // For the L2 (without MMU) we connect memAccess directly to the
  // memspace, for MMU systems, we will need to connect memAccessL2
  // instead, and to connect memAccess to the CPU memory access
  // interface. See /opt/temu/x/share/temu/sysconfig dir for exmples.
  temu_connect(Cpu, "memAccess", MemSpace, "MemAccessIface");
  temu_connect(Cpu, "memory", MemSpace, "MemoryIface");
  temu_connect(MemSpace, "invalidaccess", Cpu,
               "InvalidMemAccessIface");

  // In TEMU 2.2, we do not need to connect the reverse link
  // this has been automated through the "ports" mechanism.
  temu_connect(Cpu, "irqClient", L2SoC, "IrqClientIface");

  // Attach the L2SoC to its time source.
  temu_setTimeSource(L2SoC, Cpu);

  // Add Device to CPU device array, this is used to distribute
  // CPU resets to device models.
  temu_connect(Cpu, "devices", L2SoC, "DeviceIface");

  // The console implements the serial interface and simply
  // redirects it to stdout. For a GUI console use the ConsoleUI
  // class instead
  temu_connect(Console, "serial", L2SoC, "UartAIface");
  temu_setTimeSource(Console, Cpu);

  // Check sanity of the object graph, pass non-zero to enable
  // automatic printouts (with info on which objects are not
  // sane). 0 means the function is silent, and we only care
  // about the result. Note, this is a debugging help, some
  // interfaces do not need to be connected (e.g. the LEON2
  // cache interfaces).
  if (temu_checkSanity(1)) {
    fprintf(stderr, "Sanity check failed\n");
  }

  // Can pass CPU or MemorySpace, which you pass doesn't matter
  // loadImage handles both SREC and ELF files.
  temu_loadImage(Cpu, "obsw.elf");

  // To get the CPU interface to run the CPU directly, we query
  // for the interface. The CpuIface implements the basic CPU
  // control functionality like RESET
  temu_CpuIface *CpuIf = temu_getInterface(Cpu, "CpuIface", 0);

  CpuIf->reset(Cpu, 0); // Cold-reset, 1 is a warm reset
  CpuIf->setPc(Cpu, 0x40000000);        // Starting location

  // Fake low level boot software setting up the stack pointers
  CpuIf->setGpr(Cpu, 24+6, 0x40050000); // %i6 or %fp
  CpuIf->setGpr(Cpu, 8+6, 0x40050000);  // %o6 or %sp

  // You can step or run the CPU. Running runs for N cycles
  // while stepping executes the given number of instructions
  // as an instruction can take longer than a cycle, these are
  // not the same. For multi-core systems, you will not run or
  // step the CPU but rather a machine object, which will ensure
  // that all of the CPUs advance as requested. Also a CPU in
  // idle or powerdown mode does not advance any steps, but only
  // cycles.
  CpuIf->run(Cpu, 1000000); // Run 1 M cycles
  //  CpuIf->step(Cpu, 1000000); // Step 1 M steps
  //  CpuIf->runUntil(Cpu, 1000000); // Run until absolute time is
                                     // 1000000 cycles

  // Step 10 instructions, but return early if until absolute time
  // reaches 1000000 cycles
  //  CpuIf->stepUntil(Cpu, 10, 1000000);
  return 0;
}
