//===--------------------------------------------------------------*- C -*-===//
//
// TEMU: The Terma Emulator
// (c) Terma 2018, 2019
// Authors: Mattias Holm <maho (at) terma.com>
//
//===----------------------------------------------------------------------===//

#ifndef TEMU_INSTRUMENTER_H
#define TEMU_INSTRUMENTER_H
#include "temu-c/Support/Objsys.h"
#include <stdint.h>

/*!
  Instruction classification flags
 */
typedef enum {
  teIF_Branch = 1 << 0, //!< Instruction is a branch
  teIF_IndirectBranch = 1 << 1, //!< Instruction is an indirect branch
  teIF_Load = 1 << 2, //!< Instruction is a load
  teIF_Store = 1 << 3, //!< Instruction is a store
  teIF_Integer = 1 << 4, //!< Integer instruction
  teIF_Float = 1 << 5, //!< Floating point instruction
  teIF_Arithmetic = 1 << 6, //!< Arithmetic instruction
  teIF_Annulled = 1 << 7, //!< Annulled branch
  teIF_UnconditionalTaken = 1 << 8, //!< Unconditional taken branch
  teIF_UnconditionalNotTaken = 1 << 9, //!< Unconditional not-taken branch
  teIF_OnPage = 1 << 10, //!< On page branch
  teIF_ModeSwitch = 1 << 11, //!< Privilege mode switching instruction
  teIF_Call = 1 << 12, //!< Call instruction
  teIF_Unimplemented = 1 << 13, //!< Permanently unimplemented
} temu_InstructionFlags;

/*!
  Binary translation instrumentation interface
 */
typedef struct {
  //! Called on start of a block
  int (*beginBlock)(void *Obj, uint64_t VA, uint64_t PA);
  //! Called at the end of a block
  int (*endBlock)(void *Obj, uint64_t VA, uint64_t PA);
  //! Called when instruction is started
  int (*beginInstr)(void *Obj, uint64_t VA, uint64_t PA, uint32_t Instr,
                    uint32_t Flags);
  
  //! Called when instruction is finished.
  //! Arm: 0 = normal / not taken branch.
  //!      1 = taken conditional instruction.
  int (*endInstr)(void *Obj, uint64_t VA, uint64_t PA, uint32_t Instr,
                  uint32_t Flags, int Arm);
} temu_InstrumenterIface;

#define TEMU_INSTRUMENTER_IFACE_TYPE "temu::InstrumenterIface"
TEMU_IFACE_REFERENCE_TYPE(temu_Instrumenter);

#endif // ! TEMU_INSTRUMENTER_H
