//===------------------------------------------------------------*- C++ -*-===//
//
// TEMU: The Terma Emulator
// (c) Terma 2021
// Authors: Daria Vorotnikova <davo (at) terma.com>
//
//===----------------------------------------------------------------------===//

#ifndef TEMU_BUS_PCIe_H
#define TEMU_BUS_PCIe_H
#include <assert.h>
#include <stddef.h>
#include <stdint.h>

#include "temu-c/Bus/PCI.h"
#include "temu-c/Bus/PCIDefines.h"
#include "temu-c/Memory/Memory.h"
#include "temu-c/Support/Logging.h"
#include "temu-c/Support/Objsys.h"

//===----------------------------------------------------------------------===//
//
// NOTE: PCIe interfaces is an experimental and unstable API
//       It is subject to change until it is deemed stable enough.
//
//===----------------------------------------------------------------------===//

#ifdef __cplusplus
extern "C" {
#endif
/**
 * temu_PCIeMessageTypes:
 * PCI Express Messages
 * name = code
 */
typedef enum {
  tePMT_AssertIntA = 32,
  tePMT_AssertIntB = 33,
  tePMT_AssertIntC = 34,
  tePMT_AssertIntD = 35,
  tePMT_DeassertIntA = 36,
  tePMT_DeassertIntB = 37,
  tePMT_DeassertIntC = 38,
  tePMT_DeassertIntD = 39,
  tePMT_PmActiveStateNak = 20,
  tePMT_PmPme = 24,
  tePMT_PmeTurnOff = 25,
  tePMT_PmToAck = 27,
  tePMT_ErrCor = 48,
  tePMT_ErrNonFatal = 49,
  tePMT_ErrFatal = 51,
  tePMT_Unlock = 0,
  tePMT_SetSlotPowerLimit = 80,
  tePMT_VendorDefinedType0 = 126,
  tePMT_VendorDefinedType1 = 127,
  tePMT_AttentionIndicatorOn = 65,
  tePMT_AttentionIndicatorBlink = 67,
  tePMT_AttentionIndicatorOff = 64,
  tePMT_PowerIndicatorOn = 69,
  tePMT_PowerIndicatorBlink = 71,
  tePMT_PowerIndicatorOff = 68,
  tePMT_AttentionButtonPressed = 72
} temu_PCIeMessageTypes;

typedef struct {
  void (*mapPciIo)(temu_Object *Obj, unsigned Device, unsigned Bar,
                   uint64_t Addr, uint64_t Len);
  void (*unmapPciIo)(temu_Object *Obj, unsigned Device);
  void (*mapPciMem)(temu_Object *Obj, unsigned Device, unsigned Bar,
                    uint64_t Addr, uint64_t Len);
  void (*unmapPciMem)(temu_Object *Obj, unsigned Device);
} temu_PCIExpressBusIface;
TEMU_IFACE_REFERENCE_TYPE(temu_PCIExpressBus)
#define TEMU_PCIe_BUS_IFACE_TYPE "temu::PCIExpressBusIface"

typedef struct {
  void (*raiseReset)(temu_Object *);
  void (*lowerReset)(temu_Object *);
  void (*connect)(temu_Object *, temu_Object *);
  void (*disconnect)(temu_Object *, temu_Object *);
  void (*sendMessage)(void *, temu_PCIeMessageTypes msgType, uint8_t *payload);
  void (*setUpstreamBridge)(temu_Object *);
} temu_PCIExpressBridgeIface;
TEMU_IFACE_REFERENCE_TYPE(temu_PCIExpressBridge)
#define TEMU_PCIe_BRIDGE_IFACE_TYPE "temu::PCIExpressBridgeIface"

typedef struct temu_PCIExpressBus temu_PCIExpressBus;
typedef struct temu_PCIExpressBridge temu_PCIExpressBridge;

typedef struct {
  temu_PCIDevice Super;
  temu_PCIExpressBus *PrimaryBus;
} temu_PCIExpressDevice;

struct temu_PCIExpressBus {
  temu_Object Super;
  uint8_t BusId;

  temu_PCIExpressBridge *ParentDev;
  temu_PCIDeviceIfaceRefArray ChildrenDevs;
};

struct temu_PCIExpressBridge {
  temu_PCIExpressDevice Super;
  temu_PCIExpressBus *SecondaryBus;
  uint8_t PrimBusId;
  uint8_t SecBusId;
  uint8_t SubBusId;
};

static inline temu_PCIExpressBus *
temu_pciGetRootBus(temu_PCIExpressDevice *device)
{
  temu_PCIExpressBus *bus = device->PrimaryBus;

  while (bus->ParentDev != NULL) {
    device = &bus->ParentDev->Super;
    bus = device->PrimaryBus;
  }
  return bus;
}

static inline void
temu_pcieSetPrimaryBusId(temu_PCIExpressBridge *C, uint8_t Val)
{
  C->PrimBusId = Val;
}
static inline void
temu_pcieSetSecondaryBusId(temu_PCIExpressBridge *C, uint8_t Val)
{
  C->SecBusId = Val;
}
static inline void
temu_pcieSetSubordinaryBusId(temu_PCIExpressBridge *C, uint8_t Val)
{
  C->SubBusId = Val;
}

static inline void
temu_pcieSetBusId(temu_PCIExpressBus *C, uint8_t Val)
{
  C->BusId = Val;
}
static inline void
temu_pcieSetDeviceId(temu_PCIConfig config, uint16_t Val)
{
  temu_pciSetConfig(config, TEMU_PCI_CONFIG_DEVICE_ID_OFFSET,
                    TEMU_PCI_CONFIG_DEVICE_ID_SIZE, Val);
}

static inline void
temu_pcieSetVendorId(temu_PCIConfig config, uint16_t Val)
{
  temu_pciSetConfig(config, TEMU_PCI_CONFIG_VENDOR_ID_OFFSET,
                    TEMU_PCI_CONFIG_VENDOR_ID_SIZE, Val);
}

static inline void
temu_pcieSetStatus(temu_PCIConfig config, uint16_t Val)
{
  temu_pciSetConfig(config, TEMU_PCI_CONFIG_STATUS_OFFSET,
                    TEMU_PCI_CONFIG_STATUS_SIZE, Val);
}
static inline uint16_t
temu_pcieReadStatus(temu_PCIConfig config)
{
  return temu_pciGetConfig(config, TEMU_PCI_CONFIG_STATUS_OFFSET,
                           TEMU_PCI_CONFIG_STATUS_SIZE);
}

static inline void
temu_pcieSetClassCode(temu_PCIConfig config, uint32_t Val)
{
  temu_pciSetConfig(config, TEMU_PCI_CONFIG_CLASS_CODE_OFFSET,
                    TEMU_PCI_CONFIG_CLASS_CODE_SIZE, Val);
}

static inline void
temu_pcieSetRevId(temu_PCIConfig config, uint8_t Val)
{
  temu_pciSetConfig(config, TEMU_PCI_CONFIG_REVISION_ID_OFFSET,
                    TEMU_PCI_CONFIG_REVISION_ID_SIZE, Val);
}

static inline void
temu_pcieSetBist(temu_PCIConfig config, uint8_t Val)
{
  temu_pciSetConfig(config, TEMU_PCI_CONFIG_BIST_OFFSET,
                    TEMU_PCI_CONFIG_BIST_SIZE, Val);
}

static inline void
temu_pcieSetHeaderType(temu_PCIConfig config, uint8_t Val)
{
  temu_pciSetConfig(config, TEMU_PCI_CONFIG_HEADER_TYPE_OFFSET,
                    TEMU_PCI_CONFIG_HEADER_TYPE_SIZE, Val);
}

static inline void
temu_pcieSetLatencyTimer(temu_PCIConfig config, uint8_t Val)
{
  temu_pciSetConfig(config, TEMU_PCI_CONFIG_LATENCY_TIMER_OFFSET,
                    TEMU_PCI_CONFIG_LATENCY_TIMER_SIZE, Val);
}

static inline void
temu_pcieSetCacheLineSize(temu_PCIConfig config, uint8_t Val)
{
  temu_pciSetConfig(config, TEMU_PCI_CONFIG_CACHELINE_SIZE_OFFSET,
                    TEMU_PCI_CONFIG_CACHELINE_SIZE_SIZE, Val);
}

static inline void
temu_pcieSetInterruptPin0(temu_PCIConfig config, uint8_t Val)
{
  assert(Val < 0x05);
  temu_pciSetConfig(config, TEMU_PCI_TYPE_00_INTERRUPT_PIN_OFFSET,
                    TEMU_PCI_TYPE_00_INTERRUPT_PIN_SIZE, Val);
}

static inline void
temu_pcieSetSubsystemId(temu_PCIConfig config, uint16_t Val)
{
  assert(temu_pciGetConfig(config, TEMU_PCI_CONFIG_HEADER_TYPE_OFFSET,
                           TEMU_PCI_CONFIG_HEADER_TYPE_SIZE) == 0);
  temu_pciSetConfig(config, TEMU_PCI_TYPE_00_SUBSYSTEM_ID_OFFSET,
                    TEMU_PCI_TYPE_00_SUBSYSTEM_ID_SIZE, Val);
}

static inline void
temu_pcieSetSubsystemVendorId(temu_PCIConfig config, uint16_t Val)
{
  assert(temu_pciGetConfig(config, TEMU_PCI_CONFIG_HEADER_TYPE_OFFSET,
                           TEMU_PCI_CONFIG_HEADER_TYPE_SIZE) == 0);
  temu_pciSetConfig(config, TEMU_PCI_TYPE_00_SUBSYSTEM_VENDOR_ID_OFFSET,
                    TEMU_PCI_TYPE_00_SUBSYSTEM_VENDOR_ID_SIZE, Val);
}
static inline void
temu_pcieSetPrimaryBusNumber(temu_PCIConfig config, uint8_t Val)
{
  temu_pciSetConfig(config, TEMU_PCI_TYPE_01_PRIMARY_BUS_NUMBER_OFFSET,
                    TEMU_PCI_TYPE_01_PRIMARY_BUS_NUMBER_SIZE, Val);
}
static inline void
temu_pcieSetSecondaryBusNumber(temu_PCIConfig config, uint8_t Val)
{
  temu_pciSetConfig(config, TEMU_PCI_TYPE_01_SECONDARY_BUS_NUMBER_OFFSET,
                    TEMU_PCI_TYPE_01_SECONDARY_BUS_NUMBER_SIZE, Val);
}
static inline void
temu_pcieSetSubordinateBusNumber(temu_PCIConfig config, uint8_t Val)
{
  temu_pciSetConfig(config, TEMU_PCI_TYPE_01_SUBORDINATE_BUS_NUMBER_OFFSET,
                    TEMU_PCI_TYPE_01_SUBORDINATE_BUS_NUMBER_SIZE, Val);
}
static inline void
temu_pcieSetIOBase(temu_PCIConfig config, uint8_t Val)
{
  temu_pciSetConfig(config, TEMU_PCI_TYPE_01_IO_BASE_OFFSET,
                    TEMU_PCI_TYPE_01_IO_BASE_SIZE, Val);
}
static inline void
temu_pcieSetIOLimit(temu_PCIConfig config, uint8_t Val)
{
  temu_pciSetConfig(config, TEMU_PCI_TYPE_01_IO_LIMIT_OFFSET,
                    TEMU_PCI_TYPE_01_IO_LIMIT_SIZE, Val);
}
static inline void
temu_pcieSetSecondaryStatus(temu_PCIConfig config, uint16_t Val)
{
  temu_pciSetConfig(config, TEMU_PCI_TYPE_01_SECONDARY_STATUS_OFFSET,
                    TEMU_PCI_TYPE_01_SECONDARY_STATUS_SIZE, Val);
}
static inline void
temu_pcieSetMemoryBase(temu_PCIConfig config, uint16_t Val)
{
  temu_pciSetConfig(config, TEMU_PCI_TYPE_01_MEMORY_BASE_OFFSET,
                    TEMU_PCI_TYPE_01_MEMORY_BASE_SIZE, Val);
}
static inline void
temu_pcieSetMemoryLimit(temu_PCIConfig config, uint16_t Val)
{
  temu_pciSetConfig(config, TEMU_PCI_TYPE_01_MEMORY_LIMIT_OFFSET,
                    TEMU_PCI_TYPE_01_MEMORY_LIMIT_SIZE, Val);
}
static inline void
temu_pcieSetPrefetchableMemoryBase(temu_PCIConfig config, uint16_t Val)
{
  temu_pciSetConfig(config, TEMU_PCI_TYPE_01_PREFETCHABLE_MEMORY_BASE_OFFSET,
                    TEMU_PCI_TYPE_01_PREFETCHABLE_MEMORY_BASE_SIZE, Val);
}
static inline void
temu_pcieSetPrefetchableMemoryLimit(temu_PCIConfig config, uint16_t Val)
{
  temu_pciSetConfig(config, TEMU_PCI_TYPE_01_PREFETCHABLE_MEMORY_LIMIT_OFFSET,
                    TEMU_PCI_TYPE_01_PREFETCHABLE_MEMORY_LIMIT_SIZE, Val);
}
static inline void
temu_pcieSetInterruptLine(temu_PCIConfig config, uint8_t Val)
{
  temu_pciSetConfig(config, TEMU_PCI_TYPE_01_INTERRUPT_LINE_OFFSET,
                    TEMU_PCI_TYPE_01_INTERRUPT_LINE_SIZE, Val);
}
static inline void
temu_pcieSetInterruptPin1(temu_PCIConfig config, uint8_t Val)
{
  temu_pciSetConfig(config, TEMU_PCI_TYPE_01_INTERRUPT_PIN_OFFSET,
                    TEMU_PCI_TYPE_01_INTERRUPT_PIN_SIZE, Val);
}
static inline void
temu_pcieSetBridgeControl(temu_PCIConfig config, uint16_t Val)
{
  temu_pciSetConfig(config, TEMU_PCI_TYPE_01_BRIDGE_CONTROL_OFFSET,
                    TEMU_PCI_TYPE_01_BRIDGE_CONTROL_SIZE, Val);
}

static inline uint16_t
temu_pcieReadBridgeControl(temu_PCIConfig config)
{
  return temu_pciGetConfig(config, TEMU_PCI_TYPE_01_BRIDGE_CONTROL_OFFSET,
                           TEMU_PCI_TYPE_01_BRIDGE_CONTROL_SIZE);
}

#ifdef __cplusplus
}
#endif
#endif // !TEMU_BUS_PCIe_H
