//===------------------------------------------------------------*- C++ -*-===//
//
// TEMU: The Terma Emulator
// (c) Terma 2015
// Authors: Mattias Holm <maho (at) terma.com>
//
//===----------------------------------------------------------------------===//

/*
 * WARNING: The interfaces defined in this header are at present
 *          considered: EXPERIMENTAL!!!  In particularly, this means
 *          that there is no guaranteed API stability for the cache
 *          interfaces at the moment.
 */

#ifndef TEMU_CACHE_H
#define TEMU_CACHE_H

#include "temu-c/Support/Objsys.h"
#include "temu-c/Memory/Memory.h"

#include <stdint.h>

#ifdef __cplusplus
extern "C" {
#endif

// List of cache algorithms and corresponding numbers, numbers 0
// through 128 are reserved. Note that NONE implies that the cache is
// directly mapped, i.e. sets should be 1.
#define TEMU_CACHE_NONE 0
#define TEMU_CACHE_LRU 1
#define TEMU_CACHE_LRR 2
#define TEMU_CACHE_RND 3

//! Caches implement the memory access interface and the cache control
//! interface. For invalidate and evict operation the address given is
//! a physical address which should be mapped to a line. For Harward
//! style caches, the cache model would typically implement the
//! interface twice, once for instructions and once for data.

//! Invalidation: cache line will be tagged as invalid (i.e. flushed)
//! Eviction: cache line will be invalidated and in writeBack caches
//!           the content will be written back to memory.

typedef struct temu_CacheIface {
  void (*enable)(void *Obj); //!< Enable cache
  void (*disable)(void *Obj); //!< Disable cache
  void (*freeze)(void *Obj); //!< Freeze cache

  void (*lockLine)(void *Obj, uint64_t Addr); //!< Lock line for addr
  void (*unlockLine)(void *Obj, uint64_t Addr); //!< Unlock line


  void (*invalidateAll)(void *Obj); //!< Invalidate entire cache
  void (*invalidateLine)(void *Obj, uint64_t Addr); //!< Invalidate single line

  void (*evictAll)(void *Obj); //!< Evict all (i.e. spill to memory)
  void (*evictLine)(void *Obj, uint64_t Addr); //!< Evict line

  //! Interrogation of cache properties
  //! - A directly mapped cache has 1 way, but typically several sets
  //! - A fully associative cache has 1 set, but typically several ways
  //! - Total cache size in bytes is sets * ways * linesize
  uint32_t (*getReplacementPolicy)(void *Obj);
  //! Get number of sets
  uint32_t (*getSets)(void *Obj);
  //! Get number of ways
  uint32_t (*getWays)(void *Obj);
  //! Get line size in bytes
  uint32_t (*getLineSize)(void *Obj);

  //! Check if an address belong to a cached line
  int (*isValid)(void *Obj, uint64_t Addr);

  //! Access the flags data, the format is cache model specific and can
  //! for specific models include also the tag. Typically, this is used
  //! for diagnostic access for more accurate cache models. Note that
  //! addr is not a physical address, but rather an address interpreted
  //! differently for different caches. I.e. the address is in a cache
  //! local address space.
  uint64_t (*readFlags)(void *Obj, uint64_t Addr);
  
  //! Write flags data
  void (*writeFlags)(void *Obj, uint64_t Addr, uint64_t Flags);

  //! Get cached data for the given address. The data returned comes
  //! from RAM using a normal memory transaction in case the cache is
  //! modelled for timing only, otherwise it comes from the cache
  //! model. Addr is a cache local address space, which depends on the
  //! cache implementation. The functions are intended for diagnostic
  //! cache access which is implemented in some systems. Note that on
  //! writes, the data will NOT be written to memory except in the case
  //! the cache line would be evicted in an accurate cache model. All
  //! reads from valid addresses succeed, other addresses are
  //! undefined. writeData returns non-zero on failure (e.g. invalid
  //! address).
  uint64_t (*readData)(void *Obj, uint64_t Addr);
  
  //! Write data in cache
  int (*writeData)(void *Obj, uint64_t Addr, uint64_t Data);
} temu_CacheIface;
#define TEMU_CACHE_IFACE_TYPE "temu::CacheIface"
TEMU_IFACE_REFERENCE_TYPE(temu_Cache);

//! The cache control interface can be implemented by a cache
//! controller. Which may be embedded in a CPU or device
//! model. Typically, the assumption is that the cache controller will
//! turn on and off the cache and that it may need to be notified on
//! global eviction operations, which may or may not be reflected in
//! the cache controllers status registers.
typedef struct temu_CacheCtrlIface {
  //! Global evict operation ongoing
  void (*evictionInProgress)(void *Obj);

  //! Global evict operation completed
  void (*evictionCompleted)(void *Obj);
} temu_CacheCtrlIface;
#define TEMU_CACHE_CTRL_IFACE_TYPE "temu::CacheCtrlIface"
TEMU_IFACE_REFERENCE_TYPE(temu_CacheCtrl);

#ifdef __cplusplus
}
#endif

#endif /* ! TEMU_CACHE_H */
