//===-- temu-c/Preferences.h - TEMU Global Preferences ----------*- C++ -*-===//
//
// TEMU: The Terma Emulator
// (c) Terma 2025
// Authors: Mattias Holm <maho (at) terma.com>
//
//===----------------------------------------------------------------------===//
#ifndef TEMU_CONFIGURATION_H
#define TEMU_CONFIGURATION_H

#include "temu-c/Support/Attributes.h"
#include <stdbool.h>
#include <stdint.h>

#ifdef __cplusplus
extern "C" {
#endif

/*!
    Get unsigned config entry

    \param path '/' separated path in the config file
        E.g. "custom/foo"
    \param defaultValue The default value returned if path is not found in
    \result The value found in ~/.config/temu/config.yaml
      or the defaultValue if not found.
 */
TEMU_API uint64_t temu_configGetUnsigned(const char *path,
                                        uint64_t defaultValue);

/*!
    Get signed config entry

    \param path '/' separated path in the config file
        E.g. "custom/foo"
    \param defaultValue The default value returned if path is not found in
    \result The value found in ~/.config/temu/config.yaml
      or the defaultValue if not found.
 */
TEMU_API int64_t temu_configGetSigned(const char *path, uint64_t defaultValue);

/*!
  Get string config entry

  \param path '/' separated path in the config file
    E.g. "custom/foo"
  \param defaultValue The default value returned if path is not found in
  \result A pointer to a thread local copy of a value found in
    ~/.config/temu/config.yaml or the defaultValue if not found.
    The returned string is valid until the next invocation only.
*/
TEMU_API const char *temu_configGetString(const char *path,
                                         const char *defaultValue);

/*!
    Get double config entry

    \param path '/' separated path in the config file
        E.g. "custom/foo"
    \param defaultValue The default value returned if path is not found in
    \result The value found in ~/.config/temu/config.yaml
      or the defaultValue if not found.
 */
TEMU_API double temu_configGetDouble(const char *path, double defaultValue);
/*!
    Get boolean config entry

    \param path '/' separated path in the config file
        E.g. "custom/foo"
    \param defaultValue The default value returned if path is not found in
    \result The value found in ~/.config/temu/config.yaml
      or the defaultValue if not found.
 */
TEMU_API bool temu_configGetBool(const char *path, bool defaultValue);

#ifdef __cplusplus
}
#endif

#endif // !TEMU_CONFIGURATION_H
