//===-- temu-c/Support/Time.h - Portable Timers -----------------*- C++ -*-===//
//
// TEMU: The Terma Emulator
// (c) Terma 2015
// Authors: Mattias Holm <maho (at) terma.com>
//
//===----------------------------------------------------------------------===//

#ifndef TEMU_SUPPORT_TIME
#define TEMU_SUPPORT_TIME
#include "temu-c/Support/Temu3Compat.h"
#include "temu-c/Support/Attributes.h"
#include "temu-c/Support/Events.h"
#include <stdint.h>
#ifdef __cplusplus
extern "C" {
#endif

/*!
  Get monotonic time in nanoseconds.

  The monotonic time is relative since some epoch of undefined start,
  but it is monotonic, unaffected by adjustments due to leap seconds,
  setting of the system clock, etc.

  This function is primarily useful for doing performance measurements
  since the time returned is relative to an undefined point (although
  that undefined point will be consistent while the program is
  running).

  In practice, on systems with clock_gettime() implemented, this
  function returns the timespec converted to nanoseconds, but on
  systems without clock_gettime(), e.g. Darwin and Windows, the
  function will only ensure that some notion of monotonic nanoseconds
  are returned. In Darwin for example, this results in a monotonic
  time returned which is relative to the first call to the function.

  \result Wall clock nanoseconds since an unspecified epoch.
 */
TEMU_API uint64_t temu_timeGetMonotonicWct(void);

/*!
 * Returns wall clock nanoseconds of thread time
 *
 * NOTE: This is how much time the thread this is called from has been
 * scheduled.
 *
 * \return wall clock nanoseconds of thread time
 */
TEMU_API uint64_t temu_timeGetThreadWct(void);

/*!
 * Get the current simulated real-time in nanoseconds
 * \param TS The object in question
 * \return simulated time for the given object
 */
TEMU_API uint64_t temu_timeGetCurrentSrtNanos(temu_TimeSource_ *TS);


/*!
 * Convert cycles to steps, rounding upwards
 * \param TS Time source object
 * \param Cycles Cycle count to convert to steps
 * \result Cycles converted to steps given the time source IPC/CPI configuration.
 */
TEMU_API uint64_t temu_cyclesToStepsRoundedUp(temu_TimeSource *TS, uint64_t Cycles);

/*!
 * Convert steps to cycles, rounding upwards
 * \param TS Time source object
 * \param Steps Step count to convert to cycles
 * \result Steps converted to cycles given the time source IPC/CPI configuration.
 */
TEMU_API uint64_t temu_stepsToCyclesRoundedUp(temu_TimeSource *TS, uint64_t Steps);


/*!
 * Convert cycles to steps
 * \param TS Time source object
 * \param Cycles Cycle count to convert to steps
 * \result Cycles converted to steps given the time source IPC/CPI configuration.
 */
TEMU_API uint64_t temu_cyclesToSteps(temu_TimeSource *TS, uint64_t Cycles);

/*!
 * Convert steps to cycles
 * \param TS Time source object
 * \param Steps Step count to convert to cycles
 * \result Steps converted to cycles given the time source IPC/CPI configuration.
 */
TEMU_API uint64_t temu_stepsToCycles(temu_TimeSource *TS, uint64_t Steps);

/*!
 * Get current step count
 * \param TS Time source object
 * \result Current steps as it is understood by the object.
 */
TEMU_API int64_t temu_getSteps(temu_TimeSource *TS);



/*!
 * Get current time in cycles.
 * \param TS Time source object
 * \result Current cycles as it is understood by the object.
 */
TEMU_API int64_t temu_getCycles(temu_TimeSource_ *TS);

/*!
 * Get current time in nanoseconds
 * \param TS Time source object
 * \result Current nanoseconds as it is understood by the object.
 */
TEMU_API int64_t temu_getNanos(temu_TimeSource_ *TS);

/*!
 * Get current time in seconds
 * \param TS Time source object
 * \result Current seconds as it is understood by the object.
 */
TEMU_API double temu_getSecs(temu_TimeSource_ *TS);

/*
 * Time conversion, note that this isn't super-trivial due to
 * overflows. So use these functions, they are well tested and
 * hardened against overflow. Note that due to rounding, there is no
 * strict guarantee that the inverse functions are strictly inverse
 * e.g. WE DO NOT GUARANTEE that:
 *    (cyclesToNanos(nanosToCycles(N, F), F)) == N
 */

/*!
 * Convert cycles to nanoseconds
 * \param Cycles Cycle count to convert
 * \param Freq Frequency in Hz
 * \result Cycles converted to nanoseconds
 */
TEMU_API int64_t temu_cyclesToNanos(int64_t Cycles, int64_t Freq);

/*!
 * Convert cycles to another frequency base, with truncated result
 * \param Cycles Cycle count in source frequency ticks
 * \param SourceFreq Frequency in Hz of the source
 * \param TargetFreq Frequency in Hz for the converted value
 * \result Cycles, converted from source frequency to target frequency
 */
TEMU_API int64_t temu_cyclesToOtherFreqTruncated(int64_t Cycles,
                                                 int64_t SourceFreq,
                                                 int64_t TargetFreq);

/*!
 * Convert cycles to another frequency base, rounding upwards
 * \param Cycles Cycle count in source frequency ticks
 * \param SourceFreq Frequency in Hz of the source
 * \param TargetFreq Frequency in Hz for the converted value
 * \result Cycles, converted from source frequency to target frequency
 */
TEMU_API int64_t temu_cyclesToOtherFreqRoundedUp(int64_t Cycles,
                                                 int64_t SourceFreq,
                                                 int64_t TargetFreq);

/*!
 * Convert cycles to seconds
 * \param Cycles Cycle count to convert
 * \param Freq Frequency in Hz
 * \result Cycles converted to seconds
 */
TEMU_API double temu_cyclesToSecs(int64_t Cycles, int64_t Freq);

/*!
 * Convert nanoseconds to cycles
 *
 * This function truncates the result in case of non-exact conversion.
 * \param Nanos Nanoseconds to convert
 * \param Freq Frequency in Hz
 * \result Nanoseconds converted to cycles
 */
TEMU_API int64_t temu_nanosToCycles(int64_t Nanos, int64_t Freq);

/*!
 * Convert nanoseconds to cycles rounded upwards
 *
 * This function rounds up the result in case of non-exact conversion.
 * \param Nanos Nanoseconds to convert
 * \param Freq Frequency in Hz
 * \result Nanoseconds converted to cycles
 */
TEMU_API int64_t temu_nanosToCyclesRoundedUp(int64_t Nanos, int64_t Freq);

/*!
 * Convert nanoseconds to seconds
 * \param Nanos Nanoseconds to convert
 * \result Nanoseconds converted to seconds
 */
TEMU_API double temu_nanosToSecs(int64_t Nanos);

/*!
 * Convert seconds to nanoseconds
 * \param Secs Seconds to convert
 * \result Seconds converted to an integral number of nanoseconds
 */
TEMU_API int64_t temu_secsToNanos(double Secs);

/*!
 * Convert seconds to cycles
 * \param Secs Seconds to convert
 * \param Freq Frequency in Hz
 * \result Seconds converted to an integral number of cycles
 */
TEMU_API int64_t temu_secsToCycles(double Secs, int64_t Freq);

#ifdef __cplusplus
}
#endif

#endif /* ! TEMU_SUPPORT_TIME */
