#include <inttypes.h>
#include <stdint.h>
#include <stdio.h>
#include <string.h>

#include "temu-c/Bus/Ethernet.h"
#include "temu-c/Support/Attributes.h"
#include "temu-c/Support/Bitmanip.h"
#include "temu-c/Support/Buffer.h"
#include "temu-c/Support/Logging.h"
#include "temu-c/Support/Objsys.h"

typedef struct {
  temu_Object Super;

  uint8_t MAC[6];

  uint8_t TargetMAC[6];

  temu_MDIOIfaceRef mdio;
  temu_PHYIfaceRef phy;
} EthDevice;

// Called when link is connected (i.e. cable inserted)
static void
connected(void *Dev)
{
}
// Called when link is disconnected (i.e. cable removed)
static void
disconnected(void *Dev)
{
}

// Called when link goes up
static void
up(void *Dev)
{
}

// Called when link goes down
static void
down(void *Dev)
{
}

// Frame reception
static int
receive(void *Dev, temu_EthFrame *Frame)
{
  const uint8_t *Data = temu_buffReadableData(&Frame->Data);
  size_t Len = temu_buffLen(&Frame->Data);
  assert(Len >= 64); // Minimum ethernet frame length

  uint64_t DestMAC = 0;
  uint64_t SrcMAC = 0;

  memcpy(&DestMAC, &Data[0], 6);
  memcpy(&SrcMAC, &Data[6], 6);

  DestMAC = temu_swapBigHost64(DestMAC);
  SrcMAC = temu_swapBigHost64(SrcMAC);

  temu_logSimInfo(Dev, "received frame from: %.12" PRIx64 " to: %.12" PRIx64,
                  SrcMAC, DestMAC);

  // Note, if you want to retain the buffer after the receive function has
  // finished you should do the following:

  // temu_Buff BC = temu_buffCopy(&Frame->Data);
  // temu_buffDispose(&BC);
  return 0;
}

static uint64_t
getMAC(void *Dev)
{
  EthDevice *E = (EthDevice *)Dev;
  uint64_t M = 0;
  memcpy(&M, E->MAC, 6);
  M = temu_swapBigHost64(M);
  return M;
}

static void
setMAC(void *Dev, uint64_t MAC)
{
  EthDevice *E = (EthDevice *)Dev;
  uint8_t M[8];
  MAC = temu_swapBigHost64(MAC);
  memcpy(M, &MAC, 8);
  memcpy(E->MAC, &M[2], 6);
}

static temu_MACIface MACIface = {connected, disconnected, up,    down,
                                 receive,   getMAC,       setMAC};

static void *
create(const char *Name, int Argc TEMU_UNUSED,
       const temu_CreateArg *Argv TEMU_UNUSED)
{
  EthDevice *Obj = malloc(sizeof(EthDevice));
  memset(Obj, 0, sizeof(EthDevice));

  printf("Creating Object '%s'\n", Name);

  return Obj;
}

static void
dispose(void *Obj)
{
  free(Obj);
}

static int
sendFrame(void *Dev)
{
  static uint8_t Preamble[] = {0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa};
  EthDevice *E = (EthDevice *)Dev;

  temu_EthFrame F;

  F.Flags = TEMU_ETH_ETH_CRC_NOT_SET; // We do not compute a CRC (receiving
                                      // device should ignore it).
  F.Data = temu_buffCreate(64);       // Create 64 byte COW buffer

  // Preamble and SFD are optional fields, they should
  // if non-standard be accompanied by the TEMU_ETH_NON_STANDARD_PREAMBLE flag.
  memcpy(F.Preamble, Preamble, 7);
  F.Sfd = 0xab;
  uint8_t *Bytes = temu_buffWritableData(&F.Data);

  memcpy(Bytes, E->TargetMAC, 6);
  memcpy(&Bytes[6], E->MAC, 6);

  // Populate rest of frame

  // Optionally add a CRC

  // Send frame
  E->phy.Iface->send(E->phy.Obj, &F);

  temu_buffDispose(&F.Data);
  return 0;
}

static void
triggerTestFrame(void *Obj, temu_Propval PV, int idx)
{
  EthDevice *E = (EthDevice *)Obj;
  uint8_t M[8];
  uint64_t MAC = temu_swapBigHost64(temu_asUnsigned(PV));
  memcpy(M, &MAC, 8);
  memcpy(E->TargetMAC, &M[2], 6);
  sendFrame(Obj);
}

static void
setMACProp(void *Obj, temu_Propval PV, int idx)
{
  setMAC(Obj, temu_asUnsigned(PV));
}

static temu_Propval
getMACProp(void *Obj, int idx)
{
  return temu_makePropU64(getMAC(Obj));
}

TEMU_PLUGIN_INIT
{
  temu_Class *Cls = temu_registerClass("EthDevice", create, dispose);

  temu_addInterfaceReference(Cls, "phy", offsetof(EthDevice, phy),
                             TEMU_PHY_IFACE_TYPE, 1, 0, NULL, NULL,
                             "PHY reference");

  temu_addInterfaceReference(Cls, "mdio", offsetof(EthDevice, mdio),
                             TEMU_MDIO_IFACE_TYPE, 1, 0, NULL, NULL,
                             "MDIO reference");

  temu_addPseudoProperty(Cls, "mac", teTY_U64,
                         1, // Number of elements (1 = scalar)
                         setMACProp, getMACProp, setMACProp, getMACProp,
                         "MAC address of device reference");

  temu_addInterface(Cls, "MACIface", TEMU_MAC_IFACE_TYPE, &MACIface, 0,
                    "MAC interface");

  temu_addPseudoProperty(Cls, "sendTestFrameTo", teTY_U64,
                         1, // Number of elements (1 = scalar)
                         triggerTestFrame, NULL, NULL, NULL,
                         "Send test frame to set MAC address.");
}
