#include "temu-c/Bus/MilStd1553.h"
#include "temu-c/Support/Events.h"
#include "temu-c/Support/Logging.h"
#include "temu-c/Support/Objsys.h"
#include <string.h>

typedef enum {
  CS_None,
  CS_TxCmd,
  CS_RxCmd,
  CS_Data,
} CommandState;

typedef struct {
  temu_Object Super;
  CommandState State;

  uint16_t Cmd;
  uint16_t RxBuffer[32][32];
  uint16_t TxBuffer[32][32];

  uint16_t RxBufferSize[32];
  uint16_t TxBufferSize[32];

  int64_t TxEventID;
  int64_t StatusEventID;

  temu_Mil1553BusIfaceRef Bus;
  uint16_t RTAddr;

  struct {
    int8_t InfiniteSpeed;
  } Config;
} RTU;

void transmitDataEvent(temu_Event *EV);
void transmitStatusEvent(temu_Event *EV);

void *
create(const char *Name TEMU_UNUSED, int Argc TEMU_UNUSED,
       const temu_CreateArg Argv[] TEMU_UNUSED)
{
  RTU *RT = malloc(sizeof(RTU));
  memset(RT, 0, sizeof(RTU));

  RT->TxEventID = temu_eventPublish("txEvent", RT, transmitDataEvent);
  RT->StatusEventID = temu_eventPublish("statusEvent", RT, transmitStatusEvent);

  RT->Config.InfiniteSpeed = 1;

  return RT;
}

void
dispose(void *Obj)
{
  RTU *RT = (RTU *)Obj;
  free(RT);
}

void
connected(void *Obj, temu_Mil1553BusIfaceRef Bus, int RTAddr)
{
  RTU *RT = (RTU *)Obj;
  RT->Bus = Bus;
  RT->RTAddr = RTAddr;
}
void
disconnected(void *Obj, temu_Mil1553BusIfaceRef Bus, int RTAddr)
{
  RTU *RT = (RTU *)Obj;
  RT->Bus = (temu_Mil1553BusIfaceRef){NULL, NULL};
  RT->RTAddr = 0;
}

void
transmitDataEvent(temu_Event *EV)
{
  RTU *RT = (RTU *)EV->Obj;

  // Send status first
  uint16_t Stat;
  temu_Mil1553Msg Msg = temu_mil1553CreateStatMsg(&Stat, RT->RTAddr);
  RT->Bus.Iface->send(RT->Bus.Obj, RT, &Msg);

  // Then send data
  uint16_t SA = (RT->Cmd >> 5) & 0x1f;
  uint16_t WC = RT->Cmd & 0x1f;
  WC = WC == 0 ? 32 : WC;

  Msg = temu_mil1553CreateDataMsg(RT->TxBuffer[SA], WC);

  RT->Bus.Iface->send(RT->Bus.Obj, RT, &Msg);
}

void
transmitStatusEvent(temu_Event *EV)
{
  RTU *RT = (RTU *)EV->Obj;

  temu_Mil1553Msg Msg;
  uint16_t Stat;

  Msg = temu_mil1553CreateStatMsg(&Stat, RT->RTAddr);
  RT->Bus.Iface->send(RT->Bus.Obj, RT, &Msg);
}

void
receive(void *Obj, temu_Mil1553Msg *Msg)
{
  RTU *RT = (RTU *)Obj;

  if (Msg->MsgTyp == teMT_Cmd) {
    RT->Cmd = Msg->Data[0];
  }

  uint16_t RTId = (RT->Cmd >> 11) & 0x1f;
  uint16_t SA = (RT->Cmd >> 5) & 0x1f;
  bool IsBcast = (RTId == 31);

  // Note that the bus model handles state transitions and
  // verification, we just need to switch on the type and take into
  // account wether it is a broadcast or direct message.
  if (Msg->MsgTyp == teMT_Cmd) {
    int TransmitRequest = temu_mil1553CmdTR(Msg);
    RT->Cmd = Msg->Data[0];

    if (temu_mil1553CmdIsModeCodeCmd(Msg)) {
      uint16_t MC = temu_mil1553ModeCode(Msg);
      bool ExpectsData = (MC >> 4) & 1;

      if (ExpectsData) {
        if (TransmitRequest) {
          if (!IsBcast) {
            // Semantics for handling TX request can be implemented here,
            // e.g.

            if (RT->Config.InfiniteSpeed) {
              temu_eventPostStack(RT->Super.TimeSource, RT->TxEventID,
                                  teSE_Cpu);
            } else {
              uint64_t Nanos = temu_mil1553TransferTime(1);
              temu_eventPostNanos(RT->Super.TimeSource, RT->TxEventID, Nanos,
                                  teSE_Cpu);
            }
          }
        }
        // Expects data but is received request, we should get a data phase
        // message next
      } else {
        // Does not expect data
        if (TransmitRequest) {
          if (!IsBcast) {
            // Transmit requests should be replied to with a status
            if (RT->Config.InfiniteSpeed) {
              temu_eventPostStack(RT->Super.TimeSource, RT->StatusEventID,
                                  teSE_Cpu);
            } else {
              uint64_t Nanos = temu_mil1553TransferTime(1);
              temu_eventPostNanos(RT->Super.TimeSource, RT->StatusEventID,
                                  Nanos, teSE_Cpu);
            }
          }
        } else {
          temu_logError(RT, "receive request for no data word mc %x",
                        (unsigned)MC);
        }
      }
    } else {
      // Normal message (not mode code)
      // For transmit requests which are not broadcast we should respond
      // with status and data
      if (TransmitRequest) {
        if (IsBcast) {
          temu_logError(RT, "transmit request broadcast cmd = %x",
                        (unsigned)RT->Cmd);
        } else {
          if (RT->Config.InfiniteSpeed) {
            temu_eventPostStack(RT->Super.TimeSource, RT->TxEventID, teSE_Cpu);
          } else {
            unsigned Words = RT->Cmd & 0x1f;
            Words = Words == 0 ? 32 : 0;

            // Also account for status word
            uint64_t Nanos = temu_mil1553TransferTime(Words + 1);
            temu_eventPostNanos(RT->Super.TimeSource, RT->TxEventID, Nanos,
                                teSE_Cpu);
          }
        }
      } else {
        // Receive request, next message will be data
      }
    }
  } else if (teMT_Data) {
    memcpy(RT->RxBuffer[SA], Msg->Data, Msg->WordCount * 2);

    // We are receiving data (perhaps due to a mode code command),
    // will now do two things, firstly handle the command itself in the
    // OBC model and secondly return the data.

    // This is data from a receive request, logics for handling this can be
    // implemented here

    // Data for receive request, we need to respond with status,
    // unless it was a broadcast.
    if ((RT->Cmd >> 11) != 0x1f) {
      if (RT->Config.InfiniteSpeed) {
        temu_eventPostStack(RT->Super.TimeSource, RT->StatusEventID, teSE_Cpu);
      } else {
        uint64_t Nanos = temu_mil1553TransferTime(1);
        temu_eventPostNanos(RT->Super.TimeSource, RT->StatusEventID, Nanos,
                            teSE_Cpu);
      }
    }
  }
}

temu_Mil1553DevIface MilBusDevIface = {connected, disconnected, receive};

TEMU_PLUGIN_INIT
{
  temu_Class *C = temu_registerClass("MilbusRTU", create, dispose);

  temu_addProperty(C, "config.infiniteSpeed",
                   offsetof(RTU, Config.InfiniteSpeed), teTY_I8, 1, NULL, NULL,
                   "Infinite speed mode");

  temu_addProperty(C, "rtAddr", offsetof(RTU, RTAddr), teTY_U16, 1, NULL, NULL,
                   "Remote terminal address, set on connect.");

  temu_addProperty(C, "milbus", offsetof(RTU, Bus), teTY_IfaceRef, 1, NULL,
                   NULL, "Milbus interface reference");
  temu_requireInterface(C, "milbus", TEMU_MIL1553_BUS_IFACE_TYPE);

  temu_addInterface(C, "Mil1553DevIface", TEMU_MIL1553_DEV_IFACE_TYPE,
                    &MilBusDevIface, 0, "Milbus device interface");
}
