//===------------------------------------------------------------*- C++ -*-===//
//
// TEMU: The Terma Emulator
// (c) Terma 2019
// Authors: Mattias Holm <maho (at) terma.com>
//
//===----------------------------------------------------------------------===//

#ifndef TEMU_BUS_PCI_H
#define TEMU_BUS_PCI_H
#include <assert.h>
#include <stdint.h>
#include <stdlib.h>

#include "temu-c/Bus/PCIDefines.h"
#include "temu-c/Memory/Memory.h"
#include "temu-c/Support/Logging.h"
#include "temu-c/Support/Objsys.h"

//===----------------------------------------------------------------------===//
//
// NOTE: PCI interfaces is an experimental and unstable API
//       It is subject to change until it is deemed stable enough.
//
//===----------------------------------------------------------------------===//

typedef struct {
  temu_MemAccessIface ConfigAccess;
  temu_MemAccessIface IoAccess;
  temu_MemAccessIface MemAccess;
  temu_MemoryIface ConfigBlock;
  temu_MemoryIface IoBlock;
  temu_MemoryIface MemBlock;
} temu_PCIDeviceVTable;

#define TEMU_PCI_CONFIG_BYTES 256
#define TEMU_PCIE_CONFIG_BYTES 4096
#define TEMU_PCI_CONFIG_WORDS 64
#define TEMU_PCIE_CONFIG_WORDS 1024

typedef struct {
  size_t Size; // In bytes
  uint32_t
      *Data; // Configuration space data must be aligned at 4 byte boundaries
} temu_PCIConfig;

typedef struct {
  void (*mapPciIo)(temu_Object *Obj, unsigned Device, unsigned Bar,
                   uint64_t Addr, uint64_t Len);
  void (*unmapPciIo)(temu_Object *Obj, unsigned Device);
  void (*mapPciMem)(temu_Object *Obj, unsigned Device, unsigned Bar,
                    uint64_t Addr, uint64_t Len);
  void (*unmapPciMem)(temu_Object *Obj, unsigned Device);
} temu_PCIBusIface;
TEMU_IFACE_REFERENCE_TYPE(temu_PCIBus)
#define TEMU_PCI_BUS_IFACE_TYPE "temu::PCIBusIface"

typedef struct {
  void (*raiseReset)(temu_Object *);
  void (*lowerReset)(temu_Object *);
  void (*connect)(temu_Object *, temu_Object *);
  void (*disconnect)(temu_Object *, temu_Object *);
} temu_PCIBridgeIface;
TEMU_IFACE_REFERENCE_TYPE(temu_PCIBridge)
#define TEMU_PCI_BRIDGE_IFACE_TYPE "temu::PCIBridgeIface"

typedef struct {
  void (*startSelfTest)(
      temu_Object *); // Optional, called when BIST bit 6 is set
  temu_PCIConfig (*getPciConfig)(temu_Object *);
  // Writes to config space forwarded here
  void (*writeConfig)(temu_Object *, uint32_t offset, uint32_t value);
  uint32_t (*readConfig)(temu_Object *Obj, uint32_t offset);
  uint64_t (*getPciBarSize)(temu_Object *, unsigned bar);
  uint64_t (*getPciExpansionROMSize)(temu_Object *);
  temu_MemAccessIface *(*getPciBarIface)(temu_Object *, unsigned BarId,
                                         uint8_t type);
} temu_PCIDeviceIface;
TEMU_IFACE_REFERENCE_TYPE(temu_PCIDevice)
#define TEMU_PCI_DEVICE_IFACE_TYPE "temu::PCIDeviceIface"

typedef struct {
  temu_Object Super;
  temu_PCIConfig Conf;
} temu_PCIDevice;

#ifdef __cplusplus
extern "C" {
#endif

TEMU_API void temu_pciSetConfig(temu_PCIConfig config, uint32_t offset,
                                uint32_t size, uint32_t value);
TEMU_API uint32_t temu_pciGetConfig(temu_PCIConfig config, uint32_t offset,
                                    uint32_t size);

#ifdef __cplusplus
}
#endif

static inline void
temu_pciDeviceRegister(temu_Class *C)
{
  /*
  temu_addProperty(C, "pciConfigDeviceVendorID",
                   offsetof(temu_PCIDevice, Conf.DeviceVendorID), teTY_U32,
                   1, // Number of elements (1 = scalar)
                   NULL, NULL, "PCI Device and Vendor ID");

  temu_addProperty(C, "pciConfigStatusCommand",
                   offsetof(temu_PCIDevice, Conf.StatusCommand), teTY_U32,
                   1, // Number of elements (1 = scalar)
                   NULL, NULL, "PCI Status and Command Register");

  temu_addProperty(C, "pciConfigClassCodeRevID",
                   offsetof(temu_PCIDevice, Conf.ClassCodeRevID), teTY_U32,
                   1, // Number of elements (1 = scalar)
                   NULL, NULL, "PCI Class Code and Revision ID");

  temu_addProperty(
      C, "pciConfigBistHeaderLatencyCacheLineAndSize",
      offsetof(temu_PCIDevice, Conf.BISTHeaderTypeLatencyTimerCacheLineSize),
      teTY_U32,
      1, // Number of elements (1 = scalar)
      NULL, NULL, "PCI BIST, Header Type, Latency and cache line size");

  temu_addProperty(C, "pciConfigBAR", offsetof(temu_PCIDevice, Conf.BAR),
                   teTY_U32,
                   6, // Number of elements (1 = scalar)
                   NULL, NULL, "PCI Base Address Registers");

  temu_addProperty(C, "pciConfigCardbusCISPointer",
                   offsetof(temu_PCIDevice, Conf.CardbusCISPointer), teTY_U32,
                   1, // Number of elements (1 = scalar)
                   NULL, NULL, "PCI Cardbus CIS Pointer");

  temu_addProperty(C, "pciConfigSubsystemVendorID",
                   offsetof(temu_PCIDevice, Conf.SubsystemVendorID), teTY_U32,
                   1, // Number of elements (1 = scalar)
                   NULL, NULL, "PCI Subsystem and Subsystem Vendor ID");

  temu_addProperty(C, "pciConfigExpansionROMBaseAddress",
                   offsetof(temu_PCIDevice, Conf.ExpansionROMBaseAddress),
                   teTY_U32,
                   1, // Number of elements (1 = scalar)
                   NULL, NULL, "PCI Expansion ROM base address");

  temu_addProperty(C, "pciConfigCapabilityPointer",
                   offsetof(temu_PCIDevice, Conf.CapPointer), teTY_U32,
                   1, // Number of elements (1 = scalar)
                   NULL, NULL, "PCI Capability list pointer");

  temu_addProperty(
      C, "pciConfigMaxLatMinGntIntPinLine",
      offsetof(temu_PCIDevice, Conf.MaxLatMinGntIntPinLine), teTY_U32,
      1, // Number of elements (1 = scalar)
      NULL, NULL, "PCI Max latency, min gnt, int pin and int line");
*/
}

static inline void
temu_pciSetDeviceId(temu_PCIConfig C, uint16_t Val)
{
  temu_pciSetConfig(C, TEMU_PCI_CONFIG_DEVICE_ID_OFFSET,
                    TEMU_PCI_CONFIG_DEVICE_ID_SIZE, Val);
}

static inline void
temu_pciSetVendorId(temu_PCIConfig C, uint16_t Val)
{
  temu_pciSetConfig(C, TEMU_PCI_CONFIG_VENDOR_ID_OFFSET,
                    TEMU_PCI_CONFIG_VENDOR_ID_SIZE, Val);
}

static inline void
temu_pciSetStatus(temu_PCIConfig C, uint16_t Val)
{
  temu_pciSetConfig(C, TEMU_PCI_CONFIG_STATUS_OFFSET,
                    TEMU_PCI_CONFIG_STATUS_SIZE, Val);
}

static inline void
temu_pciSetClassCode(temu_PCIConfig C, uint32_t Val)
{
  temu_pciSetConfig(C, TEMU_PCI_CONFIG_CLASS_CODE_OFFSET,
                    TEMU_PCI_CONFIG_CLASS_CODE_SIZE, Val);
}

static inline void
temu_pciSetRevId(temu_PCIConfig C, uint8_t Val)
{
  temu_pciSetConfig(C, TEMU_PCI_CONFIG_REVISION_ID_OFFSET,
                    TEMU_PCI_CONFIG_REVISION_ID_SIZE, Val);
}

static inline void
temu_pciSetBist(temu_PCIConfig C, uint8_t Val)
{
  temu_pciSetConfig(C, TEMU_PCI_CONFIG_BIST_OFFSET, TEMU_PCI_CONFIG_BIST_SIZE,
                    Val);
}

static inline void
temu_pciSetHeaderType(temu_PCIConfig C, uint8_t Val)
{
  temu_pciSetConfig(C, TEMU_PCI_CONFIG_HEADER_TYPE_OFFSET,
                    TEMU_PCI_CONFIG_HEADER_TYPE_SIZE, Val);
}

static inline void
temu_pciSetLatencyTimer(temu_PCIConfig C, uint8_t Val)
{
  temu_pciSetConfig(C, TEMU_PCI_CONFIG_LATENCY_TIMER_OFFSET,
                    TEMU_PCI_CONFIG_LATENCY_TIMER_SIZE, Val);
}

static inline void
temu_pciSetCacheLineSize(temu_PCIConfig C, uint8_t Val)
{
  temu_pciSetConfig(C, TEMU_PCI_CONFIG_CACHELINE_SIZE_OFFSET,
                    TEMU_PCI_CONFIG_CACHELINE_SIZE_SIZE, Val);
}

static inline void
temu_pciSetInterruptPin(temu_PCIConfig C, uint8_t Val)
{
  assert(Val < 0x05);
  temu_pciSetConfig(C, TEMU_PCI_TYPE_00_INTERRUPT_PIN_OFFSET,
                    TEMU_PCI_TYPE_00_INTERRUPT_PIN_SIZE, Val);
}

static inline void
temu_pciSetSubsystemId(temu_PCIConfig C, uint16_t Val)
{
  assert(temu_pciGetConfig(C, TEMU_PCI_CONFIG_HEADER_TYPE_OFFSET,
                           TEMU_PCI_CONFIG_HEADER_TYPE_SIZE) == 0);
  temu_pciSetConfig(C, TEMU_PCI_TYPE_00_SUBSYSTEM_ID_OFFSET,
                    TEMU_PCI_TYPE_00_SUBSYSTEM_ID_SIZE, Val);
}

static inline void
temu_pciSetSubsystemVendorId(temu_PCIConfig C, uint16_t Val)
{
  assert(temu_pciGetConfig(C, TEMU_PCI_CONFIG_HEADER_TYPE_OFFSET,
                           TEMU_PCI_CONFIG_HEADER_TYPE_SIZE) == 0);
  temu_pciSetConfig(C, TEMU_PCI_TYPE_00_SUBSYSTEM_VENDOR_ID_OFFSET,
                    TEMU_PCI_TYPE_00_SUBSYSTEM_VENDOR_ID_SIZE, Val);
}

#endif // !TEMU_BUS_PCI_H
