//===------------------------------------------------------------*- C++ -*-===////
// TEMU: The Terma Emulator
// (c) Terma 2016
// Authors: Alberto Ferrazzi <albf (at) terma.com>
//          Mattias Holm <maho (at) terma.com>
//
//===----------------------------------------------------------------------===//

/*!
  This is the spacewire device modelling API. It provides an interface
  to be used for implementing routers, spacewire controllers and
  devices.

  The spacewire device interface is EXPERIMENTAL at the moment.
 */

#ifndef TEMU_SPACEWIRE_H
#define TEMU_SPACEWIRE_H


#include "temu-c/Support/Objsys.h"
#include "temu-c/Support/Buffer.h"

#include <assert.h>
#include <stdbool.h>
#include <stdint.h>

#ifdef __cplusplus
extern "C" {
#endif


typedef enum {
  teSMT_Data = 1,  // Normal data packet
  teSMT_Err = 2,   // EEP packet marker
  teSMT_Time = 3,  // Time code
} temu_SpwPacketType;


typedef struct temu_SpwPacket {
  temu_SpwPacketType MsgType;
  temu_Buff PktData;

  // Store Rmap crc fast value.
  // Bit 1 signal an RMAP crc error in header.
  // Bit 2 signal an RMAP crc error in data.
  uint8_t Flags;
} temu_SpwPacket;

#define TEMU_SPW_FLAG_RMAP_HCRC_ERROR 0x01
#define TEMU_SPW_FLAG_RMAP_DCRC_ERROR 0x02

#define TEMU_SPW_PORT_IFACE_TYPE "SpwPortIface"

typedef struct temu_SpwPortIface temu_SpwPortIface;
TEMU_IFACE_REFERENCE_TYPE(temu_SpwPort);

// Space wire link states.
typedef enum {
  teSPWLS_ErrorReset = 0,
  teSPWLS_ErrorWait = 1,
  teSPWLS_Ready = 2,
  teSPWLS_Started = 3,
  teSPWLS_Connecting = 4,
  teSPWLS_Run = 5
} temu_SpwLinkState;

/*!
 * SpaceWire port interface.
 * A model must implement this interface for each SpaceWire port.
 */
struct temu_SpwPortIface {
  /*!
   * Implement to receive and handle the SpaceWire packet.
   * This will be called by the model on the other end that sends or
   * forwards the packet.
   */
  void (*receive)(void *Device, void *Sender, temu_SpwPacket *Pkt);

  /*!
   * The other end device (A) uses this to inform this device (B) about
   * its (A) change of link state.
   */
  void (*signalLinkStateChange)(void* Device, temu_SpwLinkState LinkState);

  /*!
   * Should return the link state of the device.
   * Called by the other end device to handle connection.
   */
  temu_SpwLinkState (*getOtherSideLinkState)(void* Device);

  /*!
   * Connect a device to this port.
   */
  void (*connect)(void *Device, temu_SpwPortIfaceRef Dest);

  /*!
   * Disconnects the device currently connected to this port.
   */
  void (*disconnect)(void *Device);

  /**
   * Return the amount of time required to send a packet through the port,
   * in nano seconds.
   */
  uint64_t (*timeToSendPacketNs)(void* Device, uint64_t PacketLength);
};

/*!
 * Connect two SpaceWire ports.
 */
static inline void
temu_spwConnect(temu_SpwPortIfaceRef Dev1PortIface,
                temu_SpwPortIfaceRef Dev2PortIface)
{
  Dev1PortIface.Iface->connect(Dev1PortIface.Obj, Dev2PortIface);
  Dev2PortIface.Iface->connect(Dev2PortIface.Obj, Dev1PortIface);
}


/*!
 * Disconnect two SpaceWire ports.
 */
static inline void
temu_spwDisconnect(temu_SpwPortIfaceRef Dev1PortIface,
                   temu_SpwPortIfaceRef Dev2PortIface)
{
  Dev1PortIface.Iface->disconnect(Dev1PortIface.Obj);
  Dev2PortIface.Iface->disconnect(Dev2PortIface.Obj);
}

/*!
 * Initialize the link state.
 */
void temu_spwLsmInit(temu_SpwLinkState *StatePtr);

/*!
 * Updates the link state.
 */
uint8_t temu_spwLsmUpdate(temu_SpwLinkState *StatePtr, uint8_t AS,
                          uint8_t LS, uint8_t LD, uint8_t PortConnect,
                          temu_SpwLinkState otherSideLinkState);

/*!
 * Returns the string name for the link state.
 */
const char * temu_spwLinkStateToStr(uint8_t linkState);

#define TEMU_SPW_BITS_PER_DATA_CHAR 10
#define TEMU_SPW_BITS_PER_CTRL_CHAR 4
#define TEMU_SPW_EOP_BITS TEMU_SPW_BITS_PER_CTRL_CHAR
#define TEMU_SPW_FCT_BITS TEMU_SPW_BITS_PER_CTRL_CHAR
#define TEMU_SPW_EEP_BITS TEMU_SPW_BITS_PER_CTRL_CHAR
#define TEMU_SPW_ESC_BITS TEMU_SPW_BITS_PER_CTRL_CHAR

static inline uint64_t
temu_spwBitsForData(uint64_t Bytes)
{
  return Bytes * TEMU_SPW_BITS_PER_DATA_CHAR + TEMU_SPW_EOP_BITS;
}

// Standard protocol IDs
#define TEMU_SPW_PROT_EXT        0
#define TEMU_SPW_PROT_RMAP       1
#define TEMU_SPW_PROT_CCSDSPKT   2
#define TEMU_SPW_PROT_GOES_R   238
#define TEMU_SPW_PROT_STUP     239

// Logical addresses
#define TEMU_SPW_LOGICAL_ADDRESS_DEFAULT 0xfe

typedef struct {
  uint8_t TargetLogicalAddr;
  uint8_t ProtocolId;
  uint8_t Instruction;
  uint8_t Key;
  uint8_t ReplyAddresses[12];
  uint8_t ReplyAddressesNum;
  uint8_t InitiatorLogicalAddr;
  uint16_t TransactionId;
  uint64_t Address;
  uint32_t DataLength;
} temu_SpwRmapDecodedCommandHeader;

typedef struct {
  uint8_t InitiatorLogicalAddr;
  uint8_t ProtocolId;
  uint8_t Instruction;
  uint8_t Status;
  uint8_t TargetLogicalAddr;
  uint16_t TransactionId;
  uint32_t DataLength;
} temu_SpwRmapDecodedReadReplyHeader;

typedef struct {
  uint8_t InitiatorLogicalAddr;
  uint8_t ProtocolId;
  uint8_t Instruction;
  uint8_t Status;
  uint8_t TargetLogicalAddr;
  uint16_t TransactionId;
} temu_SpwRmapDecodedWriteReplyHeader;

typedef enum {
  teSPWRMAPPT_Response = 0,
  teSPWRMAPPT_Command = 1,
  teSPWRMAPPT_Invalid = 2
} temu_SpwRmapPacketType;

typedef enum {
  teSPWRMAPCT_Read = 0,
  teSPWRMAPCT_Write = 1,
  teSPWRMAPCT_Rmw = 2
} temu_SpwRmapCommandType;

typedef struct {
  temu_SpwRmapDecodedCommandHeader Header;
} temu_SpwRmapReadCmdPacket;

typedef struct {
  temu_SpwRmapDecodedCommandHeader Header;
  /// Pointer to the first data char.
  const uint8_t *Data;
  /// The amount of data available.
  uint32_t AvailableDataLength;
  /// Data crc. Valid only if AvailableDataLength > Header.DataLength.
  uint8_t DataCrc;
} temu_SpwRmapWriteCmdPacket;

typedef struct {
  temu_SpwRmapDecodedCommandHeader Header;
  /// Size of the access.
  uint8_t AccessSize;
  /// Pointer to the first data char.
  const uint8_t *Data;
  /// Pointer to the first mask char.
  const uint8_t *Mask;
  /// The amount of data available.
  uint32_t AvailableDataLength;
  /// Data crc. Valid only if AvailableDataLength > Header.DataLength.
  uint8_t DataCrc;
} temu_SpwRmapRmwCmdPacket;

typedef struct {
  const uint8_t *Data;
  uint32_t Length;
  uint32_t Crc;
} temu_SpwRmapRawHeader;

typedef struct {
  temu_SpwRmapDecodedReadReplyHeader Header;
  /// Pointer to the first data char.
  const uint8_t *Data;
  /// The amount of data available.
  uint32_t AvailableDataLength;
  /// Data crc. Valid only if AvailableDataLength > Header.DataLength.
  uint8_t DataCrc;
} temu_SpwRmapDecodedReadReply;

typedef struct {
  temu_SpwRmapDecodedWriteReplyHeader Header;
} temu_SpwRmapDecodedWriteReply;

typedef struct {
  temu_SpwRmapDecodedReadReplyHeader Header;
  /// Pointer to the first data char.
  const uint8_t *Data;
  /// The amount of data available.
  uint32_t AvailableDataLength;
  /// Data crc. Valid only if AvailableDataLength > Header.DataLength.
  uint8_t DataCrc;
} temu_SpwRmapDecodedRmwReply;

typedef enum {
  teSPWRMAPDP_Invalid = 0,
  teSPWRMAPDP_ReadCmd  = 1,
  teSPWRMAPDP_WriteCmd = 2,
  teSPWRMAPDP_RmwCmd = 3,
  teSPWRMAPDP_ReadReply = 4,
  teSPWRMAPDP_WriteReply  = 5,
  teSPWRMAPDP_RmwReply  = 6
} temu_SpwRmapDecodedPacketType;

typedef struct {
  temu_SpwRmapCommandType CmdType;
  uint8_t Value;
  uint8_t Write     : 1;
  uint8_t Verify    : 1;
  uint8_t Ack       : 1;
  uint8_t Inc       : 1;
} temu_SpwRmapDecodedCmdField;

typedef struct {
  /// Total size of the packet received.
  uint32_t TotalSize;
  /// The packet type as identified by bits [7,6] in instruction.
  temu_SpwRmapPacketType PacketType;
  /// The command field, bits [5,4,3,2] in instruction.
  temu_SpwRmapDecodedCmdField CmdField;
  /// Raw header data access.
  temu_SpwRmapRawHeader RawHeader;

  // Decoded packet.
  temu_SpwRmapDecodedPacketType DecodedPacketType;
  union {
    temu_SpwRmapReadCmdPacket ReadCmd;
    temu_SpwRmapWriteCmdPacket WriteCmd;
    temu_SpwRmapRmwCmdPacket RmwCmd;
    temu_SpwRmapDecodedReadReply ReadReply;
    temu_SpwRmapDecodedWriteReply WriteReply;
    temu_SpwRmapDecodedRmwReply RmwReply;
  };
} temu_SpwRmapDecodedPacket;

typedef enum {
  teSPWRMAPDO_NoError = 0,
  teSPWRMAPDO_HeaderIncomplete = 1,
  teSPWRMAPDO_InvalidPacketType = 2
} temu_SpwRmapDecodingOutcome;

/*!
 * Provided a SpaceWire Rmap packet attempts to decode it.
 */
temu_SpwRmapDecodingOutcome
temu_spwRmapDecodePacket(const temu_SpwPacket *Pkt,
                         temu_SpwRmapDecodedPacket *PktDecoded);
/*!
 * Provided a buffer containing a SpaceWire Rmap packet attempts to decode it.
 */
temu_SpwRmapDecodingOutcome
temu_spwRmapDecodeBuffer(const temu_Buff *PktDataBuffer,
                         temu_SpwRmapDecodedPacket *PktDecoded);

/*!
 * Returns the total packet-size required to reply to the command.
 */
uint32_t
temu_spwRmapHeaderReplySize(const temu_SpwRmapDecodedPacket *DCmdPkt);

/*!
 * Encodes the reply for a read command.
 */
uint32_t
temu_spwRmapEncodeReadReplyHeaderForPacket(
  const temu_SpwRmapDecodedPacket *DCmdPkt,
  uint8_t *Data, uint32_t AllocatedDataSize,
  uint8_t Status,
  uint32_t DataLength);

/*!
 * Encodes the reply for a rmw command.
 */
uint32_t
temu_spwRmapEncodeRmwHeaderForPacket(
  const temu_SpwRmapDecodedPacket *DCmdPkt,
  uint8_t *Data, uint32_t AllocatedDataSize,
  uint8_t Status,
  uint32_t DataLength);

/*!
 * Encodes the reply for a write command.
 */
uint32_t
temu_spwRmapEncodeWriteReplyHeaderForPacket(
  const temu_SpwRmapDecodedPacket *DCmdPkt,
  uint8_t *Data, uint32_t AllocatedDataSize,
  uint8_t Status);

/*!
 * Provided the previous calculated crc and a the current byte
 * returns the next CRC value.
 */
uint8_t
temu_spwRmapCRCNextCode(uint8_t InCRC, uint8_t InByte);

/*!
 * Calculates the CRC over the specified data.
 */
uint8_t
temu_spwRmapCRC(const uint8_t *Data, uint32_t DataSize);

#define TEMU_SPW_RMAP_INST_CMD     0x40;
#define TEMU_SPW_RMAP_INST_WRITE   0x20;
#define TEMU_SPW_RMAP_INST_VERIFY  0x10;
#define TEMU_SPW_RMAP_INST_ACK     0x8;
#define TEMU_SPW_RMAP_INST_INC     0x4;

#ifdef __cplusplus
}
#endif

#endif /* ! TEMU_MIL_STD_1553_H */
