//===------------------------------------------------------------*- C++ -*-===//
//
// TEMU: The Terma Emulator
// (c) Terma 2015
// Authors: Mattias Holm <maho (at) terma.com>
//
//===----------------------------------------------------------------------===//

#ifndef TEMU_MEMORY_IF_H
#define TEMU_MEMORY_IF_H

#include "temu-c/Support/Memory.h"
#include "temu-c/Support/Objsys.h"
#include <stdbool.h>
#include <stdint.h>

#ifdef __cplusplus
extern "C" {
#endif

typedef enum {
  teIT_Cpu, //!< Memory transaction initiated by CPU
  teIT_Device, //!< Memory transaction initiated by device
  teIT_Unknown, //!< Memory transaction initiator unknown
} temu_InitiatorType;
typedef enum {
  teME_BigEndian, //!< Memory access interface in big endian
  teME_LittleEndian, //!< Memory access interface in little endian
  teME_UnspecifiedEndian, //!< Memory access interface in undefined endian
} temu_MemoryEndianness;
// Memory transaction flags
#define TEMU_MT_CACHEABLE 1
#define TEMU_MT_BYPASS_CACHE (1 << 1)
#define TEMU_MT_FORCE_CACHE_MISS (1 << 2)
#define TEMU_MT_FAILED (1 << 3)
#define TEMU_MT_CACHE_HIT (1 << 4)

#define TEMU_MT_PRIV_SHIFT (5)

#define TEMU_MT_PRIV_MASK (7 << 5)
#define TEMU_MT_PRIV_USER (0 << 5)
#define TEMU_MT_PRIV_SUPER (1 << 5)
#define TEMU_MT_PRIV_HYPER (2 << 5)
// Note that we reserve an extra privilege level bit for future extension

// Probing memory transaction, this is used to probe the memory system without
// actually triggering any faults
#define TEMU_MT_PROBE (1 << 8)
#define TEMU_MT_SELF_MODIFYING (1 << 9)

// Bit 10 is used to indicate the endianness of the transaction.
// The memory space will automatically swap the bytes when it arrives at a
// device that is mapped as the opposite endianness. Note that swapping is done
// when arriving at the end point device, not when passing through nested memory
// spaces. Swapping is only done for small transactions, larger transactions
// must be swapped by the the target device itself. For large transactions we
// also need a "target endianness" / "host endianness".

// Small transactions (i.e. those where the payload fit in Value):
//   Always in host endianness when initiating, but have a "virtual endianness".
//   Memory space swaps the endianness automatically.
// Large transactions:
//   The value actually contains a pointer to a datablock
//   The datablock may be in host endianness, or in target endianness
//   Used for for example DMA transfers or transfers of bytes in the sim (image loading etc)
//   Usually:
//     - Data transfer in bytes of target endianness (image loading)
//     - Transfers of memory blocks of words which the sim need to interpret (send lists)
//     - Transfers of memory blocks of words of data
#define TEMU_MT_ENDIAN_BIT 10
#define TEMU_MT_BIG_ENDIAN (0 << 10)
#define TEMU_MT_LITTLE_ENDIAN (1 << 10)
static inline bool
temu_hasDifferentEndianness(uint32_t a, uint32_t b)
{
  return ((a ^ b) >> 10) & 1;
}

#define TEMU_MT_ISA_0 (0 << 11)
#define TEMU_MT_ISA_1 (1 << 11)
#define TEMU_MT_ISA_2 (2 << 11)

#define TEMU_MT_GET_ISA(flags) ((flags >> 11) & 7)

#define TEMU_MT_PROF_ACCESS (1 << 13)
#define TEMU_MT_PROF_ACCESS_BIT 13

// Bypass builtin protections (e.g. write protections for ROM memories)
// This simplifies the use of large transactions for implementing ROM loading functionality
#define TEMU_MT_BYPASS_PROT (1 << 14)

#define TEMU_MT_READ (1 << 15)
#define TEMU_MT_WRITE (1 << 16)

// Lock flags controlling split memory transaction locks.
// These are especially important for atomics, where the read op should lock,
// and the write op should release.
#define TEMU_MT_LOCK_ACQUIRE (1 << 17)
#define TEMU_MT_LOCK_RELEASE (1 << 18)
#define TEMU_MT_LOCK_ACQUIRE_AND_RELEASE (3 << 17)
#define TEMU_MT_CHECK_FETCH_ATTRIBS (1 << 19)

#define TEMU_MT_ALLOC_IR (1 << 20)
#define TEMU_MT_FORCE_REBIND (1 << 21)

#define TEMU_MT_RESET_ENTRY (1 << 22)

//! Generic memory transaction.
//!
//! This type is kept in sync with the emulator core. The layout is guaranteed.
//! and will remain as is, although more fields may be added (at the bottom).
//!
//! When the emulator core issues a memory transaction (assuming no ATC hit),
//! the core allocates this structure on the stack and fills in some of the
//! fields with default values. The memory transaction is passed by pointer or
//! reference. By filling in the different fields you can adapt the result of
//! the memory transaction.
typedef struct temu_MemTransaction {
  uint64_t Va; //!< 64 bit virtual for unified 32/64 bit interface.
  uint64_t Pa; //!< 64 bit physical address

  //! Resulting value (or written value). On MMIO Reads the model
  //! fills in this value, and on writes the written value will be
  //! stored here.
  uint64_t Value;

  //! Two-logarithm of the size of the transaction in bytes it is at
  //! most the size of the CPUs max bus size. In case of SPARCv8, this
  //! is 4 bytes (double words are issued as two accesses).  As this
  //! is the 2-log of the size in bytes, a single byte access will
  //! have a size of 0, a 2 byte transaction will have size 1, a 4
  //! byte transaction will have size 2 and an 8 byte transaction will
  //! have size 3.
  //!
  //! In TEMU3 this field was changed to an uint64_t from uint8_t. This
  //! as it does not add any additional space. And we can repurpose value and
  //! size as follows:
  //!   - The lower 2 bits define the base unit of the transaction (same as
  //!     before), 0 => 1 byte, 1 => 2 bytes, 2 => 4 bytes, 3 => 8 bytes.
  //!   - The upper bits define the number of transferred units 0 implies
  //!     one unit.
  //! If the transferred units is more than 0, we are dealing with a large
  //! transaction. These can be used by e.g. RAM models for block transfers etc.
  //! In that case the Value field is to be reinterpreted as a pointer to the
  //! location of the data. This means that we can use the memory access
  //! interface to e.g. read out send lists and similar items.
  //!
  //! The memory space must thus have a way of determining which type of
  //! transaction is legal. The MemoryAccessIface has been extended with a
  //! capability query, which if implemented has the ability to query for
  //! supported features.
  uint64_t Size;

  //! Used for device models, this will be filled in with the offset
  //! from the start address of the device (note it is in practice
  //! possible to add a device at multiple locations (which happens in
  //! some rare cases)).
  uint64_t Offset;

  //! InitiatorType identifies the type of object starting the transaction.
  //! this is only relevant when Initiator is set to non-null, and allows
  //! for the specification of device pointers in the initator. This field
  //! is new in TEMU 2.2. The field was introduced to support the
  //! implementation of I/O MMUs.
  temu_InitiatorType InitiatorType;
  //! Initiator of the transaction (a CPU object). It can be null,
  //! which indicate that the transaction was not initiated by normal
  //! CPU activity (e.g. fetch, read or write). When the initiator is
  //! null, models should not attempt to stop the processor core, go
  //! to idle mode or raise traps (posting events is fine). An example
  //! when the initiator is null is when an MMU does a table walk. The
  //! MMU will normally special case handle table walks which access
  //! un-mapped memory.
  temu_Object_ *Initiator;
  //! Page pointer (for caching), this can be filled in by a memory
  //! model to have the emulator core inject the page translation in
  //! the ATC. If a model sets this, the page pointer will
  //! automatically be cleared if the page has attributes
  //! (breakpoints, etc). Models that implement normal memory mapped
  //! registers should NOT populate the page pointer.
  void *Page;
  uint64_t Cycles; //!< Cycle cost for memory access (initialised to 0).
  uint32_t Flags;  //!< Flags for use in the memory hierarchy.

  void *IR; //!< Intermediate code for interpreter (internally managed do not
            //!< modify)
  void *Meta;
} temu_MemTransaction;

// Fetch capabilities
#define TEMU_MEM_ACCESS_CAP_F8 (1 << 0)
#define TEMU_MEM_ACCESS_CAP_F16 (1 << 1)
#define TEMU_MEM_ACCESS_CAP_F32 (1 << 2)
#define TEMU_MEM_ACCESS_CAP_F64 (1 << 3)
#define TEMU_MEM_ACCESS_CAP_F_ALL (0xf)

// Read
#define TEMU_MEM_ACCESS_CAP_R8 (1 << (0 + 4))
#define TEMU_MEM_ACCESS_CAP_R16 (1 << (1 + 4))
#define TEMU_MEM_ACCESS_CAP_R32 (1 << (2 + 4))
#define TEMU_MEM_ACCESS_CAP_R64 (1 << (3 + 4))
#define TEMU_MEM_ACCESS_CAP_R_ALL (0xf << 4)

// Write
#define TEMU_MEM_ACCESS_CAP_W8 (1 << (0 + 8))
#define TEMU_MEM_ACCESS_CAP_W16 (1 << (1 + 8))
#define TEMU_MEM_ACCESS_CAP_W32 (1 << (2 + 8))
#define TEMU_MEM_ACCESS_CAP_W64 (1 << (3 + 8))
#define TEMU_MEM_ACCESS_CAP_W_ALL (0xf << 8)

// Exchange
#define TEMU_MEM_ACCESS_CAP_E8 (1 << (0 + 12))
#define TEMU_MEM_ACCESS_CAP_E16 (1 << (1 + 12))
#define TEMU_MEM_ACCESS_CAP_E32 (1 << (2 + 12))
#define TEMU_MEM_ACCESS_CAP_E64 (1 << (3 + 12))
#define TEMU_MEM_ACCESS_CAP_E_ALL (0xf << 12)

typedef struct {
  uint16_t TransactionBaseSizes; //!< Flags indicating legal transaction sizes
  uint16_t
      LargeTransactions : 1; //!< Supports large transactions (e.g. RAM/ROM)
  temu_MemoryKind Kind; //!< Mapping type
  temu_MemoryEndianness Endianness; //!< Endianess of transaction interface
} temu_MemAccessCapabilities;

//! Memory access interface implemented by all memory mapped devices
//! Exposed to the emulator core by a memory object.
struct temu_MemAccessIface {
  //! Function called on fetches. The function can be null in case
  //! fetches are not allowed from the model.
  void (*fetch)(void *Obj, temu_MemTransaction *Mt);

  //! Function called on reads.
  void (*read)(void *Obj, temu_MemTransaction *Mt);

  //! Function called on writes.
  void (*write)(void *Obj, temu_MemTransaction *Mt);

  //! Function called on atomic exchanges, by default if this is not
  //! defined, the memory space will call read followed by write in
  //! order.
  void (*exchange)(void *Obj, temu_MemTransaction *Mt);

  //! Optional method, called when interface is mapped
  void (*mapped)(void *Obj, uint64_t Pa, uint64_t Len);

  //! Query for supported features
  //! Function is optional. By default the assumption is that
  //! the base capabilities are equal to R_ALL | W_ALL, custom memories
  //! must explicitly register itself as fetchable memory.
  //! in addition, the default assumption is that the device does not
  //! support large transactions.
  const temu_MemAccessCapabilities *(*getCapabilities)(void *Obj);

  //! Probe access
  //! Must be implemented for RAM and ROM models
  //! When probing the memory subsystem,
  //! the transaction is not supposed to trigger semantics.
  //! Instead the models should populate the memory transaction contents,
  //! with metadata.
  //! Thus a probe should for example fill in the page pointer, IR pointer, etc.
  //! To support correct probing for atomic transactions, probes should handle
  //! the TEMU_MT_READ and TEMU_MT_WRITE bits.
  void (*probe)(void *Obj, temu_MemTransaction *Mt);
};
#define TEMU_MEM_ACCESS_IFACE_TYPE "MemAccessIface"
TEMU_IFACE_REFERENCE_TYPE(temu_MemAccess);

//! For objects which have actualm memory (not just registers) This is
//! for the simulator (not the emu core). The procedures should write
//! the data given in bytes to the given physical offset. The offset
//! is a 64 bit uint to support 64 bit targets. The interface is used
//! for example by DMA transactions.
//!
//! The size argument is in bytes.
//!
//! The swap argument is used to swap bytes to the host
//! endianess. Specify the log size of the read data types.
//!   - 0:  We are reading bytes (bytes will be in target memory order)
//!   - 1:  We are reading half words (will be swapped to host order)
//!   - 2:  We are reading words (will be swapped)
//!   - 3:  We are reading double words (will be swapped)
//! With 0 for swap, we are basically reading a byte array
//!
//! readBytes and writeBytes should return the number of bytes
//! read / written or negative on error.

typedef struct temu_MemoryIface {
  int (*readBytes)(void *Obj, void *Dest, uint64_t Offs, uint32_t Size,
                   int Swap);
  int (*writeBytes)(void *Obj, uint64_t Offs, uint32_t Size, const void *Src,
                    int Swap);
} temu_MemoryIface;
#define TEMU_MEMORY_IFACE_TYPE "MemoryIface"
TEMU_IFACE_REFERENCE_TYPE(temu_Memory);

/*!
 * Map memory object into address space
 * \param Obj The memory space object
 * \param Addr Physical address where to map the device
 * \param Len Length in bytes of area where the object is mapped.
 * \param MemObj The memory object. This object must correspond to the
 *               MemAccessIface
 * \result Zero on success, other values indicates that the mapping failed
 */
int temu_mapMemorySpace(void *Obj, uint64_t Addr, uint64_t Len,
                        temu_Object_ *MemObj);

/*!
 * Map memory object into address space with flags
 * \param Obj The memory space object
 * \param Addr Physical address where to map the device
 * \param Len Length in bytes of area where the object is mapped.
 * \param MemObj The memory object. This object must correspond to the
 *               MemAccessIface
 * \param Flags Sticky flags for memory accesses to that object (e.g. cachable)
 * \result Zero on success, other values indicates that the mapping failed
 */
int temu_mapMemorySpaceFlags(void *Obj, uint64_t Addr, uint64_t Len,
                             temu_Object_ *MemObj, uint32_t Flags);

/*!
 * Set attribute on memory space location
 * \param Obj The memory space object
 * \param Addr Physical address where to map the device
 * \param Len Length in bytes of area where the attribute should be set.
 * \param Attr The attribute to set.
 */

void temu_setMemAttr(void *Obj, uint64_t Addr, uint64_t Len,
                     temu_MemoryAttr Attr);

/*!
 * Clear attribute on memory space location
 * \param Obj The memory space object
 * \param Addr Physical address where to map the device
 * \param Len Length in bytes of area where the attribute should be set.
 * \param Attr The attribute to clear.
 */
void temu_clearMemAttr(void *Obj, uint64_t Addr, uint64_t Len,
                       temu_MemoryAttr Attr);


int temu_memoryExchangeTransaction(void *Obj, temu_MemTransaction *MT);
int temu_memoryWriteTransaction(void *Obj, temu_MemTransaction *MT);
int temu_memoryReadTransaction(void *Obj, temu_MemTransaction *MT);
int temu_memoryFetchTransaction(void *Obj, temu_MemTransaction *MT);

#ifdef __cplusplus
}
#endif

#endif /* ! TEMU_MEMORY_IF_H */
