//===-- temu-c/Scheduler.h - TEMU Scheduler API -----------------*- C++ -*-===//
//
// TEMU: The Terma Emulator
// (c) Terma 2023
// Authors: Mattias Holm <maho (at) terma.com>
//
//===----------------------------------------------------------------------===//
#ifndef TEMU_SCHEDULER_H
#define TEMU_SCHEDULER_H
#include <stdint.h>
#include "temu-c/Support/Attributes.h"

#ifdef __cplusplus
extern "C" {
#endif

typedef struct temu_TimeSource temu_TimeSource;
typedef struct temu_Object temu_Object;

typedef enum temu_ExecutionState {
  teES_Stopped,
  teES_Running,
  teES_Stepping,
} temu_ExecutionState;


/*!
 * Run the current scheduler for a fixed number of steps
 * 
 * \param steps Time in steps (in scheduler units).
 */
TEMU_API void temu_runForSteps(int64_t steps);


/*!
 * Run the current scheduler for a fixed time
 * 
 * \param secs Time in simulated seconds
 */
TEMU_API void temu_runForTime(double secs);

/*!
 * Run the current scheduler forever
 */
TEMU_API void temu_run();

/*!
 * Step a processor in the current scheduler
 *
 * Note that, the other processors may run if the step exceeds the quanta.
 * The step function currently runs processors in sequence,
 * it is not running processors in parallel.
 * This means that stepping is always deterministic.
 * 
 * \param cpu The processor to step
 * \param steps The number of steps or instructions to execute
 */
TEMU_API void temu_step(temu_TimeSource *cpu, int steps);

/*!
 * Stop the current scheduler
 *
 * The function signals the scheduler to stop,
 * and then waits until the scheduler is in the stopped state.
 */
TEMU_API void temu_stop();
/*!
 * Stop the current scheduler
 * 
 * The function returns immediately, before the scheduler has stopped.
 * The function is suitable for use in e.g. signal handlers.
 */
TEMU_API void temu_asyncStop();

/*!
 * Get the current execution status of TEMU
 *
 * \result Current execution status
 */

TEMU_API temu_ExecutionState temu_getExecutionState();

/*!
 * Get the number of processors in the global scheduler.
 *
 * \result Number of processors in global scheduler.
 */
TEMU_API unsigned temu_getScheduledProcessorCount();

/*!
 * Get processor in global scheduler
 * \param cpu Processor number to get
 * \result Processor pointer as a time source.
 */
TEMU_API temu_TimeSource* temu_getScheduledProcessor(unsigned cpu);

/*!
 * Serialize execution for some time when notification is triggered
 * \param notificationSource Object emitting notification.
 * \param notificationName Name of notification to intercept.
 * \param seconds Number of seconds (simulated time, for which to serialize execution).
 */
TEMU_API void
temu_serializeOnNotification(temu_Object *notificationSource,
                             const char *notificationName, double seconds);


/*!
 * Experimental function to get the accumulated nanoseconds for the scheduler.
 * 
 * To enable sampling the Scheduler-object's `measureTime` property must be set.
 * \result Accumulated nanoseconds of which temu_run* functions have run.
 */
TEMU_API uint64_t temu_schedulerGetAccumulatedTime();
/*!
 * Experimental function to get the accumulated nanoseconds
 * for the scheduler synchronization.
 * 
 * To enable sampling, the Scheduler-object's `measureTime` property must be set.
 * \result Accumulated nanoseconds of which synchronisation code was run.
 */
TEMU_API uint64_t temu_schedulerGetSyncTime(unsigned threadId);

/*!
 * Experimental function to get the accumulated nanoseconds
 * for the scheduler barrier waits.
 * 
 * The wait time, includes both the actual barrier and the synchronization time.
 * 
 * So the true wait time is the wait - sync time.
 * 
 * To enable sampling, the Scheduler-object's `measureTime` property must be set.
 * \result Accumulated nanoseconds of which the wait was run.
 */
TEMU_API uint64_t temu_schedulerGetWaitTime(unsigned threadId);

/*!
 * Experimental function to set thread affinity for the scheduler.
 *
 * The function should be called before setting the number of threads,
 * as threads automatically set the affinity if the hostCore is zero or above.
 * 
 * \param threadId Thread ID starting with 0 to the number of threads
 *                 set for the scheduler.
 * \param hostCore Host core id to bind thread to. Negative to not set affintity.
 */
TEMU_API void temu_schedulerSetAffinity(unsigned threadId, int hostCore);

/*!
 * Experimental function to get the thread affinity for the scheduler.
 *  
 * \param threadId Thread ID starting with 0 to the number of threads
 *                 set for the scheduler.
 * \result Affinity value, -1 implies not set. Other values indicate HOST CPUID.
 */
TEMU_API int64_t temu_schedulerGetAffinity(unsigned threadId);


#ifdef __cplusplus
} // extern c
#endif

#endif /* ! TEMU_REGISTER_H */
