//===-- temu-c/Cpu.h - CPU Interfaces ---------------------------*- C++ -*-===//
//
// TEMU: The Terma Emulator
// (c) Terma 2015
// Authors: Mattias Holm <maho (at) terma.com>
//
//===----------------------------------------------------------------------===//

#ifndef TEMU_CPU_H
#define TEMU_CPU_H

#include "temu-c/Models/Power.h"
#include "temu-c/Support/Objsys.h"
#include <stdint.h>

#ifdef __cplusplus
extern "C" {
#endif

typedef enum temu_CpuState {
  teCS_Nominal, //!< Normal all ok CPU state
  teCS_Halted,  //!< Halted CPU (e.g. SPARC error mode), the CPU can go to
                //!< the normal state using a reset
  teCS_Idling,  //!< The CPU is in idle mode. It will not run
                //!< instructions, only advance the CPUs event queue
                //!< (until the CPU moves to another mode).
} temu_CpuState;

typedef enum temu_CpuExitReason {
  teCER_Normal = 0, //!< Normal exit (cannot be passed to early exit)
  teCER_Trap = 2,   //!< Exited due to trap (sync trap)
  teCER_Halt,       //!< Exited due to halting (e.g. sparc error mode)
  teCER_Event,      //!< Exited due to synchronised event (internally,
                    //!< returned for any event)
  teCER_Break,      //!< Exited due to breakpoint hit
  teCER_WatchR,     //!< Exited due to watchpoint read hit
  teCER_WatchW,     //!< Exited due to watchpoint write hit
  teCER_Early,      //!< Other early exit reason
  teCER_Panic,      //!< Emulator panic (e.g. illegal mode transition)
  teCER_Sync,       //!< Instruction cache sync operation
} temu_CpuExitReason;

// ATC flags
#define TEMU_ATC_FETCH 1
#define TEMU_ATC_READ (1 << 1)
#define TEMU_ATC_WRITE (1 << 2)
#define TEMU_ATC_USER (1 << 3)
#define TEMU_ATC_SUPER (1 << 4)
#define TEMU_ATC_HYPER (1 << 5)

// Some types for querying CPU info

//!< Endianness of target architecture
typedef enum temu_Endian {
  teEN_Little,  //!< Always little endian
  teEN_Big,     //!< Always big endian
  teEN_Dynamic, //!< Can switch at runtime
} temu_Endian;

/*!
  CPU architecture info that can be queried by the user.
 */
typedef struct {
  const char *ArchName;  //!< Architecture name
  const char *ModelName; //!< Processor model name

  unsigned VASize;     //!< Virtual address size in bits
  unsigned PASize;     //!< Physical address size in bits
  unsigned VATypeSize; //!< Virtual address type size in bytes
  unsigned PATypeSize; //!< Physical address type size in bytes (i.e. 8 for 36
                       //!< bit PA)

  unsigned GPRCount; //!< GPR register count
  unsigned FPRCount; //!< FPR register count

  temu_Endian InstructionEndianess; //!< Instruction endianness
  temu_Endian DataEndianess;        //!< Data endianness

  unsigned NumInstructionSets;      //!< Number of instruction sets
  const char **InstructionSetNames; //!< Names of instruction sets
} temu_CpuInfo;

// Named struct type here keeps libclang happy

/*!
  Common CPU interface.

  The CPU interface provides common functionality that all processors
  must implement. This includes the reset and run methods. But also
  different register access functions. The register access functions
  are functions, because registers may be banked and we want some type
  of common interface that can access the current registers. Note that
  the interface currently only support 32 bit processors.

  \warning Some functions will not return and execute longjmps to the
  emulation loop instead. These functions should NOT be called from
  any C++ code that needs destructors to be called on exit or from any
  events handlers.

  \field reset The function executing a reset. It takes as parameter
  the reset type, which is 0 for a default cold reset.

  \field run Run the main emulator loop for a given number of cycles.

  \field runUntil Run the main emulator loop until its cycle counter
                  reach the end cycles.

  \field step Run the given number of steps. A step is one
  instruction, completed or not. E.g. a trapping instruction does not
  complete but is counted as a step.

  \field stepUntil Step the emulator, but stop if the step exceeds a
                   certain fixed time.

  \field raiseTrap Raises a trap on the CPU. THIS FUNCTION DOES NOT
  RETURN!!!  \warning The raiseTrap function will normally be
  constructed using longjmp, that implies that you should not call
  this from a model which must properly unwind the stack and call
  destructors on the way.

  \field enterIdleMode Will call longjmp and enter idle mode
  immediately, this is useful for devices that activates power down
  mode. \warning This function does not return and destructors will
  not be called when the stack is unwound.

  \field exitEmuCore Will call longjmp to exit the emulator core. This
  can be used in certain cases where the other exit functions do not
  suite well. One being that a reset is called from an MMIO write or
  read, or from a watchdog event. The reset can otherwise be called
  when the emulator is not running, so in a model you can call reset
  and then exitEmuCore.

  \field getGpr Read the currently visible general purpose
  registers. For banked registers (e.g. SPARC reg windows), you should
  look at the arch specific interface instead.

  \field getFpr32 Read the currently visible floating point
  registers.

  \field setSpr Set special purpose registers. The indexes have been
  explicitly choosen to be equal to what is assumed by the GDB
  protocol, but re-based at zero.

  \field getSpr Read special purpose registers. The indexes have been
  explicitly choosen to be equal to what is assumed by GDB but rebased
  at zero.

  \field disassemble This function will disassemble an instruction.
  The function returns a heap-allocated string (allocated with
  malloc()). The caller is responsible for calling free() and managing
  the lifetime.

  \field invalidateAtc Invalidates the ATC cache for the given address
  range. Flags can be set to control the invalidation. Bit 0: don't
  invalidate fetch. Bit 1: don't Invalidate read, bit 2: don't
  invalidate write, bit 3 don't invalidate user, bit 4 don't
  invalidate super.

  \field translateAddress Does a table walk and translates the
  virtual address to physical page address. For MMU free systems, the
  function returns the Va masked with ~(page size-1). Otherwise, the
  return is the translated page address and in the case of failure -1.

  \field raiseTrapNoJmp Raises a trap without longjmp to emulator
                        core main loop. This can be used in e.g.
                        timed event handlers and when the core isn't
                        running.
*/
typedef struct temu_CpuIface {
  void (*reset)(void *Cpu, int ResetType);
  temu_CpuExitReason (*run)(void *Cpu, uint64_t Cycles);
  temu_CpuExitReason (*runUntil)(void *Cpu, uint64_t Cycles);
  temu_CpuExitReason (*step)(void *Cpu, uint64_t Steps);
  temu_CpuExitReason (*stepUntil)(void *Cpu, uint64_t Steps, uint64_t Cycles);

  void __attribute__((noreturn)) (*raiseTrap)(void *Obj, int Trap);
  void (*enterIdleMode)(void *Obj);
  void __attribute__((noreturn)) (*exitEmuCore)(void *Cpu,
                                                temu_CpuExitReason Reason);

  uint64_t (*getFreq)(void *Cpu);
  int64_t (*getCycles)(void *Cpu);
  int64_t (*getSteps)(void *Cpu);
  temu_CpuState (*getState)(void *Cpu);
  void (*setPc)(void *Cpu, uint64_t Pc);
  uint64_t (*getPc)(void *Cpu);
  void (*setGpr)(void *Cpu, int Reg, uint64_t Value);
  uint64_t (*getGpr)(void *Cpu, unsigned Reg);
  void (*setFpr32)(void *Cpu, unsigned Reg, uint32_t Value);
  uint32_t (*getFpr32)(void *Cpu, unsigned Reg);
  void (*setFpr64)(void *Cpu, unsigned Reg, uint64_t Value);
  uint64_t (*getFpr64)(void *Cpu, unsigned Reg);
  void (*setSpr)(void *Cpu, unsigned Reg, uint64_t Value);
  uint64_t (*getSpr)(void *Cpu, unsigned Reg);
  int (*getRegId)(void *Cpu, const char *RegName);
  const char *(*getRegName)(void *Cpu, int RegId);
  uint32_t (*assemble)(void *Cpu, const char *AsmStr);
  char *(*disassemble)(void *Cpu, uint32_t Instr);
  void (*enableTraps)(void *Cpu);
  void (*disableTraps)(void *Cpu);
  void (*invalidateAtc)(void *Obj, uint64_t Addr, uint64_t Pages,
                        uint32_t Flags);

  uint64_t (*translateAddress)(void *Cpu, uint64_t Va, uint32_t *Flags);

  temu_PowerState (*getPowerState)(void *Cpu);
  void (*setPowerState)(void *Cpu, temu_PowerState Ps);

  void (*enableTrapEvents)(void *Cpu);
  void (*disableTrapEvents)(void *Cpu);

  void (*enableErrorModeEvents)(void *Cpu);
  void (*disableErrorModeEvents)(void *Cpu);

  void *(*getMachine)(void *Cpu);
  void (*raiseTrapNoJmp)(void *Cpu, int Trap);

  const char *(*getTrapName)(void *Cpu, int Trap); //!< Get name for trap

  const temu_CpuInfo *(*getCPUInfo)(void *Cpu); //!< Experimental

  int (*wakeUp)(void *Cpu); //!<  Wake up CPU if it is idling, return 1 if woken
                            //!<  up 0 if no state change occurred

  void (*forceEarlyExit)(void *Cpu); //!< Force early return of core (after
                                     //!< event cleanup), unless the core
                                     //!< returns for normal reasons, the
                                     //!< emulator will return teCER_Early after
                                     //!< the current instruction.
  void *(*translateIRAddress)(
      void *Obj, uint64_t Va); //!< Calculate IR pointer from virtual address

  void (*enableModeSwitchEvents)(
      void *Obj); //!< Enable events when switching CPU mode (e.g. user to
                  //!< supervisor)
  void (*disableModeSwitchEvents)(
      void *Obj); //!< Disable events when switching CPU mode (e.g. user to
                  //!< supervisor)

  void (*enableProfiling)(void *Obj);    //!< Enable profiling mode
  void (*disableProfiling)(void *Obj);   //!< Disable profiling mode
  void (*flushProfileCaches)(void *Obj); //!< Flush profile caches

  int64_t (*getIdleSteps)(void *Cpu);  //!< Get the number of idle steps
  int64_t (*getIdleCycles)(void *Cpu); //!< Get the number of idle cycles
  void (*enterHaltedMode)(void *Obj);  //!< Make the processor enter halted mode
                                       //!< using a stack posted event.
  void (*forceSpecificExit)(
      void *Cpu, temu_CpuExitReason reason); //!< Force specific CPU exit

  void (*purgeDirtyPages)(void *Obj);    //!< Purge all dirty memory pages
  void (*invalidateFetchAtc)(void *Obj); //!< Invalidate ATC for fetches

  int (*translateAddressWithRoot)(
      void *Cpu, uint64_t Va, uint32_t *Flags, uint64_t RootPointer,
      uint64_t *PA); //!< MMU translation with a specific ROOT pointer

  temu_CpuExitReason (*synchronizingRun)(
      void *Obj,
      uint64_t
          Cycles); //!< Run function for use by scheduler, do not call directly.
  temu_CpuExitReason (*synchronizingRunUntil)(
      void *Obj,
      uint64_t
          Cycles); //!< Run function for use by scheduler, do not call directly.
  temu_CpuExitReason (*synchronizingStep)(
      void *Obj,
      uint64_t
          Steps); //!< Step function for use by scheduler, do not call directly.
  temu_CpuExitReason (*synchronizingStepUntil)(
      void *Obj, uint64_t Steps,
      uint64_t Cycles); //!< Step function for use by scheduler, do not call
                        //!< directly.

} temu_CpuIface;
#define TEMU_CPU_IFACE_TYPE "temu::CpuIface"
TEMU_IFACE_REFERENCE_TYPE(temu_Cpu);

typedef struct {
  uint32_t TrapId; //!< Trap number (architecture specific)
  uint64_t PC;     //!< Program counter when trap occurred
  uint64_t nPC;    //!< Only valid for targets with delay slots
} temu_TrapEventInfo;

typedef struct {
  uint32_t OldMode; //!< Old processor privilege level
  uint32_t NewMode; //!< New processor privilege level
} temu_ModeSwitchInfo;

typedef struct {
  int ResetType; //!< 0 == cold, 1 == warm reset
} temu_ResetInfo;

// Instruction classification bits
#define TEMU_INSTR_BRANCH (1 << 0)
#define TEMU_INSTR_INDIRECT_BRANCH (1 << 1)
#define TEMU_INSTR_LOAD (1 << 2)
#define TEMU_INSTR_STORE (1 << 3)
#define TEMU_INSTR_INTEGER (1 << 4)
#define TEMU_INSTR_FLOAT (1 << 5)
#define TEMU_INSTR_ARITHMETIC (1 << 6)
#define TEMU_INSTR_ANNULLED (1 << 7)
#define TEMU_INSTR_UNCOND (1 << 8)
#define TEMU_INSTR_UNCOND_NEVER (1 << 9)
#define TEMU_INSTR_ON_PAGE (1 << 10)
#define TEMU_INSTR_MODE_SWITCH (1 << 11)
#define TEMU_INSTR_ILLEGAL (1 << 12)
#define TEMU_INSTR_NO_DBT (1 << 13)

// Permanently unimplemented
#define TEMU_INSTR_UNIMP (1 << 14)

/*!
  Internal interface for binary translator
 */
typedef struct {
  //! Profile counter overflow handler
  void (*profileCounterOverflow)(void *Obj, uint64_t VA);
  //! Page written (for flushing caches)
  void (*wrotePage)(void *Obj, uint64_t Va, uint64_t Pa);
  //! Internal usage
  void *(*getRawRuntime)(void *Obj);
} temu_TargetExecutionIface;
#define TEMU_TARGET_EXEC_IFACE_TYPE "temu::TargetExecutionIface"
TEMU_IFACE_REFERENCE_TYPE(temu_TargetExecution);

#define TEMU_STICKY_DO_NOT_EXIT_AT_HALT 1
#define TEMU_STICKY_PROFILE_MODE 2

// TEMU_STICKY_DISABLE_IDLE is deprecated
#define TEMU_STICKY_DISABLE_IDLE (1 << 2)

/*!
  Interface for controlling the reset address used in a processor.

  Processors may in some cases have their reset addresses set dynamically.
  This interface expose such functionality to peripherals.
 */
typedef struct {
  //! Update reset address in processor
  void (*setResetAddress)(void *Obj, uint64_t Address);
  //! Get reset address in processor, optional method
  uint64_t (*getResetAddress)(void *Obj);
} temu_DynamicResetAddressIface;
#define TEMU_DYNAMIC_RESET_ADDRESS_IFACE_TYPE "temu::DynamicResetAddressIface"
TEMU_IFACE_REFERENCE_TYPE(temu_DynamicResetAddress);

/*!
  Statistics ID for controlling collection of individual statistics.
 */
typedef enum {
  teBTS_TranslatedInstructions, //!< Number of translated instructions
  teBTS_ExecutedInstructions,   //!< Number of executed instructions
  teBTS_TranslatedBlocks,       //!< Number of translated blocks
  teBTS_ExecutedBlocks,         //!< Number of executed blocks
  teBTS_CodeSize,               //!< Translated code size in bytes
} temu_BTStatID;

typedef struct {
  //! Enable translator for processor
  void (*enableBinaryTranslator)(void *Obj);
  //! Disable translator for processor
  void (*disableBinaryTranslator)(void *Obj);
  //! Set threshold (of call target execution) for triggering translation
  void (*setThreshold)(void *Obj, unsigned Threshold);
  //! Translate specific instruction range
  int (*translateInstructions)(void *Obj, uint64_t VA, uint64_t PA,
                               unsigned NumInstructions);
  //! Translate block (terminating with a branch, or end of page (special rules
  //! for delay slots apply)).
  int (*translateBlock)(void *Obj, uint64_t VA, uint64_t PA);
  //! Translate a complete function (determined to be statically reachable from
  //! the PA on this page).
  int (*translateFunc)(void *Obj, uint64_t VA, uint64_t PA);
  //! Manually chain blocks.
  int (*chainBlocks)(void *Obj, uint64_t SourceBlockPA, uint64_t TargetBlockPA,
                     int TakenArm);
  //! Disassemble block (using the host assembler)
  const char *(*disassembleBlock)(void *Obj, uint64_t PA);

  //! Remove a specific block and unlink incoming/outgoing chains
  int (*clearBlock)(void *Obj, uint64_t PA);
  //! Remove all translated blocks on a specific physical page
  int (*clearBlocksOnPage)(void *Obj, uint64_t PA);
  //! Enable collection of statistic in translated code
  void (*enableStatistics)(void *Obj, temu_BTStatID ID);
  //! Disable collection of statistic in translated code
  void (*disableStatistics)(void *Obj, temu_BTStatID ID);
  //! Get statistic
  uint64_t (*getStatistics)(void *Obj, temu_BTStatID ID);
  //! Reset statistics
  void (*clearStatistics)(void *Obj, temu_BTStatID ID);

} temu_BinaryTranslationControlIface;
#define TEMU_BINARY_TRANSLATION_CONTROL_IFACE_TYPE                             \
  "temu::BinaryTranslationControlIface"
TEMU_IFACE_REFERENCE_TYPE(temu_BinaryTranslationControl);

#ifdef __cplusplus
}
#endif

#endif /* ! TEMU_CPU_H */
