//===-- temu-c/CpuUnstable.h - Generic CPU Layout ---------------*- C++ -*-===//
//
// TEMU: The Terma Emulator
// (c) Terma 2024
// Authors: Mattias Holm <maho (at) terma.com>
//
//===----------------------------------------------------------------------===//
#ifndef TEMU_CPU_UNSTABLE_H
#define TEMU_CPU_UNSTABLE_H

#include "temu-c/Target/Cpu.h"
#include "temu-c/Support/Events.h"
#include "temu-c/Memory/Memory.h"
#include "temu-c/Models/Power.h"
#include "temu-c/Models/Reset.h"
#include "temu-c/Support/Events.h"
#include "temu-c/Target/Cpu.h"

#include <setjmp.h>
#include <stdint.h>
#include <stdlib.h>

#ifdef __cplusplus
extern "C" {
#endif

// These structures are unstable and should not be relied on in user code
// Types in this file are subject to:
//   - Additions
//   - Removal
//   - Change, even between patch versions

typedef struct {
  jmp_buf buf;
} temu_jmp_buf_t;

typedef enum {
  teIC_Alu,
  teIC_Mem,
  teIC_Branch,
  teIC_Float,
} temu_InstrClass;

typedef enum {
  teSC_AluInstr,
  teSC_MemInstr,
  teSC_BranchInstr,
  teSC_FloatInstr,
  teSC_ExecutedTranslatedBlocks,
  teSC_ExecutedTranslatedInstrs,
} temu_StatClass;

typedef struct {
  uint64_t InstrClassCounter[4];
  uint64_t ExecutedTranslatedBlocks;
  uint64_t ExecutedTranslatedInstructions;
  uint64_t GeneratedBlocks;
  uint64_t TranslatedInstructions;
  uint64_t ConnectedChains;
  uint64_t ExecutedTrampolines;

  uint64_t BlocksWithoutBranches;
  uint64_t BlocksWithoutAnnulledBranches;
  uint64_t BlocksWithAnnulledBranches;
  uint64_t BlocksIndirectBranches;
  uint64_t BlocksRelativeBranches;
  uint64_t BlocksWithUncondBranches;
  uint64_t BlocksWithoutUncondBranches;

  uint64_t AtcFetchHits;
  uint64_t AtcReadHits;
  uint64_t AtcWriteHits;
  uint64_t AtcFetchMisses;
  uint64_t AtcReadMisses;
  uint64_t AtcWriteMisses;
  uint64_t AtcFullFlushes;
  uint64_t AtcUserPageEvictions;
  uint64_t AtcSuperPageEvictions;
  uint64_t MMUFlushes;
  uint64_t SyncOps;

  uint64_t LastBlockPageVA;
  uint64_t LastBlockPA;

  uint64_t EarlyReturns;

  uint64_t AtcFetchIRMisses;
  uint64_t BypassReads;
  uint64_t BypassWrites;
  uint64_t ForceCacheMissReads;
  uint64_t ForceCacheMissWrites;
  uint64_t TrapsRaised;
  uint64_t InterruptsTaken;
  uint64_t EndOfPageFallthroughs;

  uint64_t AccumulatedTime;
  uint64_t AccumulatedSyncTime;

  uint64_t CasaCount;
} temu_ExecutionStatistics;

typedef struct RootPattern RootPattern;

#define TEMU_NUMBER_OF_TRACE_ENTRIES 4096

typedef enum {
  teTT_Instr,
  teTT_Enter,
  teTT_Exit,
  teTT_Trampoline,
  teTT_Attribute,
  teTT_Unknown,
  teTT_Profile,
  teTT_Idle,
  teTT_Illegal,
  teTT_RebindPC,
  teTT_RebindNPC,
  teTT_Interrupt
} temu_TraceType;

typedef struct {
  uint64_t StepCount; // Steps in processor
  uint64_t PC; // Program counter
  uint64_t TimeStampCounter; // Host timestamp counter
  temu_TraceType Type; // Kind of trace entry
  uint32_t Instruction; // Instruction if tracing entry is an instruction
} temu_CpuTraceEntry;

typedef struct {
  unsigned CurrentEntry;
  temu_CpuTraceEntry Entry[TEMU_NUMBER_OF_TRACE_ENTRIES];
} temu_CpuTrace;

typedef struct {
  void *Data;
  void (*Call)(void *, void *);
} temu_CallOp;

typedef struct {
  temu_TimeSource Super;

  int64_t IdleSteps;

  temu_CpuState State;
  temu_jmp_buf_t jmpbuf;
  uint32_t Flags;
  uint32_t CPUId; //!< Local CPU core ID
  int32_t CPUType;
  uint32_t CPUIndex; //!< Global CPU ID

  temu_PowerState PowerState;
  uint32_t StickyFlags;

  int64_t NullEventID;
  int64_t EnterHaltedEventID;

  uint8_t EnableExecutionTimeStat;
  temu_ExecutionStatistics Stats;
  temu_TargetExecutionIfaceRef Target;

  // Binary translation helper values
  uint64_t BlockPC;
  uint64_t BlockSteps;

  void *memSpace;      // Memory space, direct pointer
  temu_MemAccessIfaceRef memAccess;
  temu_ResetIfaceRef memReset;
  // temu_PDCIfaceRef PDCManager;

  intptr_t bac_i_diff; // Branch arc cache / intermediate difference
  void *branchArcCache;
  uint64_t bacValidity[32]; // Normally only uses 16 of these (4096 bytes /
                            // page, divided by 64)
  uint64_t HostFlags; // For speeding up ICC operations

  temu_CpuExitReason ForceReturnResult;

  RootPattern *CodePatterns;

  uint8_t ExitOnSync;

  temu_MemTransaction *fetchTransaction;
  temu_MemTransaction *readTransaction;
  temu_MemTransaction *writeTransaction;
  
  temu_CpuExitReason ExitReason;

  uint64_t SkipNextIdleTag;

#ifdef TEMU_DEBUG_CORE
  temu_CpuTrace Trace;
#endif

  size_t CallOpsSize;
  size_t CallOpsCapacity;
  temu_CallOp *CallOps;

  int64_t BreakpointTimeStamp;
  int64_t ReadWatchpointTimeStamp;
  int64_t WriteWatchpointTimeStamp;
} temu_Cpu;

#ifdef __cplusplus
}
#endif

#endif // !TEMU_CPU_UNSTABLE_H
