//===------------------------------------------------------------*- C++ -*-===//
//
// TEMU: The Terma Emulator
// (c) Terma 2015
// Authors: Mattias Holm <maho (at) terma.com>
//
//===----------------------------------------------------------------------===//

#ifndef TEMU_IRQ_CONTROLLER_H
#define TEMU_IRQ_CONTROLLER_H

#include "temu-c/Support/Objsys.h"

#ifdef __cplusplus
extern "C" {
#endif

//! Interrupt controller interface. An interrupt controller can raise
//! and lower IRQ signals. For systems which has interrupts which can
//! be configured to be active high, low or rising or falling, the
//! raise and lower irq have different semantics. For rising edge
//! triggering, the raiseInterrupt function should trigger the IRQ.

typedef struct temu_IrqControllerIface {
  //! Raise interrupt
  void (*raiseInterrupt)(void *Obj, uint8_t Irq);
  //! Lower interrupt
  void (*lowerInterrupt)(void *Obj, uint8_t Irq);
} temu_IrqCtrlIface;
#define TEMU_IRQ_CTRL_IFACE_TYPE "IrqCtrlIface"
TEMU_IFACE_REFERENCE_TYPE(temu_IrqCtrl);

//! Interface to be defined for classes that uses an IRQ controller
//! object. An IRQ controller may acknowledge to the IRQ controller
//! client, or it may notify the IrqClient that the underlying IRQ
//! controller has been modified and any pending IRQs should be
//! re-issued.  Typically, an updateInterrupt call happens if the IRQ
//! registers change in such a way that the IRQ controller does not
//! know the next interrupt to be issued. E.g. On the sparc, the
//! updateInterrupts function will be called on its IRQ controller
//! whenever the ET or PIL field has changed.
//! That is updateInterrupts are called by lazy IRQ controllers.

typedef struct temu_IrqClientIface {
  //! Acknowledge interrupt (IRQ controller should clear interrupt)
  void (*ackInterrupt)(void *Obj, uint8_t Irq);
  //! Called in case IRQ re-issuing is needed eg. when the IRQ
  //! controlling registers have been modified.
  //! IRQ controller should reevaluate interrupts and reraise them upstream.
  void (*updateInterrupts)(void *Obj);
} temu_IrqClientIface;
#define TEMU_IRQ_CLIENT_IFACE_TYPE "IrqClientIface"
TEMU_IFACE_REFERENCE_TYPE(temu_IrqClient);

#ifdef __cplusplus
}
#endif

#endif /* ! TEMU_IRQ_CONTROLLER_H */
