//===-- temu-c/Assembler.h - Assembler and Dissassebler----------*- C++ -*-===//
//
// TEMU: The Terma Emulator
// (c) Terma 2015
// Authors: Mattias Holm <maho (at) terma.com>
//
//===----------------------------------------------------------------------===//
/*!
  \file

  This file contains the declarations of the assembler and disassembler.
*/

#ifndef TEMU_SUPPORT_ASSEMBLER
#define TEMU_SUPPORT_ASSEMBLER
#ifdef __cplusplus
extern "C" {
#endif

#include "temu-c/Support/Attributes.h"
#include "temu-c/Support/Objsys.h"
#include <stdint.h>

/*! Assemble an instruction
 * \param Cpu processor for which to assemble.
 * \param AsmStr String with instruction in the CPUs assembler.
 * \result Instruction bitpattern.
 */
TEMU_API uint32_t temu_assemble(void *Cpu, const char *AsmStr);

/*! Disassemble an instruction.
 * \warning The function returns a malloced string. You have ownership
 *           and is responsible for calling free() on it.
 * \result A pointer to a malloced instruction string
 *         or null incase of failure.
 */
TEMU_API char *temu_disassemble(void *Cpu, uint32_t Instr);

/*! Disassemble an instruction.
 * \warning The function returns a thread local string. It is borrowed
 * and not allowed to escape the current thread without a copy.
 * \result A pointer to a a thread local instruction string
 *         or null incase of failure.
 */
TEMU_API const char *temu_disassembleAuto(void *Cpu, uint32_t Instr);

/*! Assemble an instruction to memory
 * \param Cpu processor for which to assemble.
 * \param AsmStr String with instruction in the CPUs assembler.
 * \param Addr Physical address of where to put the instruction.
 */
TEMU_API void temu_assembleToMemory(void *Cpu, const char *AsmStr,
                                    uint64_t Addr);

/*! Disassemble an instruction in memory
 *
 * The function dissasmbles an instruction and retuns a string allocated
 * on the heap. You are responsible for its release using free().
 *
 * \warning The function returns a malloced string. You have ownership
 *           and are responsible for calling free() on it.
 *
 * \param Cpu CPU whose memory space will be used for disasembling.
 * \param Addr Physical address of instruction to dissassemble.
 * \result A pointer to a malloced instruction string
 *         or null incase of failure.
 */
TEMU_API char *temu_disassembleMemory(void *Cpu, uint64_t Addr);

// Returns a thread local string instead
TEMU_API const char *temu_disassembleMemoryAuto(void *Cpu, uint64_t Addr);

// The assembly interface is currently experimental,
// It is intended to support generic assembly / disassembly queries,
// and advanced disassembly like split into different fields.
typedef enum temu_InstructionFieldType {
  teIFT_String,
  teIFT_Register,
  teIFT_Address,
  teIFT_Displacement,
  teIFT_Immediate,
} temu_InstructionFieldType;

typedef struct temu_InstructionField {
  temu_InstructionFieldType Type;
  const char *Text; // Used by string and registers
  int64_t Value;    // Used by address, displacement and immediate values
} temu_InstructionField;

typedef struct temu_AssemblerIface {
  // Assemble instruction to data buffer,
  // instruction bytes will be in target endianess.
  // buffer must be of length of the instruction,
  // safe values can be retrieved by maxInstructionLength.
  //   isa: instruction set id
  //   instr: instruction to assemble
  //   length: bytes of data block
  //   data: where to write the instruction
  // result: Length of instruction on success negative values indicate errors.
  int (*assembleInstruction)(void *obj, int isa, const char *instr,
                             size_t length, uint8_t *data);

  // Disassemble data buffer as one instruction
  // isa: Instruction set id
  // length: length of data buffer
  // data: data buffer
  // Result: pointer to thread local instruction string,
  // it is valid until the function is called again in the calling thread.
  // Null is returned on failures.
  const char *(*disassembleInstruction)(void *obj, int isa, size_t length,
                                        uint8_t *data);

  // Disassemble an instruction to individual fields
  // The function is optional at the moment and must be checked for null before
  // being called.
  // Result: 0 on success, numFields and fields are used to return fields.
  int (*disassembleInstructionToFields)(void *obj, int isa, size_t length,
                                        uint8_t *data, size_t *numFields,
                                        temu_InstructionField **fields);

  // Read an instruction in target byte order from memory
  //   obj: Processor pointer
  //   isa: Instruction set index (see getCPUInfo in cpu interface)
  //   flags:
  //      bit 0: ignore target endianess and use big endian
  //      bit 1: ignore target endianess and use little endian
  //   length: length of data for instruction
  //   data: Buffer to place instruction in, must be of size returned by
  //   maxInstructionLength Result: 0 on success, other values indicate errors.
  int (*readInstruction)(void *obj, int isa, unsigned flags, uint64_t address,
                         uint64_t nextPageAddress, size_t *length,
                         uint8_t *data);

  // Returns number of instruction sets
  int (*getNumberofInstructionSets)(void *obj);

  // Returns <= 0 for failures
  int (*getMaxInstructionSize)(void *obj, int isa);

  // Returns NULL on failure
  const char *(*getInstructionSetName)(void *obj, int idx);

  // Returns negative on failure
  int (*getInstructionSetIndex)(void *obj, const char *name);

} temu_AssemblerIface;
#define TEMU_ASSEMBLER_IFACE_TYPE "temu::AssemblerIface"
TEMU_IFACE_REFERENCE_TYPE(temu_Assembler);

#ifdef __cplusplus
}
#endif

#endif /* ! TEMU_SUPPORT_ASSEMBLER */
