//===-- temu-c/CommandLine.h - TEMU Command Line ---------------*- C++ -*-===//
//
// TEMU: The Terma Emulator
// (c) Terma 2015, 2019
// Authors: Mattias Holm <maho (at) terma.com>
//
//===----------------------------------------------------------------------===//

#ifndef TEMU_COMMAND_LINE
#define TEMU_COMMAND_LINE

#include "temu-c/Support/Attributes.h"
#include "temu-c/Support/Objsys.h"
#include <stddef.h>
#include <stdint.h>
#ifdef __cplusplus
extern "C" {
#endif

/*!
 * Parses command line options
 * The TEMU command supports a number of built in command line options. A
 * simulator embedding TEMU may want initialise these options in the same way
 * that the TEMU CLI does. Note that not all options are supported this way as
 * some options will only be registered by the temu CLI application itself.
 *
 * In general interactive options will not work (options that can be interpreted
 * as a command). It is possible to use temu_printCommandLineHelp() to list
 * currently registered options from an application that embedds temu.
 *
 * \param argc Number of arguments
 * \param argv Command line arguments
 * \result 0 on success, otherwise non-zero value.
 */
TEMU_API int temu_parseCommandLineOptions(int argc, const char *argv[]);

/*!
 * Print command line help to temu stderr stream.
 */
TEMU_API void temu_printCommandLineHelp(void);

/*!
 * Executes the commands in the file "File"
 * \param File Path to the file with the commands to be executed
 * \result 0 on success, otherwise a non-zero value
 */
TEMU_API int temu_execCommandFile(const char *File);

/*!
 * Executes a single command
 * \param Cmd The command to be executed
 * \result 0 on success, otherwise a non-zero value
 */
TEMU_API int temu_execCommand(const char *Cmd);

//! User command handlers implement this interface. The pointer
//! argument is a pointer to an opaque context. Use the functions below
//! to access properties from the context.
typedef int (*temu_CommandFunc)(void *);

typedef enum temu_CmdOptionKind {
  teCOK_String = 0,
  teCOK_Path,   //!< Path is a string, but with auto completion of file names
  teCOK_Object, //!< Object is a named object
  teCOK_Int,    //!< Any integer number
  teCOK_Double, //!< Any floating point number
  teCOK_Prop,   //!< Property reference
  teCOK_Iface,  //!< Interface reference
  teCOK_Reg,    //!< Register reference
  teCOK_Field,  //!< Register field reference
  teCOK_Class,  //!< Class option
} temu_CmdOptionKind;

typedef struct {
  const char *Name;
  temu_CmdOptionKind Type;
  union {
    const char *String;
    const char *Path;
    temu_Object *Obj;
    int64_t Integer;
    double Real;
    temu_PropName PRef;
    temu_IfaceRef IRef;
    temu_PropName RegRef;
    temu_PropName FieldRef;
    temu_Class *Class;
  };
} temu_CmdArg;

typedef int (*temu_ObjectCommandFunc)(temu_Object *Obj, void *I, int argc,
                                      const temu_CmdArg args[]);

/*!
 * Register a class command
 * 
 * \param Cls Class pointer
 * \param Name Name of command
 * \param Doc Documentation string for command
 * \param F Command handler function
 * \result Opaque handle to command. NULL in case of failure.
 */
TEMU_API void *temu_createClassCmd(temu_Class *Cls, const char *Name,
                                   const char *Doc, temu_ObjectCommandFunc F);
/*!
 * Get an already registered command handle from class
 * 
 * This function can be used to for example get commands
 * registered in the metaclass, such as the new command.
 * The command can then have arguments added.
 * 
 * \param Cls Class pointer
 * \param Name Name of command to get
 * \result Opaque handle to command. NULL in case of failure.
 */
TEMU_API void *temu_classGetCmd(temu_Class *Cls, const char *Name);

/*!
 * Add parameter to command
 *
 * \param Cmd Command handle
 * \param Name Name of parameter to add
 * \param Type Type of the parameter
 * \param Required Set to 1 if the parameter must be set for the command
 * \param Doc Documentation string for the parameter
 * \result 0 on success, other values indicates errors
 */
TEMU_API int temu_classCmdAddParam(void *Cmd, const char *Name,
                                   temu_CmdOptionKind Type, int Required,
                                   const char *Doc);

/*!
 * Get named command option
 * 
 * \param argc Number of arguments
 * \param args Argument vector
 * \param OptName Name of parameter / option to get
 * \result Pointer to command argument or NULL in case it is not found.
 */
TEMU_API const temu_CmdArg* temu_classCmdGetOption(int argc, const temu_CmdArg args[], const char *OptName);
/*!
 * Get named command option as signed integer
 *
 * Can only get valid numeric parameters.
 * The function will never fail if the type is numeric and the option is required.
 * For optional arguments, use the generic access function `temu_classCmdGetOption()`
 * 
 * \param argc Number of arguments
 * \param args Argument vector
 * \param OptName Name of parameter / option to get
 * \result The value of the parameter converted to an integer.
 */
TEMU_API int64_t temu_classCmdGetOptionAsInteger(int argc, const temu_CmdArg args[], const char *OptName);

/*!
 * Get named command option as unsigned integer
 *
 * Can only get valid numeric parameters.
 * The function will never fail if the type is numeric and the option is required.
 * For optional arguments, use the generic access function `temu_classCmdGetOption()`
 * 
 * \param argc Number of arguments
 * \param args Argument vector
 * \param OptName Name of parameter / option to get
 * \result The value of the parameter converted to an unsigned integer.
 */
TEMU_API uint64_t temu_classCmdGetOptionAsUnsigned(int argc, const temu_CmdArg args[], const char *OptName);

/*!
 * Invoke command on object
 * 
 * \param Obj Object
 * \param I Interpreter context
 * \param Name Name of command
 * \param Argc Number of arguments
 * \param Argv Argument vector
 * \result Command result. 0 means success.
 */
TEMU_API int temu_objectInvokeCmd(temu_Object *Obj, void *I, const char *Name,
                                  int Argc, temu_CmdArg Argv[]);

/*
 * Raise an error in the interpreter for the current command
 *
 * \param I Interpreter (I-argument to command func)
 * \param S Error string
 * \result Returns -1 so it can be used as "return temu_raiseCmdError(I, "foo")"
 * in the command handler.
 */
TEMU_API int temu_raiseCmdError(void *I, const char *S, ...);

//! Create and register global command
//! \param Name Name of command
//! \param F Command function to invoke
//! \param Doc Documentation string
//! \param Data Data pointer. Some commands e.g. disassemble will
//!             increase the address between invocations. This must be
//!             saved in some data object which can be provided when
//!             the command is created.

TEMU_API void *temu_createCmd(const char *Name, temu_CommandFunc F,
                              const char *Doc, void *Data);

//! Add named argument to command
//! \param Cmd Pointer to the command object
//! \param OptName Option name
//! \param Type Option type
//! \param Required Pass 0 if the option is not required, otherwise required
//! \param Doc Option documentation
//! \param Default Default value of the option
TEMU_API void temu_cmdAddOption(void *Cmd, const char *OptName,
                                temu_CmdOptionKind Type, int Required,
                                const char *Doc, const char *Default);

//! Get data pointer from command context
//! This function shall be called in a command handler on the passed
//! context.
//! \param Ctxt Pointer to the context of the command
//! \result data pointer
TEMU_API void *temu_cmdGetData(void *Ctxt);

//! Get pointer to the command interpreter
//! This function shall be called in a command handler on the passed
//! context.
//! \param Ctxt Pointer to the context of the command
//! \result Pointer to interpreter
TEMU_API void *temu_cmdGetInterpreter(void *Ctxt);

//! Get named option as integer from command context
//! This function shall be called in a command handler on the passed
//! context.
//! \param Ctxt Command context
//! \param OptName Option name
//! \result The integer bound to the named argument
TEMU_API int64_t temu_cmdGetOptionAsInteger(void *Ctxt, const char *OptName);

//! Get named option as object pointer from command context
//! This function shall be called in a command handler on the passed
//! context.
//! \param Ctxt Command context
//! \param OptName Option name
//! \result Pointer to the option object
TEMU_API void *temu_cmdGetOptionAsObject(void *Ctxt, const char *OptName);

//! Get named option as string from command context
//! This function shall be called in a command handler on the passed
//! context.
//! \param Ctxt Command context
//! \param OptName Option name
//! \result C-string of the name of the option
TEMU_API const char *temu_cmdGetOptionAsString(void *Ctxt, const char *OptName);

//! Get named option as double from command context
//! This function shall be called in a command handler on the passed
//! context.
//! \param Ctxt Command context
//! \param OptName Option name
//! \result The real value of the option as double
TEMU_API double temu_cmdGetOptionAsReal(void *Ctxt, const char *OptName);

//! Get number of positional options given
//! This function shall be called in a command handler on the passed
//! context.
//! \param Ctxt Command context
//! \result The number of position optionals in Ctxt
TEMU_API size_t temu_cmdGetPosOptSize(void *Ctxt);

//! Get positional option at index
//! This function shall be called in a command handler on the passed
//! context.
//! \param Ctxt Command context
//! \param Idx Index of the option
//! \result The option at position Idx as a C-string
TEMU_API const char *temu_cmdGetPosOpt(void *Ctxt, size_t Idx);

//! Return 1 if the option is valid, 0 if invalid / not set
//! This function shall be called in a command handler on the passed
//! context.
//! \param Ctxt Command context
//! \param OptName Option name
//! \result 1 if the option is valid, 0 if invalid / not set
TEMU_API int temu_cmdOptionIsValid(void *Ctxt, const char *OptName);

//! Set a variable in the command line
//! \param Key Variable name (must match [A-Za-z_][A-Za-z0-9_]*)
//! \param Value Value to assign to variable
//! \result Non-zero on errors
TEMU_API int temu_cmdSetVariable(const char *Key, const char *Value);

//! Get a variable in the command line
//! \param Key Variable name
//! \result In case the variable is not found NULL, otherwise
//!         a borrowed string.
TEMU_API const char *temu_cmdGetVariable(const char *Key);

#ifdef __cplusplus
}
#endif

#endif /* !TEMU_COMMAND_LINE */
