//===-- temu-c/Memory.h - Memory Helpers ------------------------*- C++ -*-===//
//
// TEMU: The Terma Emulator
// (c) Terma 2015-2016
// Authors: Mattias Holm <maho (at) terma.com>
//
//===----------------------------------------------------------------------===//

#ifndef TEMU_SUPPORT_MEMORY
#define TEMU_SUPPORT_MEMORY

#include "temu-c/Support/Attributes.h"
#include "temu-c/Support/Objsys.h"
#include <stdint.h>

#ifdef __cplusplus
extern "C" {
#endif

typedef enum temu_MemoryKind {
  teMK_RAM = 0,      //!< Model is a RAM
  teMK_ROM = 1,      //!< Model is a ROM
  teMK_MMIO = 2,     //!< Alias for `teMK_IO`
  teMK_IO = 2,       //!< Model is a normal IO device
  teMK_MemSpace = 4, //!< Model is a memory space
  teMK_Last = 4,     //!< DO NOT USE
} temu_MemoryKind;

//! Memory attribute enumeration

//! The emulator provides 5 standard attributes, and 3 user defined
//! ones. The attributes are set in the memory space (not the memory
//! models), so it is possible to set a watch point on memory mapped
//! devices. When an attribute is set on a page, that page will get a
//! shadow attribute page (same size as the page), enabling attributes
//! to be set on a per byte level.
//!
//! Attributes are only checked on the address being accessed, the
//! transaction size is not taken into account.
typedef enum temu_MemoryAttr {
  teMA_Break = 1,           //!< Breakpoint set
  teMA_WatchRead = 1 << 1,  //!< Read watchpoint set
  teMA_WatchWrite = 1 << 2, //!< Write watchpoint set
  teMA_Upset = 1 << 3,      //!< Single event upset
  teMA_Faulty = 1 << 4,     //!< Multiple event upset / uncorrectable
  teMA_User1 = 1 << 5,      //!< User definable
  teMA_User2 = 1 << 6,      //!< User definable
  teMA_User3 = 1 << 7,      //!< User definable
} temu_MemoryAttr;
typedef uint8_t temu_MemoryAttrs;
typedef struct temu_MemAccessIface temu_MemAccessIface;

typedef enum temu_MemoryStat {
  teMS_IOReads,
  teMS_IOWrites,
  teMS_SelfModifyingCodeWrites,
  teMS_CodeWrites,
  teMS_LocksTaken,
  teMS_IOReadTime,
  teMS_IOWriteTime,
  teMS_LockTime,
} temu_MemoryStat;

// Private interface, this is not stable yet
typedef struct temu_MemorySpaceIface {
  //! Map device (using the default MemAccessIface interface)
  int (*mapDevice)(void *Obj, uint64_t Addr, uint64_t Len, temu_Object_ *Device,
                   uint32_t Flags);
  //! Set attribute bit
  void (*setAttr)(void *Obj, uint64_t Addr, uint64_t Len, temu_MemoryAttr Attr);
  //! Clear attribute bit
  void (*clearAttr)(void *Obj, uint64_t Addr, uint64_t Len,
                    temu_MemoryAttr Attr);
  //! Get attributes for address
  temu_MemoryAttrs (*getAttrs)(void *Obj, uint64_t Addr);

  //! Map device with named interface
  int (*mapDeviceWithNamedIface)(void *Obj, uint64_t Addr, uint64_t Len,
                                 temu_Object_ *Device, const char *IfaceName,
                                 uint32_t Flags);

  //! Unmap devices in range
  int (*unmapRange)(void *Obj, uint64_t Addr, uint64_t Len);

  //! Map device with interface pointer
  int (*mapDeviceWithIface)(void *Obj, uint64_t Addr, uint64_t Len,
                            temu_Object_ *Device, temu_MemAccessIface *Iface,
                            uint32_t Flags);

  uint64_t (*getStat)(void *Obj, temu_MemoryStat Stat);
} temu_MemorySpaceIface;

TEMU_IFACE_REFERENCE_TYPE(temu_MemorySpace);
#define TEMU_MEMORY_SPACE_IFACE_TYPE "temu::MemorySpaceIface"

// Private interface, do not use directly
#define TEMU_PDC_PSEUDO_UNKNOWN_INSTR 0
#define TEMU_PDC_PSEUDO_EOP 1
#define TEMU_PDC_PSEUDO_TRAMPOLINE 2
#define TEMU_PDC_PSEUDO_POST_DS_0 3
#define TEMU_PDC_PSEUDO_POST_DS_1 4
#define TEMU_PDC_PSEUDO_ATTRIB_CHECK 5
#define TEMU_PDC_PSEUDO_PROFILE 6
#define TEMU_PDC_PSEUDO_PRE_TRAMPOLINE 7
#define TEMU_PDC_PSEUDO_IDLE 8
#define TEMU_PDC_PSEUDO_SKIP 9
#define TEMU_PDC_PSEUDO_CALL 10
#define TEMU_PDC_PSEUDO_CALL_AND_SKIP 11
#define TEMU_PDC_PSEUDO_MULTIPAGE 12
#define TEMU_PDC_PSEUDO_OP_COUNT 13

typedef struct {
  uint32_t Instruction;
  uint32_t Operands;
} temu_IRInstruction;

typedef struct {
  temu_IRInstruction Inst;
  uint64_t ExtParam;
} temu_ExtIRInstruction;

// temu_PDCIface is internal and unstable.
typedef struct {
  void *(*getPDCForCpu)(void *Obj, int Cpu, uint64_t Addr);
  void (*allocPDCForCpu)(void *Obj, int Cpu, uint64_t Addr);
  void (*disposePDCForCpu)(void *Obj, int Cpu, uint64_t Addr);
  void (*writePDCForCpu)(void *Obj, int Cpu, uint64_t Addr, void *Inst);
  void (*clearPDCForCpu)(void *Obj, int Cpu, uint64_t Addr);
  void (*setPseudoOpHandlerForCpu)(void *Obj, int Cpu, unsigned Opcode,
                                   uintptr_t Handler);
  void (*installTrampolineForCpu)(void *Obj, int Cpu, uint64_t Addr,
                                  void (*TrampHandler)(void *));
  void (*installProfileForCpu)(void *Obj, int Cpu, uint64_t Addr);
  void (*removeProfileForCpu)(void *Obj, int Cpu, uint64_t Addr);

  void (*installIdleForCpu)(void *Obj, int Cpu, uint64_t Addr, uint32_t Param);
  void (*removeIdleForCpu)(void *Obj, int Cpu, uint64_t Addr);

  void (*setBTForCpu)(void *Obj, int Cpu, uint64_t Addr, void *BTP);

  temu_ExtIRInstruction *(*allocPDCChainForCpu)(void *Obj, int Cpu,
                                                uint64_t Addr);
  void (*releasePDCChainForCpu)(void *Obj, int Cpu, uint64_t Addr);

  uint64_t (*irToPAForCpu)(void *obj, int Cpu, uint64_t addr);

  void (*installSkipForCpu)(void *Obj, int Cpu, uint64_t Addr, unsigned Steps);
  void (*removeSkipForCpu)(void *Obj, int Cpu, uint64_t Addr);

  void (*installCallForCpu)(void *Obj, int Cpu, uint64_t Addr, unsigned FuncID);
  void (*removeCallForCpu)(void *Obj, int Cpu, uint64_t Addr);
} temu_PDCIface;
TEMU_IFACE_REFERENCE_TYPE(temu_PDC);
#define TEMU_PDC_IFACE_TYPE "PDCIface"

/*!
 * Read block of data via memory block transfer interfaces
 * \param mem Pointer to memory space object
 * \param buff The buffer to which the memory should be stored
 * \param addr The address of the memory block to be read
 * \param size The size to be read
 * \param swap setting this to 0 indicates reading a byte array, 1 a
 *   uint16 array, 2 a uint32 array and 3 a uint64 array
 *
 * \result Negative on failures. Other values indicate success. The
 *   convention is to return the number of bytes read which should be
 *   == size.
 */
TEMU_API int temu_memoryRead(void *mem, uint8_t *buff, uint64_t addr,
                             uint32_t size, int swap);

/*!
 * Write block of data via memory block transfer interfaces
 * \param mem Pointer to memory space object
 * \param buff The buffer to which the memory should be stored
 * \param addr The address, at which the write should be done
 * \param size The size to be read
 * \param swap Negative on failures. Other values indicate success.
 *
 * \result Negative on failure. Other values indicate success. The
 *   convention is to return the number of bytes written, which should
 *   be == size.
 */
TEMU_API int temu_memoryWrite(void *mem, uint64_t addr, uint8_t *buff,
                              uint32_t size, int swap);

/*!
 * Read block of data via large memory transactions.
 * \param obj Memory space object
 * \param addr Physical address inside memory space
 * \param buff Data buffer
 * \param unitSize Log size of transaction unit size (0 = u8, 1 = u16, 2 = u32,
 * 3 = u64)
 * \param size Number of units to transfer
 * \param flags Memory transaction flags (e.g. TEMU_MT_LITTLE_ENDIAN)
 * \result Negative on failures. Other values indicate success.
 */

TEMU_API int temu_memoryReadData(void *obj, uint64_t addr, uint8_t *buff,
                                 unsigned unitSize, uint32_t size,
                                 unsigned flags);

/*!
 * Write block of data via large memory transactions.
 * \param obj Memory space object
 * \param addr Physical address inside memory space
 * \param buff Data buffer
 * \param unitSize Log size of transaction unit size (0 = u8, 1 = u16, 2 = u32,
 * 3 = u64)
 * \param size Number of units to transfer
 * \param flags Memory transaction flags (e.g. TEMU_MT_LITTLE_ENDIAN)
 * \result Negative on failures. Other values indicate success.
 */
TEMU_API int temu_memoryWriteData(void *obj, uint64_t addr, const uint8_t *buff,
                                  unsigned unitSize, uint32_t size,
                                  unsigned flags);

/*!
 * temu_memoryMap Maps an object into a memory space. The object must have an
 * interface named MemAccessIface, of the type defined as
 * TEMU_MEM_ACCESS_IFACE_TYPE
 * \param Obj is the memory space object
 * \param Addr the physical address
 * \param Len the length in bytes
 * \param MemObj is the object that is mapped into the memory space
 *               it needs to be of a class that follows the rules above
 * \param Flags are flags that are copied into the
 *              memory transaction object's flag field when a transaction
 *              reaches an object
 * \result Zero on success
 */
TEMU_API int temu_memoryMap(void *Obj, uint64_t Addr, uint64_t Len,
                            void *MemObj, uint32_t Flags);

/*!
 * temu_memoryMapNamedIface Maps an object into a memory space using the named
 * memory access interface. The interface named by IfaceName must be of the type
 * TEMU_MEM_ACCESS_IFACE_TYPE
 * \param Obj is the memory space object
 * \param Addr the physical address \param Len the length in bytes
 * \param MemObj is the object that is mapped into the memory space it needs to
 *               be of a class that follows the rules above
 * \param IfaceName Name of the memory access interface
 * \param Flags are flags that are copied into the memory transaction object's
 *              flag field when a transaction reaches an object
 * \result Zero on success
 */
TEMU_API int temu_memoryMapNamedIface(void *Obj, uint64_t Addr, uint64_t Len,
                                      void *MemObj, const char *IfaceName,
                                      uint32_t Flags);

TEMU_API int temu_memoryMapNamedIface2(void *Obj, uint64_t Addr, uint64_t Len,
                                       void *MemObj, const char *IfaceName,
                                       unsigned Idx, uint32_t Flags);

/*!
 * Sets an attribute on the given memory range
 * \param Obj  Memory space object
 * \param Addr Physical address of start
 * \param Len Length of memory range
 * \param Attr Attribute to set
 */
TEMU_API void temu_memorySetAttr(void *Obj, uint64_t Addr, uint64_t Len,
                                 temu_MemoryAttr Attr);

/*!
 * Clears an attribute on the given memory range
 * \param Obj  Memory space object
 * \param Addr Physical address of start
 * \param Len Length of memory range
 * \param Attr Attribute to clear
 */
TEMU_API void temu_memoryClearAttr(void *Obj, uint64_t Addr, uint64_t Len,
                                   temu_MemoryAttr Attr);

/*!
 * Get memory attributes for address
 * \param Obj Memory space
 * \param Addr Physical addres
 * \result Memory attributes for the address
 */
TEMU_API temu_MemoryAttrs temu_memoryGetAttrs(void *Obj, uint64_t Addr);

/*!
 * Issue a big endian memory read transaction (without initiator)
 * \param Obj Memory space
 * \param Addr Physical address
 * \param Word 16 bit word that is read
 * \result 0 on success, other values imply failure and will not
 *         update the word.
 */
TEMU_API int temu_memoryReadPhys16(void *Obj, uint64_t Addr, uint16_t *Word);
/*!
 * Issue a little endian memory read transaction (without initiator)
 * \param Obj Memory space
 * \param Addr Physical address
 * \param Word 16 bit word that is read
 * \result 0 on success, other values imply failure and will not
 *         update the word.
 */
TEMU_API int temu_memoryReadPhys16Little(void *Obj, uint64_t Addr,
                                         uint16_t *Word);

/*!
 * Issue a big endian memory read transaction (without initiator)
 * \param Obj Memory space
 * \param Addr Physical address
 * \param Word 32 bit word that is read
 * \result 0 on success, other values imply failure and will not
 *         update the word.
 */
TEMU_API int temu_memoryReadPhys32(void *Obj, uint64_t Addr, uint32_t *Word);
/*!
 * Issue a little endian memory read transaction (without initiator)
 * \param Obj Memory space
 * \param Addr Physical address
 * \param Word 32 bit word that is read
 * \result 0 on success, other values imply failure and will not
 *         update the word.
 */
TEMU_API int temu_memoryReadPhys32Little(void *Obj, uint64_t Addr,
                                         uint32_t *Word);
/*!
 * Issue a big endian memory write transaction (without initiator)
 * \param Obj Memory space
 * \param Addr Physical address
 * \param Word 16 bit word to write
 * \result 0 on success
 */
TEMU_API int temu_memoryWritePhys16(void *Obj, uint64_t Addr, uint16_t Word);

/*!
 * Issue a big endian memory write transaction (without initiator)
 * \param Obj Memory space
 * \param Addr Physical address
 * \param Word 32 bit word to write
 * \result 0 on success
 */
TEMU_API int temu_memoryWritePhys32(void *Obj, uint64_t Addr, uint32_t Word);

/*!
 * Issue a little endian memory write transaction (without initiator)
 * \param Obj Memory space
 * \param Addr Physical address
 * \param Word 32 bit word to write
 * \result 0 on success
 */
TEMU_API int temu_memoryWritePhys32Little(void *Obj, uint64_t Addr,
                                          uint32_t Word);

/*!
 * Normalise a value for writes where only 32 bit transactions are
 * supported by the device model, but the target is allowed to write
 * non-32 bit quantity. The function mixes the old register value with
 * the new one based on the size and offset parameter. The problem
 * exists because the memory access interface has a value entry, which
 * always ends up in the lower bits, so if we write to the higher bits
 * in in a 32 bit register, then by just forwarding the the value as
 * is to the write handler, will result in a write of the lower bits and
 * a clear of the upper bits (for a store unsigned).
 *
 * Thus a normalisation is needed where we mix the written word with the old
 * bits. So for transaction size of 16, and an offset of 16, the resulting
 * word is (old & 0x0000ffff) | (new << 16)
 *
 * \param OldVal Old regiseter value.
 * \param NewVal Content in memory transaction (new value).
 * \param Sz Size in log number of bytes
 * \param Off Offset in bytes within the 32 bit word of the transaction.
 */
TEMU_API uint32_t temu_normaliseWrite32(uint32_t OldVal, uint32_t NewVal,
                                        int Sz, int Off);

/*!
 * Normalise a value for reads where only 32 bit transactions are
 * supported by the device model, but the target is allowed to read
 * non-32 bit quantities.
 *
 * Given a 32 bit register value, the function extracts the bits from it
 * that was actually requested.
 *
 * \param Value Register value
 * \param Sz Size in log number of bytes (0, 1, or 2)
 * \param Off Offset in bytes within the 32 bit word of the transaction (0-3)
 */
TEMU_API uint32_t temu_normaliseRead32(uint32_t Value, int Sz, int Off);

/*!
 * Normalise a value for reads where only 16 bit transactions are
 * supported by the device model, but the target is allowed to read
 * non-16 bit quantities.
 *
 * Given a 32 bit register value, the function extracts the bits from it
 * that was actually requested.
 *
 * \param Value Register value
 * \param Sz Size in log number of bytes (0 or 1)
 * \param Off Offset in bytes within the 32 bit word of the transaction (0 or 2)
 */
TEMU_API uint16_t temu_normaliseRead16(uint16_t Value, int Sz, int Off);

/*!
 * Normalise a value for writes where only 16 bit transactions are
 * supported by the device model, but the target is allowed to write
 * non-16 bit quantity. The function mixes the old register value with
 * the new one based on the size and offset parameter. The problem
 * exists because the memory access interface has a value entry, which
 * always ends up in the lower bits, so if we write to the higher bits
 * in in a 16 bit register, then by just forwarding the the value as
 * is to the write handler, will result in a write of the lower bits and
 * a clear of the upper bits (for a store unsigned).
 *
 * Thus a normalisation is neded where we mix the written word with the old
 * bits. So for transaction size of 8, and an offset of8, the resulting
 * word is (old & 0x00ff) | (new << 8)
 *
 * \param OldVal Old regiseter value.
 * \param NewVal Content in memory transaction (new value).
 * \param Sz Size in log number of bytes
 * \param Off Offset in bytes within the 16 bit word of the transaction.
 */

TEMU_API uint16_t temu_normaliseWrite16(uint16_t OldVal, uint16_t NewVal,
                                        int Sz, int Off);

/*!
 * Install trampoline function in PDC cache
 * \param Obj Memory space
 * \param Addr Physical address
 * \param Tramp Trampoline function, takes CPU pointer as first argument.
 * \result 0 on success
 */
TEMU_API int temu_memoryInstallTrampoline(void *Obj, uint64_t Addr,
                                          void (*Tramp)(void *));
TEMU_API uint64_t temu_irToPhys(void *obj, uintptr_t ir);
TEMU_API uint64_t temu_irToPhysForCpu(void *obj, int cpu, uintptr_t ir);

#ifdef __cplusplus
}
#endif

#endif /* ! TEMU_SUPPORT_MEMORY */
