//===------------------------------------------------------------*- C++ -*-===//
//
// TEMU: The Terma Emulator
// (c) Terma 2016
// Authors: Mattias Holm <maho (at) terma.com>
//
//===----------------------------------------------------------------------===//

#ifndef TEMU_CAN_H
#define TEMU_CAN_H

/*
  The CAN interface is EXPERIMENTAL AND UNSTABLE at present.

  The CAN bus model consist of two interfaces, firstly a CAN device
  interface and secondly a CAN bus interface. In addition to these
  interfaces a bus class is provided that does message routing and
  some validation of device traffic. This bus class is needed as the
  CAN (like the 1553) bus is a multi-node bus.

  The bus interface allows the user to implement their own bus models
  if needed (e.g. to support connections to legacy models using
  another device interface).

  The bundled bus models, allows for the enabling of send events,
  i.e. an event being issued with whenever a transaction is on the
  bus. This can be used to inject errors in messages if needed.
 */

#include "temu-c/Support/Objsys.h"

#include <assert.h>
#include <stdbool.h>
#include <string.h>
#include <stdint.h>

#ifdef __cplusplus
extern "C" {
#endif


// The flags field contain the frame header,
// it is constructed as follows:
//   ACK : 1 / [31]
//   RTR : 1 / [30]
//   IDE : 1 / [29]
//   Ident A: 11 / [28..18]
//   Ident B: 18: [17..0]
// As can be seen, this does not fully reflect a CAN message, but
// the critical bits are there. Bits implied by the format are
// omitted.
// Also, the frame object does not contain CRCs, we do not need
// these in the bus model and CRC errors are instead modelled
// using the error field.
typedef struct {
  uint8_t Data[8];
  uint32_t Flags;
  uint8_t Length;
  uint8_t Error;
} temu_CanFrame;

static inline int
temu_canIsExtendedFrame(temu_CanFrame *Frame)
{
  return (Frame->Flags >> 29) & 1;
}
static inline int
temu_canIsRemoteTransmissionRequest(temu_CanFrame *Frame)
{
  return (Frame->Flags >> 30) & 1;
}

// User model can call this to set the ack bit
static inline void
temu_canSetAck(temu_CanFrame *Frame)
{
  Frame->Flags |= (1 << 31);
}

static inline uint32_t
temu_canGetAck(temu_CanFrame *Frame)
{
  return (Frame->Flags >> 31) & 1;
}

static inline uint32_t
temu_canGetExtendedIdent(const temu_CanFrame *Frame)
{
  return Frame->Flags & 0x1fffffff;
}

static inline uint32_t
temu_canGetIdent(temu_CanFrame *Frame)
{
  if (temu_canIsExtendedFrame(Frame)) {
    return Frame->Flags & 0x1fffffff;
  } else {
    return (Frame->Flags >> 18) & 0x7ff;
  }
}

static inline void
temu_canMakeBasicFrame(temu_CanFrame *Frame, uint32_t Ident, uint32_t RTR,
                       uint32_t Length, uint8_t *Data)
{
  Frame->Flags = (Ident << 18) | (RTR << 30);
  Frame->Length = (uint8_t)Length;

  if (Length && !RTR) {
    memcpy(Frame->Data, Data, Length > 8 ? 8 : Length);
  }
}

static inline void
temu_canMakeExtFrame(temu_CanFrame *Frame, uint32_t Ident, uint32_t RTR,
                     uint32_t Length, uint8_t *Data)
{
  Frame->Flags = Ident | (RTR << 30) | (1 << 29);
  Frame->Length = (uint8_t)Length;

  if (Length && !RTR) {
    memcpy(Frame->Data, Data, Length > 8 ? 8 : Length);
  }
}

static inline uint32_t
temu_canBitsForFrame(temu_CanFrame *Frame)
{
  if (temu_canIsExtendedFrame(Frame)) {
    uint32_t DataLen = (Frame->Length > 8 ? 8 : Frame->Length) * 8;
    if (temu_canIsRemoteTransmissionRequest(Frame)) {
      DataLen = 0;
    }

    return 1 + 11 + 1 + 1 + 18 + 1 + 2 + 4 + 15 + 1 + 1 + 1 + 7 + DataLen;

  } else {
    uint32_t DataLen = (Frame->Length > 8 ? 8 : Frame->Length) * 8;
    if (temu_canIsRemoteTransmissionRequest(Frame)) {
      DataLen = 0;
    }
    return 1 + 11 + 1 + 1 + 1 + 4 + 15 + 1 + 1 + 1 + 7 + DataLen;
  }
}

typedef struct {
  uint64_t LastReportSentBits; // Number of bits sent during last report
  uint64_t SentBits; // Current number of sent bits
} temu_CanBusStats;

struct temu_CanDevIface;
typedef struct temu_CanDevIface temu_CanDevIface;
struct temu_CanBusIface;
typedef struct temu_CanBusIface temu_CanBusIface;

#define TEMU_CAN_DEV_IFACE_TYPE "CanDevIface"
#define TEMU_CAN_BUS_IFACE_TYPE "CanBusIface"

TEMU_IFACE_REFERENCE_TYPE(temu_CanDev);
TEMU_IFACE_REFERENCE_TYPE(temu_CanBus);

struct temu_CanDevIface {
  void (*connected)(void *Dev, temu_CanBusIfaceRef Bus);
  void (*disconnected)(void *Dev);
  void (*receive)(void *Dev, temu_CanFrame *Frame);
};

struct temu_CanBusIface {
  void (*connect)(void *Bus, temu_CanDevIfaceRef Dev);
  void (*disconnect)(void *Bus, temu_CanDevIfaceRef Dev);

  void (*send)(void *Bus, void *Sender, temu_CanFrame *Frame);
  void (*enableSendEvents)(void *Bus);
  void (*disableSendEvents)(void *Bus);
  void (*reportStats)(void *Bus);

  // High performance bus models can respond to setFilter to filter
  // data based on the flags field. The use of a filtering bus model
  // means that the filtered frames will be targeted at specific devices
  // attached to the bus. Note that a device model should not assume that
  // the bus model will filter. Infact, the default bundled bus model
  // will simply broadcast the messages to all attached devices.
  // In case the bus has a lot of devices, it may make sense to use
  // a bus with routing based on filtering. For that reason a device
  // can request the bus to filter based on a mask and a code, filtering
  // is done by anding Frame->Flags with Mask and comparing with Code.
  // Note that a device must still do its own filtering.
  void (*setFilter)(void *Bus, temu_CanDevIfaceRef Dev, int FilterID,
                    uint32_t Mask, uint32_t Code);
};

#ifdef __cplusplus
}
#endif

#endif /* !TEMU_CAN_H */
