//===------------------------------------------------------------*- C++ -*-===//
//
// TEMU: The Terma Emulator
// (c) Terma 2024
// Authors: Mattias Holm <maho (at) terma.com>
//
//===----------------------------------------------------------------------===//
#ifndef NEW_REGISTER_API_H
#define NEW_REGISTER_API_H

/*
 * WARNING: This is an experimental API at the moment, do not use this
 *          on non-Terma models, and do not rely on this in production code yet.
 *          This is very volatile at the time of writing.
 */

#include "temu-c/Memory/Memory.h"
#include "temu-c/Support/Notifications.h"
#include "temu-c/Support/Objsys.h"
#include <stddef.h>
#include <stdint.h>

#ifdef __cplusplus
extern "C" {
#endif

typedef struct temu_FieldEnum {
  const char *Name;
  const char *Desc;
  uint64_t Value;
} temu_FieldEnum;

typedef struct temu_FieldMetaData {
  const char *Name;
  const char *Desc;

  uint64_t Mask;

  // Used for symbolic names of values for a register
  unsigned NumberOfEnumerators;
  temu_FieldEnum *Enumerators;
} temu_FieldMetaData;

typedef struct temu_RegisterMetaData {
  const char *Name;
  const char *Desc;

  size_t NumAltNames;
  const char **AltNames;

  // Index of register
  unsigned Index;

  // Offset from device mapping / bank
  unsigned DeviceOffset;

  // Stride in addresses to get next register
  unsigned Stride;

  unsigned Size; // Number of bytes the register is

  // It is an open question whether we want separate cold and warm reset values.
  // The difference is probably expressed by the use of the mask or not instead.
  uint64_t ColdResetValue;

  uint64_t WarmResetMask;
  uint64_t WarmResetValue;

  uint64_t ReadMask;
  uint64_t WriteMask;

  unsigned Readable : 1; // Register is readable
  unsigned Writable : 1; // Register is writable

  unsigned ReadOnce : 1;  // Value is destroyed when read
  unsigned WriteOnce : 1; // Value cannot be overwritten

  unsigned SideEffects : 1; // If writing / reading this register can change
                            // values of other registers.
  unsigned LeftToRight : 1; // Numbering of bits in register (for presentation
                            // purpose only)

  unsigned Volatile : 1; // Register can change, even if target is stopped
  unsigned BigEndian : 1;
  unsigned Float : 1; // Register is floating point
  unsigned Zero : 1;  // Register may be used as zero value in architecture

  // What about register windows, and banked registers like the higher ARM
  // registers? Is it enough with a bit for banked behaviour?
  unsigned Banked : 1; // Register may be banked (can be hidden),
                       // target / device specific as to how banking is done.

  unsigned MemoryMapped : 1; // Register is a memory mapped register

  // These are related to processor registers
  unsigned ProcessorRegister : 1;  // Register is a processor register
  unsigned ProgramCounter : 1;     // Register is a program counter register
  unsigned NextProgramCounter : 1; // Register is a nPC style value
  unsigned SP : 1;                 // Register is a stack pointer
  unsigned FP : 1;                 // Register is a frame pointer
  unsigned Ret : 1;                // Register is used to track return values.
  unsigned GPR : 1;                // General purpose register
  unsigned SPR : 1;                // Special purpose register

  unsigned GDBNumberIsValid : 1;   // Flag indicating validity of GDB number
  unsigned DWARFNumberIsValid : 1; // Flag indicating validity of DWARF number

  unsigned GDB;   // GDB register number.
  unsigned DWARF; // DWARF register number.

  // The purpose of these is to do debugging accesses to registers,
  // hence the read operation should not trigger side effects,
  // while the write obviously should.
  // Pass in Index above to these
  uint64_t (*IndexedRead)(void *, unsigned index);
  void (*IndexedWrite)(void *, unsigned index, uint64_t value);

  // For memory mapped registers we can add direct transaction handlers
  // for this register. Pass in offset in the mt offset to use these.
  // The assumption is that we set TEMU_MT_PROBE on these (when reading),
  // which can be used to eliminate side effects on reads (this is up to the
  // device to ensure).
  void (*MemRead)(void *obj, temu_MemTransaction *mt);
  void (*MemWrite)(void *obj, temu_MemTransaction *mt);

  // These are used to install pre/post access/reset handlers.
  // It should be set to offsetof(mydevice, regstruct), where regstruct is
  // either temu_Register8, temu_Register16, temu_Register32, temu_Register64.
  // Note that CPU registers do not have to set this offset.
  unsigned StandardRegister : 1; // Flag indicating that this register
                                 // is represented by is a
                                 // TEMU register struct (the exact type
                                 // 8,16,32,64 is determined by the size field).
  int OffsetToRegStruct;

  size_t NumFields;
  temu_FieldMetaData *Field;
} temu_RegisterMetaData;

typedef struct {
  const char *Name;
  const char *Desc;

  unsigned MemoryMappedRegisters : 1;
  unsigned ProcessorRegisters : 1;

  size_t NumRegisters;
  temu_RegisterMetaData *RegInfo;
} temu_BankMetaData;

typedef struct {
  size_t (*getNumberOfBanks)(void *obj);
  const temu_BankMetaData *(*getBankInfo)(void *obj, size_t index);
} temu_RegisterIface;
TEMU_IFACE_REFERENCE_TYPE(temu_Register);
#define TEMU_REGISTER_IFACE_TYPE "temu::RegisterIface"

/*!
 * 8 bit memory mapped register
 */
struct temu_Register8 {
  uint8_t Value;
  uint8_t ReadMask;
  uint8_t WriteMask;
  uint8_t ColdResetValue;
  uint8_t ResetValue;
  uint8_t ResetMask;

  uint8_t ForcedBits;
  uint8_t ForcedFlippedBits;

  // These are user configurable callbacks, that can be used to intercept
  // specific regiter reads / writes.

  // Invoked before a read has a semantic effect
  void (*preRead)(void *, temu_MemTransaction *mt);
  // Invoked after a read has a semantic effect
  void (*postRead)(void *, temu_MemTransaction *mt);
  // Invoked before a write has a semantic effect
  void (*preWrite)(void *, temu_MemTransaction *mt);
  // Invoked after a write has a semantic effect
  void (*postWrite)(void *, temu_MemTransaction *mt);

  // Invoked before register is reset
  void (*preReset)(void *);
  // Invoked after a reset has been done
  void (*postReset)(void *);

  int64_t PreReadNotification;
  int64_t PostReadNotification;
  int64_t PreWriteNotification;
  int64_t PostWriteNotification;
  int64_t PreResetNotification;
  int64_t PostResetNotification;

  temu_RegisterMetaData *MetaData; // Owned by class
};

/*!
 * 16 bit memory mapped register
 */
struct temu_Register16 {
  uint16_t Value;
  uint16_t ReadMask;
  uint16_t WriteMask;
  uint16_t ColdResetValue;
  uint16_t ResetValue;
  uint16_t ResetMask;

  uint16_t ForcedBits;
  uint16_t ForcedFlippedBits;

  // These are user configurable callbacks, that can be used to intercept
  // specific regiter reads / writes.
  // Maybe we should simply register these as notifications instead

  // Invoked before a read has a semantic effect
  void (*preRead)(void *, temu_MemTransaction *mt);
  // Invoked after a read has a semantic effect
  void (*postRead)(void *, temu_MemTransaction *mt);
  // Invoked before a write has a semantic effect
  void (*preWrite)(void *, temu_MemTransaction *mt);
  // Invoked after a write has a semantic effect
  void (*postWrite)(void *, temu_MemTransaction *mt);

  // Invoked before register is reset
  void (*preReset)(void *);
  // Invoked after a reset has been done
  void (*postReset)(void *);

  int64_t PreReadNotification;
  int64_t PostReadNotification;
  int64_t PreWriteNotification;
  int64_t PostWriteNotification;
  int64_t PreResetNotification;
  int64_t PostResetNotification;

  temu_RegisterMetaData *MetaData; // Owned by class
};

/*!
 * 32 bit memory mapped register
 */
struct temu_Register32 {
  uint32_t Value;
  uint32_t ReadMask;
  uint32_t WriteMask;
  uint32_t ColdResetValue;
  uint32_t ResetValue;
  uint32_t ResetMask;

  uint32_t ForcedBits;
  uint32_t ForcedFlippedBits;

  // These are user configurable callbacks, that can be used to intercept
  // specific regiter reads / writes.

  // Invoked before a read has a semantic effect
  void (*preRead)(void *, temu_MemTransaction *mt);
  // Invoked after a read has a semantic effect
  void (*postRead)(void *, temu_MemTransaction *mt);
  // Invoked before a write has a semantic effect
  void (*preWrite)(void *, temu_MemTransaction *mt);
  // Invoked after a write has a semantic effect
  void (*postWrite)(void *, temu_MemTransaction *mt);

  // Invoked before register is reset
  void (*preReset)(void *);
  // Invoked after a reset has been done
  void (*postReset)(void *);
  int64_t PreReadNotification;
  int64_t PostReadNotification;
  int64_t PreWriteNotification;
  int64_t PostWriteNotification;
  int64_t PreResetNotification;
  int64_t PostResetNotification;
  temu_RegisterMetaData *MetaData; // Owned by class
};

/*!
 * 64 bit memory mapped register
 */
struct temu_Register64 {
  uint64_t Value;
  uint64_t ReadMask;
  uint64_t WriteMask;
  uint64_t ColdResetValue;
  uint64_t ResetValue;
  uint64_t ResetMask;

  uint64_t ForcedBits;
  uint64_t ForcedFlippedBits;

  int64_t PreReadNotification;
  int64_t PostReadNotification;
  int64_t PreWriteNotification;
  int64_t PostWriteNotification;
  int64_t PreResetNotification;
  int64_t PostResetNotification;

  temu_RegisterMetaData *MetaData; // Owned by class
};

void temu_addMemoryMappedRegister8(temu_Class *c, const char *name, int offset,
                                   int count, const char *doc);
void temu_addMemoryMappedRegister16(temu_Class *c, const char *name, int offset,
                                    int count, const char *doc);
void temu_addMemoryMappedRegister32(temu_Class *c, const char *name, int offset,
                                    int count, const char *doc);
void temu_addMemoryMappedRegister64(temu_Class *c, const char *name, int offset,
                                    int count, const char *doc);

bool temu_registerRead(temu_Object *obj, const char *bank, const char *reg,
                       uint64_t *value, unsigned *size);
bool temu_registerWrite(temu_Object *obj, const char *bank, const char *reg,
                        uint64_t value);

temu_RegisterMetaData *
temu_registerGetMetaData(temu_Object *obj, const char *bank, const char *reg);

void temu_memoryMappedRegisterEnablePreRead(temu_Object *obj, const char *bank,
                                            const char *reg);

void temu_memoryMappedRegisterEnablePostRead(temu_Object *obj, const char *bank,
                                             const char *reg);

void temu_memoryMappedRegisterEnablePreWrite(temu_Object *obj, const char *bank,
                                             const char *reg);

void temu_memoryMappedRegisterEnablePostWrite(temu_Object *obj,
                                              const char *bank,
                                              const char *reg);

void temu_memoryMappedRegisterEnablePreReset(temu_Object *obj, const char *bank,
                                             const char *reg);

void temu_memoryMappedRegisterEnablePostReset(temu_Object *obj,
                                              const char *bank,
                                              const char *reg);

void temu_forEachDeviceWithRegisters(void (*func)(void *, void *), void *arg);

void temu_autoInitRegisters(temu_Object *obj, temu_BankMetaData *bank);

#ifdef __cplusplus
}
#endif

#define TEMU_PRE_RESET(r)                                                      \
  do {                                                                         \
    if (r.PreResetNotification)                                                \
      temu_notify(r.PreResetNotification, NULL);                               \
  } while (0)

#define TEMU_POST_RESET(r)                                                     \
  do {                                                                         \
    if (r.PostResetNotification)                                               \
      temu_notify(r.PostResetNotification, NULL);                              \
  } while (0)

#define TEMU_RESET_REGISTER(w, r)                                              \
  do {                                                                         \
    TEMU_PRE_RESET(r);                                                         \
    if (w) {                                                                   \
      r.Value = (r.Value & ~r.ResetMask) | (r.ResetValue & r.ResetMask);       \
    } else {                                                                   \
      r.Value = r.ColdResetValue;                                              \
    }                                                                          \
    TEMU_POST_RESET(r);                                                        \
  } while (0)

#define TEMU_READ_UPDATE(m, r)                                                 \
  do {                                                                         \
    m->Value = (m->Value | r.ForcedBits);                                      \
    m->Value = (m->Value ^ r.ForcedFlippedBits);                               \
  } while (0)

#define TEMU_PRE_READ(m, r)                                                    \
  do {                                                                         \
    if (r.PreReadNotification)                                                 \
      temu_notify(r.PreReadNotification, m);                                   \
  } while (0)

#define TEMU_POST_READ(m, r)                                                   \
  do {                                                                         \
    if (r.PostReadNotification)                                                \
      temu_notify(r.PostReadNotification, m);                                  \
    TEMU_READ_UPDATE(m, r);                                                    \
  } while (0)

#define TEMU_PRE_WRITE(m, r)                                                   \
  do {                                                                         \
    if (r.PreWriteNotification)                                                \
      temu_notify(r.PreWriteNotification, m);                                  \
  } while (0)

#define TEMU_POST_WRITE(m, r)                                                  \
  do {                                                                         \
    if (r.PostWriteNotification)                                               \
      temu_notify(r.PostWriteNotification, m);                                 \
  } while (0)

#endif // !NEW_REGISTER_API_H
