//===-- temu-c/PowerPC.h - PowerPC Architecture Interfaces --------*- C++
//-*-===//
//
// TEMU: The Terma Emulator
// (c) Terma 2018
// Authors: Mattias Holm <maho (at) terma.com>
//
//===----------------------------------------------------------------------===//

#ifndef TEMU_POWERPC_H
#define TEMU_POWERPC_H

#include "temu-c/Memory/Memory.h"
#include "temu-c/Support/Objsys.h"
#include <stdint.h>

#ifdef __cplusplus
extern "C" {
#endif

/*!
  Interface for PowerPC specific functionality
 */
typedef struct temu_PowerPCIface {
  uint64_t (*getXER)(const void *Cpu);       //!< Get XER register
  void (*setXER)(void *Cpu, uint64_t Value); //!< Set XER register
  uint32_t (*getCR)(const void *Cpu);        //!< Get CR register
  void (*setCR)(void *Cpu, uint32_t Value);  //!< Set XER register
  uint64_t (*getMSR)(const void *Cpu);       //!< Get MSR register
  void (*setMSR)(void *Cpu, uint64_t Value); //!< Set MSR register
  uint64_t (*getReserveAddress)(
      const void *Obj); //!< Get reserved address as used by `lwarx` and `stwcx`
  void (*setReserveAddress)(const void *Obj,
                            uint64_t address); //!< Set reserved address as used
                                               //!< by `lwarx` and `stwcx`
  int (*isReservationBitSet)(
      const void *Obj); //!< Check if reserved address as used by `lwarx` and
                        //!< `stwcx` is set
  void (*clearAddressReservation)(
      const void
          *Obj); //!< Clear reserved address as used by `lwarx` and `stwcx`
  uint32_t (*getCTR)(const void *Cpu);         //!< Get CTR register
  void (*setCTR)(void *Cpu, uint32_t Value);   //!< Set CTR register
  uint64_t (*getLR)(const void *Cpu);          //!< Get LR register
  void (*setLR)(void *Cpu, uint64_t Value);    //!< Set LR register
  uint32_t (*getFPSCR)(const void *Cpu);       //!< Get FPSCR register
  void (*setFPSCR)(void *Cpu, uint32_t Value); //!< Set FPSCR register
  void (*installEAResource)(
      void *Obj, unsigned RID,
      temu_MemAccessIfaceRef Device); //!< Install EA resource
} temu_PowerPCIface;

#define TEMU_POWERPC_IFACE_TYPE "temu::PowerPCIface"
TEMU_IFACE_REFERENCE_TYPE(temu_PowerPC);

// TLB entries for the E500 MMU are defined to contain a certain number of bits.
// These bits are essentially the contents of the MAS1, 2, 3 and 7 registers.
// These are the only SW visible parts of the MMU TLBs,
typedef struct {
  unsigned V : 1;
  unsigned TS : 1;
  unsigned TID : 2;
  unsigned EPN : 20;
  unsigned RPN : 20;
  unsigned SIZE : 4;
  unsigned SX : 1;
  unsigned SR : 1;
  unsigned SW : 1;
  unsigned UX : 1;
  unsigned UR : 1;
  unsigned UW : 1;
  unsigned WIMGE : 5;
  unsigned X0 : 1;
  unsigned X1 : 1;
  unsigned U0 : 1;
  unsigned U1 : 1;
  unsigned U2 : 1;
  unsigned U3 : 1;
  unsigned IPROT : 1;
} temu_E500TLBEntry;

typedef struct {
  temu_E500TLBEntry (*getTLB0Entry)(void *obj, int set, int way);
  void (*setTLB0Entry)(void *obj, int set, int way,
                       const temu_E500TLBEntry *entry);

  temu_E500TLBEntry (*getTLB1Entry)(void *obj, int entryId);
  void (*setTLB1Entry)(void *obj, int entryId, const temu_E500TLBEntry *entry);
} temu_E500MMUIface;
#define TEMU_E500_MMU_IFACE_TYPE "temu::E500MMUIface"
TEMU_IFACE_REFERENCE_TYPE(temu_E500MMU);

#ifdef __cplusplus
}
#endif

#endif /* ! TEMU_POWERPC_H */
