#include "temu-c/Support/Attributes.h"
#include "temu-c/Support/Logging.h"
#include "temu-c/Support/Notifications.h"
#include "temu-c/Support/Objsys.h"
#include "temu-c/Bus/Can.h"

#include <stdint.h>
#include <string.h>

namespace {

struct CANBus {
  temu_Object Super;
  temu_CanDevIfaceRefArray Devices;
  temu_CanBusStats Stats;

  int64_t StatEventId;
  int64_t SendEventId;
};
void *create(const char *Name, int argc, const temu_CreateArg *argv);
void dispose(void *Obj);
void connect(void *Obj, temu_CanDevIfaceRef Ref);
void disconnect(void *Obj, temu_CanDevIfaceRef Ref);
void send(void *Obj, void *Sender, temu_CanFrame *Frame);
void enableSendEvents(void *Obj);
void disableSendEvents(void *Obj);
void reportStats(void *Obj);
void setFilter(void *Obj, temu_CanDevIfaceRef Dev, int FilterID, uint32_t Mask,
               uint32_t Code);

temu_CanBusIface BusIface = {
    connect,           disconnect,  send,      enableSendEvents,
    disableSendEvents, reportStats, setFilter,
};

void *
create(const char *Name TEMU_UNUSED, int argc TEMU_UNUSED,
       const temu_CreateArg *argv TEMU_UNUSED)
{
  CANBus *Bus = new CANBus;
  memset(Bus, 0, sizeof(CANBus));
  Bus->Devices = temu_CanDevIfaceRefArrayAlloc(8);

  Bus->StatEventId = temu_publishNotification("temu.canStat", &Bus->Super);
  Bus->SendEventId = temu_publishNotification("temu.canSend", &Bus->Super);
  return Bus;
}

void
dispose(void *Obj)
{
  CANBus *Bus = reinterpret_cast<CANBus *>(Obj);
  temu_CanDevIfaceRefArrayDispose(&Bus->Devices);
  delete Bus;
}

void
connect(void *Obj, temu_CanDevIfaceRef Ref)
{
  CANBus *Bus = reinterpret_cast<CANBus *>(Obj);
  temu_CanDevIfaceRefArrayPush(&Bus->Devices, Ref);

  temu_CanBusIfaceRef BusRef = {&Bus->Super, &BusIface};
  Ref.Iface->connected(Ref.Obj, BusRef);
}

void
disconnect(void *Obj, temu_CanDevIfaceRef Ref)
{
  CANBus *Bus = reinterpret_cast<CANBus *>(Obj);
  for (size_t i = 0; i < Bus->Devices.Size; ++i) {
    if (Bus->Devices.Ifaces[i].Obj == Ref.Obj &&
        Bus->Devices.Ifaces[i].Iface == Ref.Iface) {
      Bus->Devices.Ifaces[i].Obj = nullptr;
      Bus->Devices.Ifaces[i].Iface = nullptr;

      // Move devices so the array keeps its denseness
      memmove(&Bus->Devices.Ifaces[i], &Bus->Devices.Ifaces[i + 1],
              sizeof(temu_CanDevIfaceRef) *
                  temu_CanDevIfaceRefArraySize(&Bus->Devices) - (i + 1));
      // Remove last element
      temu_CanDevIfaceRefArrayPop(&Bus->Devices);
      return;
    }
  }
}

void
send(void *Obj, void *Sender, temu_CanFrame *Frame)
{
  CANBus *Bus = reinterpret_cast<CANBus *>(Obj);
  temu_logDebug(Obj, "Send CAN message");

  temu_notifyFast(&Bus->SendEventId, Frame);

  Bus->Stats.SentBits += temu_canBitsForFrame(Frame);

  for (size_t i = 0; i < Bus->Devices.Size; ++i) {
    // Avoid passing the message to the sender
    if (Sender) {
      temu_logDebug(Obj, "Send CAN message from %s to %s",
                    ((temu_Object *)Sender)->Name,
                    Bus->Devices.Ifaces[i].Obj->Name);
    } else {
      temu_logDebug(Obj, "Send CAN message to %s",
                    Bus->Devices.Ifaces[i].Obj->Name);
    }
    if (Bus->Devices.Ifaces[i].Obj != Sender) {
      Bus->Devices.Ifaces[i].Iface->receive(Bus->Devices.Ifaces[i].Obj, Frame);
    }
  }
}

void
enableSendEvents(void *Obj)
{
  CANBus *Bus = reinterpret_cast<CANBus *>(Obj);
  Bus->SendEventId = temu_publishNotification("temu.canSend", &Bus->Super);
}

void
disableSendEvents(void *Obj)
{
  CANBus *Bus = reinterpret_cast<CANBus *>(Obj);
  Bus->SendEventId = 0;
}

void
reportStats(void *Obj)
{
  CANBus *Bus = reinterpret_cast<CANBus *>(Obj);
  temu_notifyFast(&Bus->StatEventId, &Bus->Stats);
  Bus->Stats.LastReportSentBits = Bus->Stats.SentBits;
}

// High performance bus models can respond to setFilter to filter
// data based on the flags field. The use of a filtering bus model
// means that the filtered frames will be targeted at specific devices
// attached to the bus. Note that a device model should not assume that
// the bus model will filter. Infact, the default bundled bus model
// will simply broadcast the messages to all attached devices.
// In case the bus has a lot of devices, it may make sense to use
// a bus with routing based on filtering. For that reason a device
// can request the bus to filter based on a mask and a code, filtering
// is done by anding Frame->Flags with Mask and comparing with Code.
// Note that a device must still do its own filtering.
void
setFilter(void *Obj TEMU_UNUSED, temu_CanDevIfaceRef Dev TEMU_UNUSED,
          int FilterID TEMU_UNUSED, uint32_t Mask TEMU_UNUSED,
          uint32_t Code TEMU_UNUSED)
{
}

} // namespace


TEMU_PLUGIN_INIT
{
  temu_Class *Cls = temu_registerClass("CustomCANBus", create, dispose);
  temu_addInterface(Cls, "CanBusIface", "CanBusIface", &BusIface, 0,
                    "CAN Bus Interface");

  temu_addProperty(Cls, "devices", offsetof(CANBus, Devices),
                   teTY_IfaceRefArray, 1, nullptr, nullptr,
                   "CAN devices attached to bus");

  temu_addProperty(Cls, "stats.lastReportSentBits",
                   offsetof(CANBus, Stats.LastReportSentBits), teTY_U64, 1,
                   nullptr, nullptr, "Statistics");

  temu_addProperty(Cls, "stats.sentBits", offsetof(CANBus, Stats.SentBits),
                   teTY_U64, 1, nullptr, nullptr, "Statistics");
}
