#include <stdint.h>
#include <stdio.h>
#include <string.h>

#include "temu-c/Bus/Can.h"
#include "temu-c/Support/Attributes.h"
#include "temu-c/Support/Logging.h"
#include "temu-c/Support/Objsys.h"

typedef struct {
  temu_Object Super;

  temu_CanBusIfaceRef Bus;
} CanDevice;

void *
create(const char *Name, int Argc TEMU_UNUSED,
       const temu_CreateArg *Argv TEMU_UNUSED)
{
  void *Obj = malloc(sizeof(CanDevice));
  memset(Obj, 0, sizeof(CanDevice));

  printf("Creating Object '%s'\n", Name);

  return Obj;
}

void
destroy(void *Obj)
{
  free(Obj);
}

void
connected(void *Obj, temu_CanBusIfaceRef Bus)
{
  CanDevice *Dev = (CanDevice *)Obj;
  Dev->Bus = Bus;
}

void
disconnected(void *Obj)
{
  CanDevice *Dev = (CanDevice *)Obj;

  // If there are pending bus transactions, these should be
  // stopped here.

  Dev->Bus.Iface = NULL;
  Dev->Bus.Obj = NULL;
}

void
receive(void *Obj, temu_CanFrame *Frame)
{
  CanDevice *Dev = (CanDevice *)Obj;

  int RTR = temu_canIsRemoteTransmissionRequest(Frame);
  int Len = Frame->Length;

  temu_logInfo(Dev, "Received CAN frame, RTR = %d, DLC = %d", RTR, Len);

  if (Len > 8)
    Len = 8;

  if (!RTR) {
    for (int i = 0; i < Len; ++i) {
      temu_logInfo(Dev, "\tFrame data %u", (unsigned)Frame->Data[i]);
    }
  }

  temu_canSetAck(Frame);
}

temu_CanDevIface CanDevIface = {
    connected,
    disconnected,
    receive,
};

TEMU_PLUGIN_INIT
{
  temu_Class *Cls = temu_registerClass("CanDevice", create, destroy);
  temu_addInterfaceReference(Cls, "bus", offsetof(CanDevice, Bus),
                             TEMU_CAN_BUS_IFACE_TYPE, 1, 0, NULL, NULL,
                             "Can bus reference");
  temu_addInterface(Cls, "CanDevIface", "CanDevIface", &CanDevIface, 0,
                    "CAN device iface");
}
