//===------------------------------------------------------------*- C++ -*-===//
//
// TEMU: The Terma Emulator
// (c) Terma 2015
// Authors: Mattias Holm <maho (at) terma.com>
//
//===----------------------------------------------------------------------===//

/*!
  This is the 1553 device modelling API. It provides an interface to
  be used for implementing bus controllers and remote terminals.

  The 1553 device interface is EXPERIMENTAL and UNSTABLE at the
  moment.

  One limitation at the moment is that we do not support a specific
  bus monitor interface. However, it is possible to get reports via
  the "temu.mil1553send" event, this way you can sort of insert a
  virtual bus monitor. It does not let you configure a terminal as a
  bus monitor however. These events take an temu_Mil1553Msg pointer as
  the event info argument.

  The 1553 bus is modeled on a phased message approach, this stem from
  the capability of supporting RT->RT transfers. Without the phases,
  each RT would need to have special logic to decode whether data is
  from the BC or an RT (i.e. is the first word a command or a status
  word).
 */
#ifndef TEMU_MIL_STD_1553_H
#define TEMU_MIL_STD_1553_H

#include "temu-c/Support/Objsys.h"

#include <assert.h>
#include <stdbool.h>
#include <stdint.h>

#ifdef __cplusplus
extern "C" {
#endif

typedef enum temu_Mil1553Error {
  teME_NoError = 0,
  teME_ParityError = 1,
  teME_SyncError = 1 << 1
} temu_Mil1553Error;

// The bus model must register who is the BC, and who is
// the receiving terminal.
typedef enum temu_Mil1553MsgType {
  teMT_Cmd,
  teMT_Data,
  teMT_Stat,
} temu_Mil1553MsgType;

#define TEMU_1553_MSG_FLAG_INHIBIT_A 0x1
#define TEMU_1553_MSG_FLAG_INHIBIT_B 0x2

typedef struct temu_Mil1553Msg {
  uint8_t WordCount; // Number of words in the data block
  temu_Mil1553MsgType
      MsgTyp;            // Pre-computed for convenience, otherwise we need
                         // to construct nasty state machines for every model
  temu_Mil1553Error Err; // We can inject errors in every message. The
                         // interpretation is up to the model, but typically:
                         // parity error applies to command or status word if
                         // cmd or stat type message, and to the last data word,
                         // if the msg is a data transaction.
  uint16_t *Data; // Pointer to data block (contains data, command or status)

  uint32_t Flags;
} temu_Mil1553Msg;

// Compute message transfer time in nanoseconds
// 1553 bus is 1 Mb/sec and we have 16 + 4 bits per word.
#define TEMU_1553_NS_PER_WORD 20000
static inline uint64_t
temu_mil1553TransferTime(unsigned Words)
{
  uint64_t Nanos = Words * TEMU_1553_NS_PER_WORD;
  return Nanos;
}

#define TEMU_1553_BITS_PER_WORD 20
static inline uint64_t
temu_mil1553BitCount(unsigned Words)
{
  return Words * TEMU_1553_BITS_PER_WORD;
}

// Conveniance functions to extract fields from the command word

static inline uint16_t
temu_mil1553CmdRtAddr(temu_Mil1553Msg *Msg)
{
  return (Msg->Data[0] >> 11) & 0x1f;
}

static inline uint16_t
temu_mil1553CmdTR(temu_Mil1553Msg *Msg)
{
  return (Msg->Data[0] >> 10) & 1;
}

static inline uint16_t
temu_mil1553CmdSubAddr(temu_Mil1553Msg *Msg)
{
  return (Msg->Data[0] >> 5) & 0x1f;
}

static inline uint16_t
temu_mil1553CmdWCMC(temu_Mil1553Msg *Msg)
{
  return Msg->Data[0] & 0x1f;
}

// Return non-zero if the command is a mode command (sub addr == 0 or 0x1f)

static inline int
temu_mil1553CmdIsModeCodeCmd(temu_Mil1553Msg *Msg)
{
  if (temu_mil1553CmdSubAddr(Msg) == 0x00 ||
      temu_mil1553CmdSubAddr(Msg) == 0x1f) {
    return 1;
  }

  return 0;
}

// Construct a bus message object from a 1553 command word
static inline temu_Mil1553Msg
temu_mil1553CreateCmd(uint16_t *Cmd)
{
  temu_Mil1553Msg Msg;
  Msg.WordCount = 1;
  Msg.Err = teME_NoError;
  Msg.Data = Cmd;
  Msg.MsgTyp = teMT_Cmd;
  Msg.Flags = 0;
  return Msg;
}

// Create mode code command (with mode = 0)
static inline temu_Mil1553Msg
temu_mil1553CreateModeTransmittMsg0(uint16_t *Mode, uint16_t RTAddr,
                                    uint16_t MC)
{
  temu_Mil1553Msg Msg;
  Msg.WordCount = 1;
  Msg.MsgTyp = teMT_Cmd;
  Msg.Err = teME_NoError;
  *Mode = (RTAddr << 11) | (1 << 10) | MC;
  Msg.Data = Mode;
  Msg.Flags = 0;
  return Msg;
}

// Create mode code command (with mode = 0)
static inline temu_Mil1553Msg
temu_mil1553CreateModeRecvMsg0(uint16_t *Mode, uint16_t RTAddr, uint16_t MC)
{
  temu_Mil1553Msg Msg;
  Msg.WordCount = 1;
  Msg.MsgTyp = teMT_Cmd;
  Msg.Err = teME_NoError;
  *Mode = (RTAddr << 11) | MC;
  Msg.Data = Mode;
  Msg.Flags = 0;
  return Msg;
}

// Create mode code command (with mode = 0x1f)
static inline temu_Mil1553Msg
temu_mil1553CreateModeTransmittMsg1(uint16_t *Mode, uint16_t RTAddr,
                                    uint16_t MC)
{
  temu_Mil1553Msg Msg;
  Msg.WordCount = 1;
  Msg.MsgTyp = teMT_Cmd;
  Msg.Err = teME_NoError;
  *Mode = (RTAddr << 11) | (1 << 10) | (0x1f << 5) | MC;
  Msg.Data = Mode;
  Msg.Flags = 0;
  return Msg;
}

// Create mode code command (with mode = 0x1f)
static inline temu_Mil1553Msg
temu_mil1553CreateModeRecvMsg1(uint16_t *Mode, uint16_t RTAddr, uint16_t MC)
{
  temu_Mil1553Msg Msg;
  Msg.WordCount = 1;
  Msg.MsgTyp = teMT_Cmd;
  Msg.Err = teME_NoError;
  *Mode = (RTAddr << 11) | (0x1f << 5) | MC;
  Msg.Data = Mode;
  Msg.Flags = 0;
  return Msg;
}

static inline temu_Mil1553Msg
temu_mil1553CreateStatMsg(uint16_t *Stat, uint16_t RTAddr)
{
  temu_Mil1553Msg Msg;
  Msg.WordCount = 1;
  Msg.MsgTyp = teMT_Stat;
  Msg.Err = teME_NoError;
  *Stat = RTAddr << 11;
  Msg.Data = Stat;
  Msg.Flags = 0;
  return Msg;
}

static inline temu_Mil1553Msg
temu_mil1553CreateDataMsg(uint16_t *Data, size_t Words)
{
  temu_Mil1553Msg Msg;
  Msg.WordCount = Words;
  Msg.MsgTyp = teMT_Data;
  Msg.Err = teME_NoError;
  Msg.Data = Data;
  Msg.Flags = 0;
  return Msg;
}

static inline temu_Mil1553Msg
temu_mil1553CreateRecvCmdMsg(uint16_t *CmdWord, uint16_t RTAddr,
                             uint16_t SubAddr, uint16_t Words)
{
  temu_Mil1553Msg Msg;
  Msg.WordCount = 1;
  Msg.MsgTyp = teMT_Cmd;
  Msg.Err = teME_NoError;
  Msg.Data = CmdWord;
  *CmdWord = (RTAddr << 11) | (SubAddr << 5) | (Words & 0x1f);
  Msg.Flags = 0;
  return Msg;
}

static inline temu_Mil1553Msg
temu_mil1553CreateTransmittCmdMsg(uint16_t *CmdWord, uint16_t RTAddr,
                                  uint16_t SubAddr, uint16_t Words)
{
  temu_Mil1553Msg Msg;
  Msg.WordCount = 1;
  Msg.MsgTyp = teMT_Cmd;
  Msg.Err = teME_NoError;
  Msg.Data = CmdWord;
  *CmdWord = (RTAddr << 11) | (1 << 10) | (SubAddr << 5) | (Words & 0x1f);
  Msg.Flags = 0;
  return Msg;
}

typedef struct temu_Mil1553DevIface temu_Mil1553DevIface;
typedef struct temu_Mil1553BusIface temu_Mil1553BusIface;
TEMU_IFACE_REFERENCE_TYPE(temu_Mil1553Dev);
TEMU_IFACE_REFERENCE_TYPE(temu_Mil1553Bus);

typedef enum temu_Mil1553BusResetType {
  teMBR_Nominal = 0,
  // To be implemented in the future:
  // teMBR_FaultyRT = 1,
  // teMBR_BusTimeOut = 1 << 1
} temu_Mil1553BusResetType;

typedef struct {
  temu_Mil1553BusResetType ResetType;
} temu_Mil1553BusIdleInfo;

struct temu_Mil1553DevIface {
  //! Called after device is connected to bus
  void (*connected)(void *Device, temu_Mil1553BusIfaceRef Bus, int RemoteTerminalAddr);
  //! Called after device is disconnected
  void (*disconnected)(void *Device, temu_Mil1553BusIfaceRef Bus, int RemoteTerminalAddr);

  //! Receive of 1553 message
  void (*receive)(void *Device, temu_Mil1553Msg *Msg);
  //! Notifies the bus controller the bus enters an idle
  void (*busEnteredIdle)(void *Bus, temu_Mil1553BusIdleInfo *idleInfo);
};
#define TEMU_MIL1553_DEV_IFACE_TYPE "Mil1553DevIface"

struct temu_Mil1553BusIface {
  void (*connect)(void *Bus, int RTAddr, temu_Mil1553DevIfaceRef Device);
  void (*disconnect)(void *Bus, int RTAddr);
  void (*reportStats)(void *Bus);
  void (*send)(void *Bus, void *Sender, temu_Mil1553Msg *Msg);

  // Controls whether events should be issued at send calls
  void (*enableSendEvents)(void *Bus);
  void (*disableSendEvents)(void *Bus);
  void (*setBusController)(void *Bus, temu_Mil1553DevIfaceRef Device);
};
#define TEMU_MIL1553_BUS_IFACE_TYPE "Mil1553BusIface"

typedef struct {
  uint64_t LastReportSentWords; // Number of words sent during last report
  uint64_t SentWords;           // Current number of sent words
} temu_Mil1553Stats;

// For accessing 16 bit words
static inline uint16_t
temu_mil1553CmdWordRa(uint16_t Msg)
{
  return (Msg >> 11) & 0x1f;
}

static inline uint16_t
temu_mil1553CmdWordSa(uint16_t Msg)
{
  return (Msg >> 5) & 0x1f;
}

static inline uint16_t
temu_mil1553CmdWordTr(uint16_t Msg)
{
  return (Msg >> 10) & 0x1;
}

static inline uint16_t
temu_mil1553CmdWordCount(uint16_t Msg)
{
  return Msg & 0x1f;
}

static inline uint16_t
temu_mil1553ModeWordCode(uint16_t Msg)
{
  return Msg & 0x1f;
}

static inline uint16_t
temu_mil1553StatWordRtAddr(uint16_t Msg)
{
  return (Msg >> 11) & 0x1f;
}

static inline uint16_t
temu_mil1553StatWordME(uint16_t Msg)
{
  return (Msg >> 10) & 1;
}

static inline uint16_t
temu_mil1553StatWordInst(uint16_t Msg)
{
  return (Msg >> 9) & 1;
}

static inline uint16_t
temu_mil1553StatWordSR(uint16_t Msg)
{
  return (Msg >> 8) & 1;
}

static inline uint16_t
temu_mil1553StatWordBC(uint16_t Msg)
{
  return (Msg >> 4) & 1;
}

static inline uint16_t
temu_mil1553StatWordBusy(uint16_t Msg)
{
  return (Msg >> 3) & 1;
}

static inline uint16_t
temu_mil1553StatWordSubFlag(uint16_t Msg)
{
  return (Msg >> 2) & 1;
}

static inline uint16_t
temu_mil1553StatWordDynBusCtrl(uint16_t Msg)
{
  return (Msg >> 1) & 1;
}

static inline uint16_t
temu_mil1553StatWordTermFlag(uint16_t Msg)
{
  return Msg & 1;
}

// Accessing messages
static inline uint16_t
temu_mil1553CmdCount(temu_Mil1553Msg *Msg)
{
  return Msg->Data[0] & 0x1f;
}

static inline uint16_t
temu_mil1553ModeCode(temu_Mil1553Msg *Msg)
{
  return Msg->Data[0] & 0x1f;
}

static inline uint16_t
temu_mil1553StatRtAddr(temu_Mil1553Msg *Msg)
{
  return (Msg->Data[0] >> 11) & 0x1f;
}

static inline uint16_t
temu_mil1553StatME(temu_Mil1553Msg *Msg)
{
  return (Msg->Data[0] >> 10) & 1;
}

static inline uint16_t
temu_mil1553StatInst(temu_Mil1553Msg *Msg)
{
  return (Msg->Data[0] >> 9) & 1;
}

static inline uint16_t
temu_mil1553StatSR(temu_Mil1553Msg *Msg)
{
  return (Msg->Data[0] >> 8) & 1;
}

static inline uint16_t
temu_mil1553StatBC(temu_Mil1553Msg *Msg)
{
  return (Msg->Data[0] >> 4) & 1;
}

static inline uint16_t
temu_mil1553StatBusy(temu_Mil1553Msg *Msg)
{
  return (Msg->Data[0] >> 3) & 1;
}

static inline uint16_t
temu_mil1553StatSubFlag(temu_Mil1553Msg *Msg)
{
  return (Msg->Data[0] >> 2) & 1;
}

static inline uint16_t
temu_mil1553StatDynBusCtrl(temu_Mil1553Msg *Msg)
{
  return (Msg->Data[0] >> 1) & 1;
}

static inline uint16_t
temu_mil1553StatTermFlag(temu_Mil1553Msg *Msg)
{
  return Msg->Data[0] & 1;
}

static inline bool
temu_mil1553MsgInhibitBusA(const temu_Mil1553Msg *Msg)
{
  return Msg->Flags & TEMU_1553_MSG_FLAG_INHIBIT_A;
}

static inline bool
temu_mil1553MsgInhibitBusB(const temu_Mil1553Msg *Msg)
{
  return Msg->Flags & TEMU_1553_MSG_FLAG_INHIBIT_B;
}

// Useful constants, use temu_mil1553CmdMode() to extract bits,
// these constants are used for the optional mode control codes that
// are defined in the 1553 standard.

#define TEMU_1553_TR_BIT 0x20

#define TEMU_1553_DYN_BUS_CTRL 0x00
#define TEMU_1553_SYNC_NO_DATA 0x01
#define TEMU_1553_TRANSMIT_STATUS_WORD 0x02
#define TEMU_1553_INITIATE_SELF_TEST 0x03
#define TEMU_1553_TRANSMITTER_SHUTDOWN 0x04
#define TEMU_1553_OVERRIDE_TRANSMITTER 0x05
#define TEMU_1553_INHIBIT_TERM_FLAG_BIT 0x06
#define TEMU_1553_OVERRIDE_INHIBIT_TERM_FLAG_BIT 0x07
#define TEMU_1553_RESET_RT 0x08
#define TEMU_1553_TRANSMIT_VECTOR_WORD 0x10
#define TEMU_1553_SYNC_WITH_DATA 0x11
#define TEMU_1553_TRANSMIT_LAST_CMD 0x12
#define TEMU_1553_TRANSMIT_BIT_WORD 0x13
#define TEMU_1553_SELECTED_TRANSMITTER 0x14
#define TEMU_1553_OVERRIDE_SELECTED_TRANSMITTER 0x15

#ifdef __cplusplus
}
#endif

#endif /* ! TEMU_MIL_STD_1553_H */
