//===-- temu-c/Buffer.h - Data logger functions ------------*- C++ -*-===//
//
// TEMU: The Terma Emulator
// (c) Terma 2016
// Authors: Mattias Holm <maho (at) terma.com>
//
//===----------------------------------------------------------------------===//

#ifndef TEMU_SUPPORT_BUFFER_H
#define TEMU_SUPPORT_BUFFER_H

#include <stdint.h>
#include <stdlib.h>
#include "temu-c/Support/Attributes.h"

#ifdef __cplusplus
extern "C" {
#endif

/*!
 * Copy-On-Write (COW) buffer. The content is private and should not
 * be manipulated directly.
 *
 * NOTE: The COW buffers are currently an EXPERIMENTAL feature. The
 * API may change. Further, the temu_Buff type itself may change
 * without notice in future versions of TEMU. The type is internal
 * for the buffer API, but is exposed as a struct here to save on
 * memory allocations when creating a temporary buffer object copy on
 * the stack.
 *
 * The COW buffer objects are suitable for network simulation. Where a
 * receiver may need to queue up multiple received packets.
 *
 * Currently, there is no way to prepend or append (e.g. injecting
 * extra headers). However, it is possible to delete bytes from the
 * head of the buffer without causing copies of the actual buffered
 * data. This is enough to support efficient SpaceWire simulation.
 *
 * Typically a device model would use the buffers in this way:
 *
 * \code{.cpp}
 * // Transaction start
 * void
 * startTransaction(mydevice_t *dev)
 * {
 *   temu_Buff buffer = temu_buffCreate(dev->DataLengthRegister);
 *   uint8_t *bufferData = temu_buffWritableData(&buffer);
 *   dev->memspace.Iface->readBytes(dev->memspace.Obj,
 *                                  bufferData,
 *                                  dev->DataPtrRegister,
 *                                  dev->DataLengthRegister,
 *                                  0); // Data in bytes
 *   sendData(dev->receiver, &buffer);
 *   temu_buffDispose(&buffer);
 * }
 *
 * // RECEIVER functionality
 * void
 * receive(myreceiver_t *receiver, const temu_Buff *buff)
 * {
 *   temu_Buff buffer = temu_buffCopy(buff);
 *   enqueue(receiver, buffer);
 *   temu_eventPostNs(receiver->Super.Queue, receiver->HandleDataEventID,
 *                    1000, teST_Cpu)
 * }
 *
 * void
 * event(temu_Event *ev)
 * {
 *   myreceiver_t *receiver = ev->Obj;
 *   temu_Buff buffer = dequeue(receiver);
 *   const uint8_t *data temu_buffReadabledata(&buffer);
 *   uint32_t bufflen = temu_buffLen(&buffer);
 *   // ...
 *   temu_buffDispose(buffer);
 * }
 *
 * \endcode
 */

#ifndef TEMU_BUFF_DEFINED
#define TEMU_BUFF_DEFINED
typedef struct {
  uintptr_t data0;
  uint32_t data1;
  uint32_t data2;
} temu_Buff;
#endif // !TEMU_BUFF_DEFINED

/*!
 * Create a new COW buffer of size bytes.
 *
 * \param size Size of buffer in bytes
 * \result Buffer object.
 */
TEMU_API temu_Buff temu_buffCreate(uint32_t size);

/*!
 * Copy COW buffer. This will make a new buff object, but the
 * underlying data will not be copied.
 *
 * \param B The buffer to copy.
 * \result Buffer object referring to the same data as B.
 */
TEMU_API temu_Buff temu_buffCopy(const temu_Buff *B);

/*!
 * Delete buffer. The buffer B will be deleted. If there are no more
 * copies of the buffer data anywhere, the data will be deallocated.
 *
 * \param B buffer to delete.
 */
TEMU_API void temu_buffDispose(temu_Buff *B);

/*!
 * Get a pointer to writable data.
 *
 * If the data have more than one reference, a new copy of the data
 * will be created.
 *
 * \param B Buffer to get data pointer from.
 * \result Pointer to writable data.
 */
TEMU_API uint8_t* temu_buffWritableData(temu_Buff *B);

/*!
 * Get a pointer to readable buffer data
 *
 * The pointer to the readable data will be returned.
 *
 * \param B Buffer to get read pointer from
 * \result Pointer to data buffer.
 */
TEMU_API const uint8_t* temu_buffReadableData(const temu_Buff *B);

/*!
 * Get buffer length
 *
 * \param B buffer
 * \result Length of buffer in bytes
 */
TEMU_API uint32_t temu_buffLen(const temu_Buff *B);

/*!
 * Remove len bytes from the start of the buffer.
 *
 * Removing bytes from the buffer start will not change the underlying
 * buffer object. And is not seen as a write for the purpose of the
 * data block.
 *
 * This does not affect other copies of this buffer.
 * \param B buffer to remove data from
 * \param len Number of bytes to remove.
 */
TEMU_API void temu_buffRemoveHead(temu_Buff *B, uint32_t len);

/*!
 * Remove len bytes from the end of the buffer.
 *
 * Removing bytes from the buffer tail will not change the underlying
 * buffer object. And is not seen as a write for the purpose of the
 * data block.
 *
 * This does not affect other copies of this buffer.
 *
 * \param B buffer to remove data from
 * \param len Number of bytes to remove.
 */
TEMU_API void temu_buffRemoveTail(temu_Buff *B, uint32_t len);

#ifdef __cplusplus
}
#endif

#endif /*  ! TEMU_BUFFER_H */
