//===-- temu-c/Console.h - TEMU Standard IO functions ---------*- C++ -*-===//
//
// TEMU: The Terma Emulator
// (c) Terma 2019
// Authors: Mattias Holm <maho (at) terma.com>
//
//===----------------------------------------------------------------------===//
/*!
 * TEMU maintains three different print I/O systems. The logging system is for
 * logging messages that models generates. The Loggin system is defined in
 * temu-c/Support/Logging.h. Logging is categorised and support a number of
 * severity levels.
 *
 * The diagnostic I/O is similar to logging, but is more tailored for
 * diagnostics stemming from internal parsers. Diagnostics include a file name
 * and line+column number.
 *
 * The third type of I/O is the out and err streams. These are essentially
 * directly mapped to stdout and stderr, but TEMU can internally remap these
 * in order to e.g. route or clone I/O to separate files. The I/O streams
 * functions exist primarily to replace printf/cout/cerr and enable programattic
 * rerouting of console i/o.
 */

#ifndef TEMU_CONSOLE_H
#define TEMU_CONSOLE_H

#include "temu-c/Support/Attributes.h"

#ifdef __cplusplus
extern "C" {
#endif
/*!
 * Printf wrapper. The function prints a formatted message to what TEMU
 * considers to be the standard output. This function respects internal
 * rerouting rules. Unlike plain printf, which writes only to stdout.
 *
 * The function is limited in that it keeps a local fixed length buffer
 * for printing. This buffer is 1024 bytes, hence it is not possible to
 * print messages longer than 1023 bytes.
 *
 * \result Number of bytes written
 * */
TEMU_API int temu_printf(const char *Fmt, ...)
    __attribute__((format(printf, 1, 2)));

/*!
 * fprintf(stderr, ...) wrapper. The function prints a formatted message to what
 * TEMU considers to be the standard error. This function respects internal
 * rerouting rules. Unlike plain fprintf, which writes to an explicit file.
 *
 * The function is limited in that it keeps a local fixed length buffer
 * for printing. This buffer is currently 1024 bytes, hence it is not possible
 * to print messages longer than 1023 bytes.
 *
 * \result Number of bytes written
 */

TEMU_API int temu_printerr(const char *Fmt, ...)
    __attribute__((format(printf, 1, 2)));

/*!
 * Print a diagnostic warning
 *
 * The function prints a message like
 *   "warning:<filename>:<line>:<col>: <msg>"
 *   "warning:<filename>:<line>: <msg>"
 *   "warning:<filename>: <msg>"
 *   "warning: <msg>"
 * Depending on whether fileName, line or col is set.
 *
 * \param fileName Optional file name, set ton nullptr to skip
 * \param line Optional line number, set to zero for no line number
 * \param col Optional column number, set to zero for no column number
 * \param fmt printf formatted string.
 * \result length of the message
 */
TEMU_API int temu_printDiagnosticWarning(const char *fileName, unsigned line,
                                         unsigned col, const char *fmt, ...)
    __attribute__((format(printf, 4, 5)));

/*!
 * Print a diagnostic error
 *
 * The function prints a message like
 *   "error:<filename>:<line>:<col>: <msg>"
 *   "error:<filename>:<line>: <msg>"
 *   "error:<filename>: <msg>"
 *   "error: <msg>"
 * Depending on whether fileName, line or col is set.
 *
 * \param fileName Optional file name, set ton nullptr to skip
 * \param line Optional line number, set to zero for no line number
 * \param col Optional column number, set to zero for no column number
 * \param fmt printf formatted string.
 * \result length of the message
 */
TEMU_API int temu_printDiagnosticError(const char *fileName, unsigned line,
                                       unsigned col, const char *fmt, ...)
    __attribute__((format(printf, 4, 5)));

#ifdef __cplusplus
}
#endif

#endif // !TEMU_CONSOLE_H
