#include <inttypes.h>
#include <stdint.h>
#include <stdio.h>
#include <string.h>

#include "temu-c/Support/Logging.h"
#include "temu-c/Support/Objsys.h"

#include "temu-c/Support/Events.h"
#include "temu-c/Target/Cpu.h"

#include "temu-c/Memory/Memory.h"

typedef struct {
  temu_Object Super;

  temu_CpuIfaceRef Cpu;
  int64_t EventID;

  uint32_t RegisterA;
} MMIOModel;

void eventFunc(temu_Event *Event);

void *
create(const char *Name, int Argc, const temu_CreateArg *Argv)
{
  void *Obj = malloc(sizeof(MMIOModel));
  memset(Obj, 0, sizeof(MMIOModel));

  temu_logInfo(NULL, "registering event func");
  MMIOModel *MMIO = (MMIOModel *)Obj;

  // Normally, you connect the time source in a script
  temu_setTimeSource(Obj, (temu_TimeSource*)temu_objectForName("cpu0"));
  MMIO->EventID = temu_eventPublish("myevent", Obj, eventFunc);

  return Obj;
}

void
destroy(void *Obj)
{
  free(Obj);
}

void
eventFunc(temu_Event *Event)
{
  MMIOModel *MMIO = (MMIOModel *)Event->Obj;
  uint32_t PC = MMIO->Cpu.Iface->getPc(MMIO->Cpu.Obj);
  int64_t Cycles = MMIO->Cpu.Iface->getCycles(MMIO->Cpu.Obj);
  temu_logInfo(Event->Obj, "event triggered at PC = %.8x @ %" PRId64, PC,
               Cycles);
}

void
regAWrite(void *Obj, temu_Propval Pv, int Idx)
{
  MMIOModel *MMIO = (MMIOModel *)Obj;
  temu_logInfo(Obj, "posting event in 10 cycles");
  MMIO->RegisterA = temu_propValueU32(Pv);

  temu_eventPostCycles(MMIO->Super.TimeSource, // Time source
                       MMIO->EventID,          // Event ID
                       5,                      // 5 Cycles in the futuru
                       teSE_Cpu);              // Post on CPU queue
}

temu_Propval
regARead(void *Obj, int Idx)
{
  MMIOModel *MMIO = (MMIOModel *)Obj;
  temu_logInfo(Obj, "reading register a");
  return temu_makePropU32(MMIO->RegisterA);
}

void
memRead(void *Obj, temu_MemTransaction *Mt)
{
  MMIOModel *MMIO = (MMIOModel *)Obj;

  switch (Mt->Offset) {
  case 0x0:
    Mt->Value = temu_propValueU32(regARead(MMIO, 0));
    break;
  default:
    temu_logError(MMIO, "read from non-existing register");
  }

  Mt->Cycles = 1; // Access cost
}

void
memWrite(void *Obj, temu_MemTransaction *Mt)
{
  MMIOModel *MMIO = (MMIOModel *)Obj;

  temu_Propval Pv = temu_makePropU32(Mt->Value);

  switch (Mt->Offset) {
  case 0x0:
    regAWrite(MMIO, Pv, 0);
    break;
  default:
    temu_logError(MMIO, "write to non-existing register");
  }

  Mt->Cycles = 1;
}

temu_MemAccessIface MemAccessIface = {
    NULL, // Illegal, cannot fetch from this device model
    memRead,
    memWrite,
};

TEMU_PLUGIN_INIT
{
  temu_Class *Cls = temu_registerClass("InterfaceExample", create, destroy);
  temu_addProperty(Cls, "regA", offsetof(MMIOModel, RegisterA), teTY_U32, 1,
                   regAWrite, regARead, "Register A triggers foo");

  temu_addInterfaceReference(Cls, "cpu", offsetof(MMIOModel, Cpu),
                             TEMU_CPU_IFACE_TYPE, 1, 0, NULL, NULL,
                             "Cpu this device accesses");

  temu_addInterface(Cls,
                    "MemAccessIface", // Interface name
                    "MemAccessIface", // Interface type name
                    &MemAccessIface,
                    1, // Reserved for future, pass 1 for now
                    "My Interface is used for MMIO transactions");
}
