//===------------------------------------------------------------*- C++ -*-===//
//
// TEMU: The Terma Emulator
// (c) Terma 2025
// Authors: Mattias Holm <maho (at) terma.com>
//
//===----------------------------------------------------------------------===//

#ifndef TEMU_EXPERIMENTAL_ETHERNET_H
#define TEMU_EXPERIMENTAL_ETHERNET_H
#include "temu-c/Support/Attributes.h"
#include "temu-c/Support/Buffer.h"
#include "temu-c/Support/Objsys.h"
#include <stdint.h>

#ifdef __cplusplus
extern "C" {
#endif

/*
  This header defines additional experimental ethernet extensions.
  Note that although functionality in models may be stable,
  the APIs in this file are not.
 */

#define TEMU_ETH_FROM_TAP (1 << 10)

typedef struct {
  void (*connected)(void *);
  void (*disconnected)(void *);
  void (*send)(void *, size_t len, const uint8_t *data);
} temu_StreamIface;
#define TEMU_STREAM_IFACE_TYPE "temu::streamIface"
TEMU_IFACE_REFERENCE_TYPE(temu_Stream)

typedef struct {
  int (*connectStream)(void *, uint32_t sourceIP, uint16_t sourcePort,
                       uint16_t targetPort, temu_StreamIfaceRef receiveStream);
  void (*disconnectStream)(void *, int stream);
  void (*send)(void *, int stream, size_t len, const uint8_t *data);
  int (*sendDatagram)(void *, uint32_t sourceIP, uint16_t sourcePort,
                      uint16_t targetPort);
} temu_IPv4NATIface;
#define TEMU_NAT_IFACE_TYPE "temu::ipv4natIface"
TEMU_IFACE_REFERENCE_TYPE(temu_IPv4NAT)

// ARP Resolver, experimental.
typedef struct {
  uint64_t MAC;
  uint32_t IP;

  void *Resolver;
} temu_ARPProtocol;

typedef struct {
  uint64_t MAC;
  uint32_t IP;
} temu_ARPResolution;

typedef enum {
  teARP_BadPacket,
  teARP_RequestHandled,
  teARP_ResponseHandled,
} temu_ARPHandlingResult;

typedef struct {
  uint16_t HType;
  uint16_t PType;
  uint8_t HLen;
  uint8_t PLen;
  uint16_t Oper;
  uint8_t SHA[6];
  uint8_t SPA[4];
  uint8_t THA[6];
  uint8_t TPA[4];
} temu_ARPPacket;

TEMU_API void temu_arpInit(temu_ARPProtocol *arp);
TEMU_API void temu_arpReset(temu_ARPProtocol *arp);
TEMU_API void temu_arpDispose(temu_ARPProtocol *arp);
TEMU_API int temu_arpResolveIP(temu_ARPProtocol *arp, uint64_t MAC,
                               uint32_t *IP);
TEMU_API void temu_arpAddResolution(temu_ARPProtocol *arp, uint64_t MAC,
                                    uint32_t IP);
TEMU_API int temu_arpResolveMAC(temu_ARPProtocol *arp, uint32_t IP,
                                uint64_t *MAC);

TEMU_API size_t temu_arpGetIPResolutionCount(temu_ARPProtocol *arp);
TEMU_API temu_ARPResolution temu_arpGetIPResolution(temu_ARPProtocol *arp,
                                                    int index);

TEMU_API temu_Buff temu_arpMakeRequest(temu_ARPProtocol *prot, uint32_t TargetIP);
TEMU_API temu_Buff temu_arpMakeReply(temu_ARPProtocol *prot,
                                     const uint8_t *requestPtr);

// If result is teARP_RequestHandled,
// then the response buffer has been filled in and it should be sent
// through the network.
TEMU_API temu_ARPHandlingResult temu_arpHandlePacket(temu_ARPProtocol *prot,
                                                     size_t packetLength,
                                                     const uint8_t *packet,
                                                     temu_Buff *response);

TEMU_API temu_ARPPacket temu_arpMakeRequestPacket(uint64_t mac, uint32_t sourceIP, uint32_t targetIP);

TEMU_API temu_ARPPacket temu_arpMakeReplyPacket(temu_ARPPacket *req,
                                                uint64_t mac, uint32_t ip);
TEMU_API void temu_arpSwapPacket(temu_ARPPacket *pkt);
TEMU_API uint64_t temu_arpGetPacketSenderHwAddr(temu_ARPPacket *pkt);
TEMU_API uint32_t temu_arpGetPacketSenderProtAddr(temu_ARPPacket *pkt);

TEMU_API uint64_t temu_arpGetPacketTargetHwAddr(temu_ARPPacket *pkt);

TEMU_API uint32_t temu_arpGetPacketTargetProtAddr(temu_ARPPacket *pkt);

typedef struct {
  uint8_t Type;
  uint8_t Code;
  uint16_t Checksum;
  uint32_t RestOfHeader;
} temu_ICMPPacket;

TEMU_API void temu_icmpSwapPacket(temu_ICMPPacket *pkt);
TEMU_API temu_ICMPPacket temu_icmpMakeEchoRequestPacket();
TEMU_API temu_ICMPPacket temu_icmpMakeEchoReplyPacket();

#ifdef __cplusplus
}
#endif

#endif // !TEMU_EXPERIMENTAL_ETHERNET_H
