//===-- temu-c/Logging.h - Logging functions --------------------*- C++ -*-===//
//
// TEMU: The Terma Emulator
// (c) Terma 2015
// Authors: Mattias Holm <maho (at) terma.com>
//
//===----------------------------------------------------------------------===//

#ifndef TEMU_LOGGING_H
#define TEMU_LOGGING_H

#include "temu-c/Support/Attributes.h"
#include "temu-c/Support/Objsys.h"
#include <stdarg.h>
#include <stdio.h>

/*!
 * \file Logging.h
 *
 * TEMU logging functions. TEMU provides a logging facility to log
 * messages from models. These messages can be of a number of severity
 * levels and will be assigned to a model object. When printing the
 * message will be formatted as follows: "<severity> : <object> :
 * message\n", where <severity> is one of debug, info, warning, error
 * and critical and <object> is the name of the object generating the
 * log message.
 *
 * TEMU provides a "critical" log message, these messages are
 * critical in the sense that after they have been logged, the
 * emulator will terminate abnormally. Critical messages should never
 * happen normally are typically placed in unreachable code locations.
 *
 * Error messages imply that something failed but that the caller of
 * the failed function is expected to take action in some way. Typical
 * case is when opening a file (e.g. an SREC or ELF binary) specified
 * by the user and the file did not exist.
 *
 * Warning message exist to notify the user about something that may
 * be an issue, but not necessarily is. An example is in the object
 * system temu_checkSanity function, which will check whether objects
 * are properly configured by being fully connected.
 *
 * Info messages are normal messages, that exist solely to provide
 * information to the user. For example, a model may print status info
 * after it has been created.
 *
 * Debug messages should not occur during normal use, and the debug
 * function will be a static inline function with no body when
 * building a model with -DNDEBUG. This is important, because the
 * compiler will eliminate the call to the empty function, while still
 * remaining fully typesafe and NOT being a macro.
 *
 */

#ifdef __cplusplus
extern "C" {
#endif

/*!
 *  Logging levels corresponds roughly to some of the RFC 5424 severity
 *  levels.
 */
typedef enum temu_LogLevel {
  teLL_Fatal = 0, //!< Fatal, emulator cannot keep on running
  teLL_Error,     //!< Error happened, in principle critical but up to user
  teLL_Warning,   //!< Warnings
  teLL_Info,      //!< Normal messages
  teLL_Trace,     //!< Trace messages, not compiled away
  teLL_Debug,     //!< Debug
} temu_LogLevel;

/*!
 * Set logging level. Messages will be logged if they are of the set
 * level or higher priority.
 *
 * \param LogLevel The logging level.
 */
TEMU_API void temu_logSetLevel(temu_LogLevel LogLevel);

/*!
 * Set logging function.
 *
 * It is possible to provide a custom function for handling logging
 * messages from TEMU. This can be used by a simulator that provides
 * a centralized logging facility to also handle TEMU logging
 * messages. By default, the messages will be printed to stderr using
 * fputs. Note that messages will be terminated by "\n\0", so if your
 * logging system adds a linefeed, the message may need to be
 * transformed.
 *
 * \param LogFunc Logging function to use. NULL to restore the default.
 */
TEMU_API void temu_logSetFunc(void (*LogFunc)(const char *));

/*!
 * Set advanced logging function.
 *
 * It is possible to provide a custom function for handling logging
 * messages from TEMU. This can be used by a simulator that provides
 * a centralized logging facility to also handle TEMU logging
 * messages. By default, the messages will be printed to stderr using
 * fputs.
 * 
 * The advanced logging function does not get messages with a terminating linefeed.
 * In addition, the function receives the object, category and log level as its own parameters.
 *
 * The message does not contain object name / time stamps.
 * The integrator would be expected to extract the time stamp from the objects time source if set.
 *
 * LogFunc will take the following parameters:
 *
 *   User data:: Void pointer passed to UserData parameter (may be NULL).
 *   Object pointer:: Pointer to object issuing the log message (may be NULL).
 *   Category:: Category id of log message.
 *   Log level:: Severity of logging message.
 *   Message:: The log message, as a '\0' terminated string.
 *  
 * \param LogFunc Logging function to use. NULL to restore the default.
 * \param UserData User data to pass to the logging functions first parameter.
 */
TEMU_API void
temu_logSetAdvancedFunc(void (*LogFunc)(void *, temu_Object *, unsigned, temu_LogLevel, const char *), void *UserData);


/*!
 * Set the logging file for the default logging function. Pass NULL to
 * restore default (stderr).
 *
 * \param FP Logging file pointer, NULL to restore default.
 */
TEMU_API void temu_logSetDefaultFile(FILE *FP);

/*!
 * Enable colours on logging
 * \param Enable 0 to disable colours, 1 to enable
 */
TEMU_API void temu_logSetColour(int Enable);

/*!
 * Log a message with fatal severity; The program will
 * terminate after calling this \param Obj is the source of the log message
 * \param Msg The message to log
 */
TEMU_API void temu_logFatal(const void *Obj, const char *Msg, ...)
    __attribute__((noreturn)) __attribute__((format(printf, 2, 3)));

/*!
 * Log a message with error severity
 * \param Obj is the source of the log message
 * \param Msg The message to log
 */
TEMU_API void temu_logError(const void *Obj, const char *Msg, ...)
    __attribute__((format(printf, 2, 3)));

/*!
 * Log a message with warning severity
 * \param Obj is the source of the log message
 * \param Msg The message to log
 */
TEMU_API void temu_logWarning(const void *Obj, const char *Msg, ...)
    __attribute__((format(printf, 2, 3)));

/*!
 * Log a message with info severity
 * \param Obj is the source of the log message
 * \param Msg The message to log
 */
TEMU_API void temu_logInfo(const void *Obj, const char *Msg, ...)
    __attribute__((format(printf, 2, 3)));

/*!
 * Log a message with trace severity
 * \param Obj is the source of the log message
 * \param Msg The message to log
 */
TEMU_API void temu_logTrace(const void *Obj, const char *Msg, ...)
    __attribute__((format(printf, 2, 3)));

/*!
 * Log a message with debug severity
 *
 * \warning This function is intended for internal use only. Do not
 * call this directly. Use temu_logDebug() instead.
 *
 * \param Obj is the source of the log message
 * \param Msg The message to log
 */
TEMU_API void temu_logDebugFunc(const void *Obj, const char *Msg, ...)
    __attribute__((format(printf, 2, 3)));

#ifdef NDEBUG
static inline void
temu_logDebug(const void *Obj TEMU_UNUSED, const char *Msg TEMU_UNUSED, ...)
{
  ; // Nothing
}

#else

#define temu_logDebug temu_logDebugFunc

#endif

/*
 * New logging API
 *
 * The new logging API provides specialised logging for temu_Object
 * derived classes.  They cannot be used by external classes.
 *
 * EXPERIMENTAL API New logging API. The New logging API provides
 * improved logging support.  Among the features include custom
 * logging categories and per device per categor control of the
 * logging.  16 logging categories are supported, of which the first
 * 8 are reserved for TEMU (i.e. global categories).
 * The next 8 can be controlled per class.
 *
 * The reserved categories is split in three main categories (sim,
 * target and config). These are further subdivided in severity.
 *
 * The config category is used to inform users during the
 * configuration phase or any issues detected with dynamic
 * configuration. For example, a device with an IRQ property may be
 * restricted to e.g. 8 different IRQs, if the user sets another value
 * than the allowed ones then a config error would be logged.
 *
 * The target category is for errors triggered by target software, for
 * example writing a register would be logged with
 * temu_logTargetDebug(). Note that the category is intended for
 * direct effects, which is mostly related to MMIO activity.
 *
 * The sim category is for other errors caused by e.g. invalid use of
 * the TEMU APIs, or other failures.
 *
 * A nice feature of this API is the support for user definable
 * categories. These categories are per class. But category enabling
 * is controlled per object and globally. A category can be added to a
 * class using temu_addLoggingCategory() (see Objsys.h), and queried
 * by id using temu_getLoggingCategory(). The ID can then be used in
 * calls to temu_logToCategory().
 *
 */

// User reserved categories are those above 8
#define teLC_FirstUserCat 8

#define teLC_DefaultCat 0
#define teLC_SimCat 1
#define teLC_TargetCat 2
#define teLC_ConfigCat 3

TEMU_API void temu_logToCategoryVA(const void *Obj, unsigned Category,
                                   temu_LogLevel Severity, const char *Msg,
                                   va_list Args);

/*!
 * Set severity for specific category.
 * \param Obj Object for which to modify logging severity
 * \param Category Category to change severity for (e.g. teLC_DefaultCat)
 * \param Severity The log level for which messages shall be emitted.
 */
TEMU_API void temu_logSetSeverity(void *Obj, unsigned Category,
                                  temu_LogLevel Severity);

/*!
 * Log message in the numbered category
 * \param Obj Object that is the source of the log message
 * \param Category Category of message (e.g. `teLC_DefaultCat`)
 * \param Severity The log level for which the message shall be emitted.
 * \param Msg Printf formatted message string.
 */
TEMU_API void temu_logToCategory(const void *Obj, unsigned Category,
                                 temu_LogLevel Severity, const char *Msg, ...)
    __attribute__((format(printf, 4, 5)));

/*!
 * Log fatal issue Fatal messages result in termination of the program
 * after the message.
 *
 * \param Obj Object
 * \param Msg Printf formatted message string
 */
TEMU_API void temu_logSimFatal(const void *Obj, const char *Msg, ...)
    __attribute__((noreturn)) __attribute__((format(printf, 2, 3)));

/*!
 * Log simulation error
 *
 * \param Obj Object
 * \param Msg Printf formatted message string
 */
TEMU_API void temu_logSimError(const void *Obj, const char *Msg, ...)
    __attribute__((format(printf, 2, 3)));

/*!
 * Log simulation warning
 *
 * \param Obj Object
 * \param Msg Printf formatted message string
 */
TEMU_API void temu_logSimWarning(const void *Obj, const char *Msg, ...)
    __attribute__((format(printf, 2, 3)));

/*!
 * Log simulation info
 *
 * \param Obj Object
 * \param Msg Printf formatted message string
 */
TEMU_API void temu_logSimInfo(const void *Obj, const char *Msg, ...)
    __attribute__((format(printf, 2, 3)));

/*!
 * Log target fatal error
 * This should typically never happen, but it may be convenient
 * in some cases.
 *
 * \param Obj Object
 * \param Msg Printf formatted message string
 */
TEMU_API void temu_logTargetFatal(const void *Obj, const char *Msg, ...)
    __attribute__((noreturn)) __attribute__((format(printf, 2, 3)));

/*!
 * Log target error
 *
 * \param Obj Object
 * \param Msg Printf formatted message string
 */
TEMU_API void temu_logTargetError(const void *Obj, const char *Msg, ...)
    __attribute__((format(printf, 2, 3)));

/*!
 * Log target warning
 *
 * \param Obj Object
 * \param Msg Printf formatted message string
 */
TEMU_API void temu_logTargetWarning(const void *Obj, const char *Msg, ...)
    __attribute__((format(printf, 2, 3)));

/*!
 * Log target info
 *
 * \param Obj Object
 * \param Msg Printf formatted message string
 */

TEMU_API void temu_logTargetInfo(const void *Obj, const char *Msg, ...)
    __attribute__((format(printf, 2, 3)));

/*!
 * Log configuration fatal error
 *
 * \param Obj Object
 * \param Msg Printf formatted message string
 */
TEMU_API void temu_logConfigFatal(const void *Obj, const char *Msg, ...)
    __attribute__((noreturn)) __attribute__((format(printf, 2, 3)));

/*!
 * Log configuration error
 *
 * \param Obj Object
 * \param Msg Printf formatted message string
 */
TEMU_API void temu_logConfigError(const void *Obj, const char *Msg, ...)
    __attribute__((format(printf, 2, 3)));

/*!
 * Log configuration warning
 *
 * \param Obj Object
 * \param Msg Printf formatted message string
 */
TEMU_API void temu_logConfigWarning(const void *Obj, const char *Msg, ...)
    __attribute__((format(printf, 2, 3)));

/*!
 * Log configuration info
 *
 * \param Obj Object
 * \param Msg Printf formatted message string
 */

TEMU_API void temu_logConfigInfo(const void *Obj, const char *Msg, ...)
    __attribute__((format(printf, 2, 3)));

/*!
 * Get object specific logging level for category
 *
 * \param Obj Object
 * \param Category Category
 * \result Logging level for the given `Category`
 */
TEMU_API temu_LogLevel temu_objectGetLogLevel(void *Obj, unsigned Category);
/*!
 * Set object specific logging level for category
 *
 * \param Obj Object
 * \param Category Category
 * \param LogLevel Level to set for the given `Category`
 */
TEMU_API void temu_objectSetLogLevel(void *Obj, unsigned Category,
                                     temu_LogLevel LogLevel);

#ifdef __cplusplus
}
#endif

#endif /* ! TEMU_LOGGING_H */
