#include <stdint.h>
#include <stdio.h>
#include <string.h>

#include "temu-c/Bus/PCI.h"
#include "temu-c/Models/IrqController.h"
#include "temu-c/Support/Attributes.h"
#include "temu-c/Support/Logging.h"
#include "temu-c/Support/Objsys.h"

#include <inttypes.h>

typedef struct {
  temu_PCIDevice Super;
  uint32_t ConfigSpace[TEMU_PCI_CONFIG_WORDS];

  temu_PCIBridgeIfaceRef Bridge;
  temu_IrqCtrlIfaceRef IRQ;
} PciDevice;

static void
startSelfTest(temu_Object *Obj)
{
}

static temu_PCIConfig
getPciConfig(temu_Object *Obj)
{
  PciDevice *P = (PciDevice *)Obj;
  return P->Super.Conf;
}
// Writes to config space forwarded here if outside the standard config space

static void
writeConfig(temu_Object *Obj, uint32_t offset, uint32_t value)
{
  // Write config is called whenver we write to a config register not in the
  // standard PCI config header (i.e. offset 0x3c and onwards).
  // PciDevice *P = (PciDevice *)Obj;
  // switch (MT->Offset) {
  // case 0x3c:
  //  break;
  //}
}
static uint32_t
readConfig(temu_Object *Obj, uint32_t offset)
{
  temu_logDebug(Obj, "Read from configuration area with offset 0x%x ",
                (unsigned)offset);
  return 0;
}
static uint64_t
getPciBarSize(temu_Object *Obj, unsigned Bar)
{
  switch (Bar) {
  case 0:
    return 4096;
  case 1:
  case 2:
  case 3:
  case 4:
  case 5:
    return 4096;
  }

  return 0;
}

static uint64_t
getPciExpansionROMSize(temu_Object *Obj)
{
  return 0;
}

void
memRd(void *Obj, temu_MemTransaction *MT)
{
  temu_logDebug(Obj, "Memory read access pa=0x%" PRIu64 " offset=0x%" PRIu64,
                MT->Pa, MT->Offset);
}

void
memWr(void *Obj, temu_MemTransaction *MT)
{
  temu_logDebug(Obj,
                "Memory write access pa=0x%" PRIu64 " offset=0x%" PRIu64
                " size=%" PRIu64,
                MT->Pa, MT->Offset, MT->Size);
}

void
ioRd(void *Obj, temu_MemTransaction *MT)
{
  temu_logDebug(
      Obj, "IO read access pa=0x%" PRIu64 " offset=0x%" PRIu64 " size=%" PRIu64,
      MT->Pa, MT->Offset, MT->Size);
}

void
ioWr(void *Obj, temu_MemTransaction *MT)
{
  temu_logDebug(Obj,
                "IO write access pa=0x%" PRIu64 " offset=0x%" PRIu64
                " size=%" PRIu64,
                MT->Pa, MT->Offset, MT->Size);
}

static void
expRomRead(void *Obj, temu_MemTransaction *MT)
{
}

static void
expRomWrite(void *Obj, temu_MemTransaction *MT)
{
  return; // ROM, ignore
}

temu_MemAccessIface ExpRomAccessIface = {NULL, expRomRead, expRomWrite, NULL,
                                         NULL};

temu_MemAccessIface MemAccessIf = {NULL, memRd, memWr, NULL, NULL, NULL};

temu_MemAccessIface IoAccessIf = {NULL, ioRd, ioWr, NULL, NULL, NULL};

static temu_MemAccessIface *
getPciBarIface(temu_Object *Obj, uint32_t index, uint8_t type)
{
  PciDevice *P = (PciDevice *)Obj;

  if (index > 5 || P == NULL) {
    return NULL;
  }
  if (type == 0 /*MemAccessInterface*/) {
    return &MemAccessIf;
  } else if (type == 1 /*IoAccessInterface*/) {
    return &IoAccessIf;
  }
  return NULL;
}

static temu_PCIDeviceIface PCIDevIface = {
    startSelfTest, getPciConfig,           writeConfig,   readConfig,
    getPciBarSize, getPciExpansionROMSize, getPciBarIface};

static void *
create(const char *Name, int Argc TEMU_UNUSED,
       const temu_CreateArg *Argv TEMU_UNUSED)
{
  PciDevice *Obj = malloc(sizeof(PciDevice));
  memset(Obj, 0, sizeof(PciDevice));

  Obj->Super.Conf.Size = TEMU_PCI_CONFIG_BYTES;
  Obj->Super.Conf.Data = &Obj->ConfigSpace[0];

  temu_pciSetDeviceId(Obj->Super.Conf, 0x0000);
  temu_pciSetVendorId(Obj->Super.Conf, 0x0000);
  temu_pciSetClassCode(Obj->Super.Conf, tePciBc_Pre);
  temu_pciSetRevId(Obj->Super.Conf, 0x01);
  temu_pciSetBist(Obj->Super.Conf, 0x00);       // Does not support BIST
  temu_pciSetHeaderType(Obj->Super.Conf, 0x00); // Normal header

  temu_pciSetConfig(Obj->Super.Conf, TEMU_PCI_TYPE_00_BAR_0_OFFSET,
                    TEMU_PCI_TYPE_00_BAR_0_SIZE, 0); // Mem space
  temu_pciSetConfig(Obj->Super.Conf, TEMU_PCI_TYPE_00_BAR_1_OFFSET,
                    TEMU_PCI_TYPE_00_BAR_1_SIZE, 0);
  temu_pciSetConfig(Obj->Super.Conf, TEMU_PCI_TYPE_00_BAR_2_OFFSET,
                    TEMU_PCI_TYPE_00_BAR_2_SIZE, 0);
  temu_pciSetConfig(Obj->Super.Conf, TEMU_PCI_TYPE_00_BAR_3_OFFSET,
                    TEMU_PCI_TYPE_00_BAR_3_SIZE, 0);
  temu_pciSetConfig(Obj->Super.Conf, TEMU_PCI_TYPE_00_BAR_4_OFFSET,
                    TEMU_PCI_TYPE_00_BAR_4_SIZE, 0);
  temu_pciSetConfig(Obj->Super.Conf, TEMU_PCI_TYPE_00_BAR_5_OFFSET,
                    TEMU_PCI_TYPE_00_BAR_5_SIZE, 1); // IO space

  return Obj;
}

void
dispose(void *Obj)
{
  free(Obj);
}

TEMU_PLUGIN_INIT
{
  temu_Class *Cls = temu_registerClass("PciDevice", create, dispose);
  temu_pciDeviceRegister(Cls);

  temu_addInterfaceReference(Cls, "bridge", offsetof(PciDevice, Bridge),
                             TEMU_PCI_BRIDGE_IFACE_TYPE, 1, 0, NULL, NULL,
                             "PCI bridge reference");

  temu_addInterfaceReference(Cls, "irqCtrl", offsetof(PciDevice, IRQ),
                             TEMU_IRQ_CTRL_IFACE_TYPE, 1, 0, NULL, NULL,
                             "IRQ controller reference");

  temu_addInterface(Cls, "PCIDevIface", TEMU_PCI_DEVICE_IFACE_TYPE,
                    &PCIDevIface, 0, "PCI device interface");

  temu_addInterface(Cls, "MemAccessIface", TEMU_MEM_ACCESS_IFACE_TYPE,
                    &MemAccessIf, 0, "Memory access interfaces per BAR");
  temu_addInterface(Cls, "ExpROMAccessIface", TEMU_MEM_ACCESS_IFACE_TYPE,
                    &ExpRomAccessIface, 0, "Exp ROM access interface");
}
